/**
 * Modules Step Component
 *
 * Allows selection and activation of ProRank SEO modules
 */

import { useState, useEffect } from '@wordpress/element';
import { __, sprintf } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { Card, Alert, Input, Toggle } from '../../../../components/ui';
import LoadingState from '../../../../components/LoadingState';
import {
  CheckCircleIcon,
  StarIcon,
  MagnifyingGlassIcon,
} from '@heroicons/react/24/solid';
import {
  CheckIcon,
} from '@heroicons/react/24/outline';

const ModulesStep = ({ wizardData, updateWizardData, onNext, onValidationChange }) => {
  const [moduleGroups, setModuleGroups] = useState([]);
  const [selectedGroups, setSelectedGroups] = useState({});
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [showOnlyRecommended, setShowOnlyRecommended] = useState(false);

  useEffect(() => {
    loadModules();
  }, []);

  useEffect(() => {
    if (onValidationChange && !loading && !error) {
      onValidationChange(true);
    }
  }, [onValidationChange, loading, error, selectedGroups]);

  const loadModules = async () => {
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/modules/groups',
      });

      if (response && response.groups) {
        const groups = response.groups || [];
        setModuleGroups(groups);

        const recommended = getRecommendedGroups(groups);
        setSelectedGroups(recommended);
      } else {
        setModuleGroups([]);
      }
    } catch (err) {
      // Don't use fake data - show actual error
      setError(__('Unable to load modules. Please check your connection and try again.', 'prorank-seo'));
      setModuleGroups([]);
    } finally {
      setLoading(false);
    }
  };

  const getRecommendedGroups = (availableGroups) => {
    const selected = {};

    availableGroups.forEach((group) => {
      if (group.enabled) {
        selected[group.slug] = true;
      }
    });

    return selected;
  };

  const handleGroupToggle = (groupSlug, checked) => {
    setSelectedGroups({
      ...selectedGroups,
      [groupSlug]: checked,
    });
  };

  const filterGroups = (groups) => {
    let filtered = groups;

    if (searchTerm) {
      filtered = filtered.filter(group =>
        group.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        group.description.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    if (showOnlyRecommended) {
      filtered = filtered.filter(group => group.recommended === true || group.enabled === true);
    }

    return filtered;
  };

  useEffect(() => {
    updateWizardData('selectedModules', selectedGroups);
  }, [selectedGroups, updateWizardData]);

  if (loading) {
    return <LoadingState message={__('Loading available modules...', 'prorank-seo')} />;
  }

  if (error) {
    return (
      <Card variant="elevated">
        <Alert variant="error">
          {error}
        </Alert>
      </Card>
    );
  }

  // If no modules available, show appropriate message
  if (moduleGroups.length === 0) {
    return (
      <div className="pr-space-y-6">
        <div className="pr-text-center">
          <h2 className="pr-text-2xl pr-font-semibold pr-text-gray-900 pr-mb-2">
            {__('Choose Module Groups', 'prorank-seo')}
          </h2>
          <p className="pr-text-sm pr-text-gray-600">
            {__('Select the SEO feature areas you want to activate.', 'prorank-seo')}
          </p>
        </div>

        <Card variant="elevated">
          <div className="pr-text-center pr-py-12">
            <MagnifyingGlassIcon className="pr-w-12 pr-h-12 pr-text-gray-400 pr-mx-auto pr-mb-4" />
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900 pr-mb-2">
              {__('No module groups available', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Modules will be configured after initial setup. You can activate them from the Modules page.', 'prorank-seo')}
            </p>
          </div>
        </Card>
      </div>
    );
  }

  const filteredGroups = filterGroups(moduleGroups);
  const selectedCount = Object.values(selectedGroups).filter(Boolean).length;
  const showFilters = moduleGroups.length > 3;

  const getGroupIcon = (slug) => {
    const icons = {
      'on-page-seo': '📝',
      'technical-seo': '⚙️',
      performance: '⚡',
    };
    return icons[slug] || '🧩';
  };

  return (
    <div className="pr-space-y-8">
      {/* Header */}
      <div className="pr-text-center">
        <div className="pr-inline-flex pr-items-center pr-justify-center pr-w-16 pr-h-16 pr-rounded-2xl pr-bg-gradient-to-br pr-from-blue-500 pr-to-blue-600 pr-mb-4">
          <span className="pr-text-3xl">🧩</span>
        </div>
        <h2 className="pr-text-2xl pr-font-bold pr-text-gray-900 pr-mb-2">
          {__('Choose Module Groups', 'prorank-seo')}
        </h2>
        <p className="pr-text-gray-600 pr-max-w-md pr-mx-auto">
          {__('Select the SEO feature areas you want to activate. Enabled groups are pre-selected for you.', 'prorank-seo')}
        </p>
      </div>

      {/* Module Stats Bar */}
      {moduleGroups.length > 0 && (
        <div className="pr-grid pr-grid-cols-2 md:pr-grid-cols-3 pr-gap-4">
          <div
            className="pr-relative pr-overflow-hidden pr-rounded-xl pr-p-5 pr-text-center"
            style={{
              background: 'linear-gradient(135deg, #3b82f6 0%, #2563eb 100%)',
              boxShadow: '0 4px 15px -3px rgba(59, 130, 246, 0.3)',
            }}
          >
            <div className="pr-text-4xl pr-font-bold pr-text-white pr-mb-1">{selectedCount}</div>
            <div className="pr-text-sm pr-text-blue-100 pr-font-medium">{__('Selected', 'prorank-seo')}</div>
            <div className="pr-absolute pr-top-0 pr-right-0 pr-w-16 pr-h-16 pr-opacity-10">
              <CheckCircleIcon className="pr-w-full pr-h-full pr-text-white" />
            </div>
          </div>
          <div
            className="pr-relative pr-overflow-hidden pr-rounded-xl pr-p-5 pr-text-center pr-bg-white pr-border pr-border-gray-200"
          >
            <div className="pr-text-4xl pr-font-bold pr-text-gray-900 pr-mb-1">{moduleGroups.length}</div>
            <div className="pr-text-sm pr-text-gray-600 pr-font-medium">{__('Available', 'prorank-seo')}</div>
          </div>
          <div className="pr-hidden md:pr-block pr-relative pr-overflow-hidden pr-rounded-xl pr-p-5 pr-text-center pr-bg-emerald-50 pr-border pr-border-emerald-200">
            <div className="pr-text-4xl pr-font-bold pr-text-emerald-600 pr-mb-1">
              {moduleGroups.length > 0 ? Math.round((selectedCount / moduleGroups.length) * 100) : 0}%
            </div>
            <div className="pr-text-sm pr-text-emerald-700 pr-font-medium">{__('Coverage', 'prorank-seo')}</div>
          </div>
        </div>
      )}

      {/* Search and Filter Bar */}
      {showFilters && (
        <Card variant="outlined">
          <div className="pr-flex pr-flex-col md:pr-flex-row pr-items-center pr-gap-4">
            <div className="pr-flex-1 pr-w-full">
              <Input
                type="search"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                placeholder={__('Search modules...', 'prorank-seo')}
                icon={MagnifyingGlassIcon}
              />
            </div>
            <div className="pr-flex pr-items-center pr-gap-2">
              <Toggle
                checked={showOnlyRecommended}
                onChange={setShowOnlyRecommended}
                label={__('Show only recommended', 'prorank-seo')}
              />
            </div>
          </div>
        </Card>
      )}

      {/* Modules Grid */}
      <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-2 pr-gap-4">
        {filteredGroups.map((group) => {
          const isSelected = selectedGroups[group.slug] || false;
          const isRecommended = group.recommended === true || group.enabled === true;

          return (
            <div
              key={group.slug}
              className={`
                pr-relative pr-p-5 pr-rounded-xl pr-cursor-pointer pr-transition-all pr-duration-200
                pr-border-2 pr-group
                ${isSelected
                  ? 'pr-border-blue-500 pr-bg-blue-50/50 pr-shadow-lg pr-shadow-blue-500/10'
                  : 'pr-border-gray-200 pr-bg-white hover:pr-border-gray-300 hover:pr-shadow-md'
                }
              `}
              onClick={() => handleGroupToggle(group.slug, !isSelected)}
            >
              {isSelected && (
                <div
                  className="pr-absolute pr-inset-0 pr-rounded-xl pr-opacity-5"
                  style={{ background: 'linear-gradient(135deg, #3b82f6 0%, #ff6b35 100%)' }}
                />
              )}

              <div className="pr-flex pr-items-start pr-gap-3 pr-mb-3 pr-relative">
                <div className={`
                  pr-p-2.5 pr-rounded-xl pr-transition-all pr-duration-200
                  ${isSelected
                    ? 'pr-bg-blue-100'
                    : 'pr-bg-gray-100 group-hover:pr-bg-gray-200'
                  }
                `}>
                  <span className="pr-text-xl">{getGroupIcon(group.slug)}</span>
                </div>
                <div className="pr-flex-1 pr-min-w-0">
                  <div className="pr-flex pr-items-start pr-justify-between pr-gap-2">
                    <div>
                      <h4 className="pr-text-base pr-font-semibold pr-text-gray-900 pr-mb-0.5">
                        {group.name}
                      </h4>
                      {isRecommended && (
                        <span
                          className="pr-inline-flex pr-items-center pr-gap-1 pr-px-2 pr-py-0.5 pr-text-xs pr-font-semibold pr-rounded-full"
                          style={{
                            background: 'linear-gradient(135deg, #ff6b35 0%, #f59e0b 100%)',
                            color: 'white',
                          }}
                        >
                          <StarIcon className="pr-w-3 pr-h-3" />
                          {__('Recommended', 'prorank-seo')}
                        </span>
                      )}
                    </div>
                    <div
                      className={`
                        pr-w-6 pr-h-6 pr-rounded-lg pr-flex pr-items-center pr-justify-center
                        pr-transition-all pr-duration-200 pr-flex-shrink-0
                        ${isSelected
                          ? 'pr-bg-blue-500 pr-text-white'
                          : 'pr-border-2 pr-border-gray-300 group-hover:pr-border-gray-400'
                        }
                      `}
                      onClick={(e) => {
                        e.stopPropagation();
                        handleGroupToggle(group.slug, !isSelected);
                      }}
                    >
                      {isSelected && <CheckIcon className="pr-w-4 pr-h-4" />}
                    </div>
                  </div>
                </div>
              </div>

              <p className="pr-text-sm pr-text-gray-600 pr-leading-relaxed pr-relative">
                {group.description}
              </p>
            </div>
          );
        })}
      </div>

      {/* Empty State after filtering */}
      {filteredGroups.length === 0 && moduleGroups.length > 0 && (
        <Card variant="elevated">
          <div className="pr-text-center pr-py-12">
            <MagnifyingGlassIcon className="pr-w-12 pr-h-12 pr-text-gray-400 pr-mx-auto pr-mb-4" />
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900 pr-mb-2">
              {__('No groups found', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Try adjusting your search or filters', 'prorank-seo')}
            </p>
          </div>
        </Card>
      )}

      {/* Summary Notice */}
      {selectedCount > 0 && (
        <div
          className="pr-flex pr-items-center pr-gap-4 pr-p-4 pr-rounded-xl pr-border"
          style={{
            background: 'linear-gradient(135deg, rgba(16, 185, 129, 0.05) 0%, rgba(59, 130, 246, 0.05) 100%)',
            borderColor: 'rgba(16, 185, 129, 0.2)',
          }}
        >
          <div className="pr-flex-shrink-0 pr-w-10 pr-h-10 pr-rounded-full pr-bg-emerald-100 pr-flex pr-items-center pr-justify-center">
            <CheckCircleIcon className="pr-w-6 pr-h-6 pr-text-emerald-600" />
          </div>
          <div className="pr-flex-1">
            <p className="pr-text-sm pr-font-semibold pr-text-gray-900">
              {sprintf(__('%d module groups selected', 'prorank-seo'), selectedCount)}
            </p>
            <p className="pr-text-xs pr-text-gray-600">
              {__('You can change these selections at any time from the Modules page.', 'prorank-seo')}
            </p>
          </div>
        </div>
      )}
    </div>
  );
};

export default ModulesStep;
