/**
 * Wizard State Hook
 * 
 * Manages the state and navigation for the Quick Setup Wizard
 */

import { useEffect, useRef, useState } from '@wordpress/element';
import apiFetch from '@wordpress/api-fetch';

const WIZARD_STORAGE_KEY = 'prorank_wizard_state';

export const useWizardState = () => {
  // Initialise from localStorage for instant UX, then hydrate from server.
  const [currentStep, setCurrentStep] = useState(() => {
    const saved = localStorage.getItem(WIZARD_STORAGE_KEY);
    if (saved) {
      const parsed = JSON.parse(saved);
      return parsed.currentStep ?? 0;
    }
    return 0;
  });

  const [completedSteps, setCompletedSteps] = useState(() => {
    const saved = localStorage.getItem(WIZARD_STORAGE_KEY);
    if (saved) {
      const parsed = JSON.parse(saved);
      return parsed.completedSteps || [];
    }
    return [];
  });

  const [wizardData, setWizardData] = useState(() => {
    const saved = localStorage.getItem(WIZARD_STORAGE_KEY);
    if (saved) {
      const parsed = JSON.parse(saved);
      return parsed.wizardData || {};
    }
    return {};
  });

  const [isWizardCompleted, setIsWizardCompleted] = useState(false);
  const [saving, setSaving] = useState(false);
  const [lastSaved, setLastSaved] = useState(null);
  const [hydrated, setHydrated] = useState(false);
  const saveTimeoutRef = useRef(null);

  // Hydrate from server option.
  useEffect(() => {
    let isMounted = true;
    const hydrate = async () => {
      try {
        const response = await apiFetch({ path: '/prorank-seo/v1/wizard/state' });
        if (!isMounted || !response?.data) return;

        const { currentStep: step = 0, completedSteps: completed = [], wizardData: data = {}, updated_at } = response.data;
        setCurrentStep(step);
        setCompletedSteps(Array.isArray(completed) ? completed : []);
        setWizardData(data || {});
        setLastSaved(updated_at || new Date().toISOString());
        setIsWizardCompleted(!!data?.wizardCompleted);
      } catch (err) {
        // Silent fail: keep local state
      } finally {
        if (isMounted) setHydrated(true);
      }
    };

    hydrate();
    return () => {
      isMounted = false;
    };
  }, []);

  // Persist to localStorage for resilience.
  useEffect(() => {
    const stateToSave = {
      currentStep,
      completedSteps,
      wizardData,
      lastUpdated: new Date().toISOString(),
    };
    localStorage.setItem(WIZARD_STORAGE_KEY, JSON.stringify(stateToSave));
  }, [currentStep, completedSteps, wizardData]);

  // Persist to server (debounced) once hydrated.
  useEffect(() => {
    if (!hydrated) return;
    if (saveTimeoutRef.current) {
      clearTimeout(saveTimeoutRef.current);
    }

    saveTimeoutRef.current = setTimeout(async () => {
      setSaving(true);
      try {
        const response = await apiFetch({
          path: '/prorank-seo/v1/wizard/state',
          method: 'POST',
          data: {
            currentStep,
            completedSteps,
            wizardData,
          },
        });
        setLastSaved(response?.data?.updated_at || new Date().toISOString());
      } catch (err) {
        // Keep silent but mark saving finished
      } finally {
        setSaving(false);
      }
    }, 300);

    return () => {
      if (saveTimeoutRef.current) {
        clearTimeout(saveTimeoutRef.current);
      }
    };
  }, [currentStep, completedSteps, wizardData, hydrated]);

  // Navigation functions
  const goToStep = (stepIndex) => {
    setCurrentStep(stepIndex);
  };

  const nextStep = () => {
    setCurrentStep((prev) => prev + 1);
  };

  const previousStep = () => {
    setCurrentStep((prev) => Math.max(0, prev - 1));
  };

  const markStepCompleted = (stepIndex) => {
    setCompletedSteps((prev) => {
      if (!prev.includes(stepIndex)) {
        return [...prev, stepIndex];
      }
      return prev;
    });
  };

  const updateWizardData = (key, value) => {
    setWizardData((prev) => ({
      ...prev,
      [key]: value,
    }));
  };

  const resetWizard = () => {
    setCurrentStep(0);
    setCompletedSteps([]);
    setWizardData({});
    localStorage.removeItem(WIZARD_STORAGE_KEY);
  };

  const completeWizard = async () => {
    setIsWizardCompleted(true);
    // Persist completion both to settings and wizard state.
    try {
      await apiFetch({
        path: '/prorank-seo/v1/settings',
        method: 'POST',
        data: {
          wizard_completed: true,
          wizard_completion_date: new Date().toISOString(),
        },
      });
    } catch (err) {
      // Ignore - wizard state still saved below
    }

    try {
      await apiFetch({
        path: '/prorank-seo/v1/wizard/state',
        method: 'POST',
        data: {
          currentStep,
          completedSteps,
          wizardData: { ...wizardData, wizardCompleted: true },
        },
      });
    } catch (err) {
      // ignore
    }

    localStorage.removeItem(WIZARD_STORAGE_KEY);
  };

  const isStepCompleted = (stepIndex) => completedSteps.includes(stepIndex);

  const canNavigateToStep = (stepIndex) => {
    // Can always go back
    if (stepIndex < currentStep) return true;
    // Can go to next step if current is completed
    if (stepIndex === currentStep + 1) {
      return isStepCompleted(currentStep);
    }
    // Can skip to any previously completed step
    return isStepCompleted(stepIndex - 1);
  };

  return {
    currentStep,
    completedSteps,
    wizardData,
    isWizardCompleted,
    goToStep,
    nextStep,
    previousStep,
    markStepCompleted,
    updateWizardData,
    resetWizard,
    completeWizard,
    isStepCompleted,
    canNavigateToStep,
    saving,
    lastSaved,
    hydrated,
  };
};
