/**
 * Wizard Progress Component
 *
 * Displays the progress bar and step indicators for the wizard
 * with ARIA labels for accessibility and mobile-responsive step names
 */

import { __, sprintf } from '@wordpress/i18n';
import { CheckIcon, SparklesIcon } from '@heroicons/react/24/solid';

const WizardProgress = ({ steps, currentStep, completedSteps, onStepClick }) => {
  const progressPercentage = ((currentStep + 1) / steps.length) * 100;

  // Get short title for mobile (first 3-4 chars or abbreviated version)
  const getShortTitle = (step) => {
    const shortTitles = {
      'welcome': 'Intro',
      'import': 'Import',
      'site-basics': 'Site',
      'modules': 'Mods',
      'optimization': 'Opt',
      'ready': 'Done',
    };
    return shortTitles[step.id] || step.title.substring(0, 4);
  };

  // Get status text for screen readers
  const getStepStatus = (index, isActive, isCompleted) => {
    if (isCompleted) return __('Completed', 'prorank-seo');
    if (isActive) return __('Current step', 'prorank-seo');
    if (index < currentStep) return __('Completed', 'prorank-seo');
    return __('Pending', 'prorank-seo');
  };

  // Get step icon based on step type
  const getStepIcon = (step) => {
    const icons = {
      'welcome': '👋',
      'import': '📥',
      'site-basics': '🏢',
      'modules': '🧩',
      'optimization': '⚡',
      'ready': '🚀',
    };
    return icons[step.id] || null;
  };

  return (
    <div className="pr-mb-10">
      {/* Modern Progress Bar */}
      <div className="pr-relative pr-mb-10">
        <div
          className="pr-h-2 pr-bg-gray-100 pr-rounded-full pr-overflow-hidden pr-shadow-inner"
          role="progressbar"
          aria-valuenow={currentStep + 1}
          aria-valuemin={1}
          aria-valuemax={steps.length}
          aria-label={sprintf(__('Step %d of %d', 'prorank-seo'), currentStep + 1, steps.length)}
        >
          <div
            className="pr-h-full pr-rounded-full pr-transition-all pr-duration-500 pr-ease-out"
            style={{
              width: `${progressPercentage}%`,
              background: 'linear-gradient(90deg, #3b82f6 0%, #2563eb 50%, #ff6b35 100%)',
              boxShadow: '0 0 20px rgba(59, 130, 246, 0.5)',
            }}
          />
        </div>
        {/* Animated glow effect */}
        <div
          className="pr-absolute pr-top-0 pr-left-0 pr-h-2 pr-rounded-full pr-transition-all pr-duration-500 pr-opacity-50 pr-blur-sm"
          style={{
            width: `${progressPercentage}%`,
            background: 'linear-gradient(90deg, #3b82f6, #ff6b35)',
          }}
        />
      </div>

      {/* Steps with ARIA roles */}
      <nav
        className="pr-relative pr-flex pr-justify-between"
        role="tablist"
        aria-label={__('Setup wizard steps', 'prorank-seo')}
      >
        {/* Connecting Line with gradient */}
        <div
          className="pr-absolute pr-top-6 pr-left-[8%] pr-right-[8%] pr-h-0.5 pr-z-0"
          style={{
            background: 'linear-gradient(90deg, #e5e7eb 0%, #d1d5db 50%, #e5e7eb 100%)',
          }}
          aria-hidden="true"
        />
        {/* Active line overlay */}
        <div
          className="pr-absolute pr-top-6 pr-left-[8%] pr-h-0.5 pr-z-0 pr-transition-all pr-duration-500"
          style={{
            width: `${Math.max(0, (currentStep / (steps.length - 1)) * 84)}%`,
            background: 'linear-gradient(90deg, #3b82f6, #2563eb)',
          }}
          aria-hidden="true"
        />

        {steps.map((step, index) => {
          const isActive = currentStep === index;
          const isCompleted = completedSteps.includes(index);
          const isPast = index < currentStep;
          const canNavigate = isCompleted || isPast || index === currentStep;
          const stepStatus = getStepStatus(index, isActive, isCompleted);
          const stepIcon = getStepIcon(step);

          return (
            <button
              key={step.id}
              className={`
                pr-relative pr-z-10 pr-flex pr-flex-col pr-items-center pr-flex-1
                pr-transition-all pr-duration-300 pr-group
                ${canNavigate ? 'hover:pr-transform hover:pr-scale-105 pr-cursor-pointer' : 'pr-cursor-not-allowed pr-opacity-50'}
              `}
              onClick={() => canNavigate && onStepClick(index)}
              type="button"
              role="tab"
              aria-selected={isActive}
              aria-current={isActive ? 'step' : undefined}
              aria-label={sprintf(
                __('%s - %s (Step %d of %d)', 'prorank-seo'),
                step.title,
                stepStatus,
                index + 1,
                steps.length
              )}
              aria-disabled={!canNavigate}
              tabIndex={canNavigate ? 0 : -1}
            >
              {/* Step Indicator */}
              <div
                className={`
                  pr-w-12 pr-h-12 pr-rounded-xl pr-flex pr-items-center pr-justify-center
                  pr-font-bold pr-text-sm pr-transition-all pr-duration-300
                  ${isActive ? 'pr-scale-110' : ''}
                  ${isCompleted ? 'pr-bg-gradient-to-br pr-from-emerald-400 pr-to-emerald-600 pr-text-white pr-shadow-lg pr-shadow-emerald-500/30' : ''}
                  ${isActive && !isCompleted ? 'pr-text-white pr-shadow-xl' : ''}
                  ${!isActive && !isCompleted ? 'pr-bg-white pr-border-2 pr-border-gray-200 pr-text-gray-400 group-hover:pr-border-blue-300 group-hover:pr-text-blue-500' : ''}
                `}
                style={isActive && !isCompleted ? {
                  background: 'linear-gradient(135deg, #3b82f6 0%, #2563eb 50%, #1d4ed8 100%)',
                  boxShadow: '0 10px 40px -10px rgba(59, 130, 246, 0.5)',
                } : {}}
              >
                {isCompleted ? (
                  <CheckIcon className="pr-w-6 pr-h-6" />
                ) : isActive ? (
                  <span className="pr-animate-pulse">{stepIcon || index + 1}</span>
                ) : (
                  <span>{stepIcon || index + 1}</span>
                )}
              </div>

              {/* Pulse animation for active step */}
              {isActive && (
                <div
                  className="pr-absolute pr-top-0 pr-w-12 pr-h-12 pr-rounded-xl pr-animate-ping pr-opacity-20"
                  style={{ background: 'linear-gradient(135deg, #3b82f6, #ff6b35)' }}
                />
              )}

              {/* Step Title - Full on desktop, abbreviated on mobile */}
              <div className="pr-mt-3 pr-text-center">
                {/* Full title - hidden on small screens */}
                <div className={`
                  pr-hidden sm:pr-block pr-text-sm pr-font-semibold pr-transition-all pr-duration-300
                  ${isActive ? 'pr-text-gray-900' : isCompleted ? 'pr-text-emerald-600' : 'pr-text-gray-500 group-hover:pr-text-gray-700'}
                `}>
                  {step.title}
                </div>
                {/* Abbreviated title - visible only on small screens */}
                <div className={`
                  sm:pr-hidden pr-text-xs pr-font-semibold pr-transition-colors
                  ${isActive ? 'pr-text-gray-900' : isCompleted ? 'pr-text-emerald-600' : 'pr-text-gray-500'}
                `}>
                  {getShortTitle(step)}
                </div>
                {/* Description - only on desktop when active */}
                {isActive && (
                  <div className="pr-hidden sm:pr-block pr-text-xs pr-text-gray-500 pr-mt-1 pr-max-w-[140px] pr-mx-auto pr-animate-fadeIn">
                    {step.description}
                  </div>
                )}
              </div>
            </button>
          );
        })}
      </nav>

      {/* Custom animation styles */}
      <style>{`
        @keyframes fadeIn {
          from { opacity: 0; transform: translateY(-5px); }
          to { opacity: 1; transform: translateY(0); }
        }
        .pr-animate-fadeIn {
          animation: fadeIn 0.3s ease-out forwards;
        }
      `}</style>
    </div>
  );
};

export default WizardProgress;
