/**
 * Wizard Navigation Component
 *
 * Provides navigation buttons for the wizard with modern styling
 */

import { __ } from '@wordpress/i18n';
import { Button } from '../../../../components/ui';
import {
  ArrowLeftIcon,
  ArrowRightIcon,
  CheckIcon,
  RocketLaunchIcon,
  CloudArrowUpIcon,
} from '@heroicons/react/24/outline';

const WizardNavigation = ({
  currentStep,
  totalSteps,
  onPrevious,
  onNext,
  onSkip,
  onComplete,
  canGoNext = true,
  canGoPrevious = true,
  isProcessing = false,
  isLastStep = false,
  saving = false,
  lastSaved = null,
}) => {
  return (
    <div className="pr-relative pr-mt-8">
      {/* Decorative top border */}
      <div
        className="pr-absolute pr-top-0 pr-left-0 pr-right-0 pr-h-px"
        style={{
          background: 'linear-gradient(90deg, transparent, #e5e7eb 20%, #e5e7eb 80%, transparent)',
        }}
      />

      <div className="pr-flex pr-items-center pr-justify-between pr-pt-6 pr-flex-wrap pr-gap-4">
        {/* Left side - Back and Skip */}
        <div className="pr-flex pr-items-center pr-gap-3 pr-flex-wrap">
          {currentStep > 0 && canGoPrevious && (
            <button
              onClick={onPrevious}
              disabled={isProcessing}
              className={`
                pr-inline-flex pr-items-center pr-gap-2 pr-px-4 pr-py-2.5
                pr-text-sm pr-font-medium pr-text-gray-600
                pr-bg-white pr-border pr-border-gray-200 pr-rounded-lg
                pr-transition-all pr-duration-200
                hover:pr-bg-gray-50 hover:pr-border-gray-300 hover:pr-text-gray-800
                focus:pr-outline-none focus:pr-ring-2 focus:pr-ring-blue-500/20 focus:pr-border-blue-400
                disabled:pr-opacity-50 disabled:pr-cursor-not-allowed
                pr-shadow-sm hover:pr-shadow
              `}
            >
              <ArrowLeftIcon className="pr-w-4 pr-h-4" />
              {__('Previous', 'prorank-seo')}
            </button>
          )}
          {onSkip && !isLastStep && (
            <button
              onClick={onSkip}
              disabled={isProcessing}
              className={`
                pr-inline-flex pr-items-center pr-gap-1.5 pr-px-3 pr-py-2
                pr-text-sm pr-font-medium pr-text-gray-500
                pr-rounded-lg pr-transition-all pr-duration-200
                hover:pr-text-gray-700 hover:pr-bg-gray-100
                disabled:pr-opacity-50 disabled:pr-cursor-not-allowed
              `}
            >
              {__('Skip this step', 'prorank-seo')}
            </button>
          )}
        </div>

        {/* Right side - Save status and Next/Complete */}
        <div className="pr-flex pr-items-center pr-gap-4 pr-flex-wrap pr-justify-end pr-ml-auto">
          {/* Save status indicator */}
          {lastSaved && (
            <div className="pr-flex pr-items-center pr-gap-2 pr-text-xs pr-text-gray-400">
              <CloudArrowUpIcon className={`pr-w-4 pr-h-4 ${saving ? 'pr-animate-pulse pr-text-blue-500' : 'pr-text-emerald-500'}`} />
              <span>
                {saving ? __('Saving…', 'prorank-seo') : __('Saved', 'prorank-seo')}
              </span>
            </div>
          )}

          {/* Next / Complete button */}
          {isLastStep ? (
            <button
              onClick={onComplete}
              disabled={isProcessing || !canGoNext}
              className={`
                pr-inline-flex pr-items-center pr-gap-2 pr-px-6 pr-py-3
                pr-text-sm pr-font-semibold pr-text-white
                pr-rounded-xl pr-transition-all pr-duration-300
                focus:pr-outline-none focus:pr-ring-2 focus:pr-ring-offset-2 focus:pr-ring-emerald-500
                disabled:pr-opacity-50 disabled:pr-cursor-not-allowed disabled:pr-transform-none
                hover:pr-scale-[1.02] hover:pr-shadow-xl
                active:pr-scale-[0.98]
              `}
              style={{
                background: 'linear-gradient(135deg, #10b981 0%, #059669 100%)',
                boxShadow: '0 4px 20px -5px rgba(16, 185, 129, 0.5)',
              }}
            >
              {isProcessing ? (
                <>
                  <svg className="pr-animate-spin pr-w-4 pr-h-4" viewBox="0 0 24 24">
                    <circle className="pr-opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none" />
                    <path className="pr-opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
                  </svg>
                  {__('Completing...', 'prorank-seo')}
                </>
              ) : (
                <>
                  {__('Complete Setup', 'prorank-seo')}
                  <RocketLaunchIcon className="pr-w-5 pr-h-5" />
                </>
              )}
            </button>
          ) : (
            <button
              onClick={onNext}
              disabled={isProcessing || !canGoNext}
              className={`
                pr-inline-flex pr-items-center pr-gap-2 pr-px-6 pr-py-3
                pr-text-sm pr-font-semibold pr-text-white
                pr-rounded-xl pr-transition-all pr-duration-300
                focus:pr-outline-none focus:pr-ring-2 focus:pr-ring-offset-2 focus:pr-ring-blue-500
                disabled:pr-opacity-50 disabled:pr-cursor-not-allowed disabled:pr-transform-none
                hover:pr-scale-[1.02] hover:pr-shadow-xl
                active:pr-scale-[0.98]
              `}
              style={{
                background: 'linear-gradient(135deg, #3b82f6 0%, #2563eb 50%, #1d4ed8 100%)',
                boxShadow: '0 4px 20px -5px rgba(59, 130, 246, 0.5)',
              }}
            >
              {isProcessing ? (
                <>
                  <svg className="pr-animate-spin pr-w-4 pr-h-4" viewBox="0 0 24 24">
                    <circle className="pr-opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none" />
                    <path className="pr-opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
                  </svg>
                  {__('Processing...', 'prorank-seo')}
                </>
              ) : (
                <>
                  {__('Continue', 'prorank-seo')}
                  <ArrowRightIcon className="pr-w-5 pr-h-5" />
                </>
              )}
            </button>
          )}
        </div>
      </div>

      {/* Step counter */}
      <div className="pr-flex pr-justify-center pr-mt-6">
        <div className="pr-flex pr-items-center pr-gap-2 pr-px-4 pr-py-2 pr-bg-gray-50 pr-rounded-full pr-text-xs pr-text-gray-500">
          <span className="pr-font-semibold pr-text-gray-700">{currentStep + 1}</span>
          <span>/</span>
          <span>{totalSteps}</span>
          <span className="pr-text-gray-400">|</span>
          <span>{Math.round(((currentStep + 1) / totalSteps) * 100)}% {__('complete', 'prorank-seo')}</span>
        </div>
      </div>
    </div>
  );
};

export default WizardNavigation;
