/* global wp */
import { useState, useEffect } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import {
  ChartBarIcon,
  Cog6ToothIcon,
  RocketLaunchIcon,
} from '@heroicons/react/24/outline';
import OverviewTab from './OverviewTab';
import QuickSetupWizard from './QuickSetupWizard';
import ModuleManagerTab from './ModuleManagerTab';
import PageHeader from '../../components/PageHeader';
import headerBadges from '../../utils/headerBadges';
import * as React from 'react';

// TypeScript interfaces
interface DashboardStats {
  seo_score?: number;
  published_pages?: number; // Local count from wp_count_posts()
  issues_count?: number;
  redirects_count?: number;
  seo_score_trend?: number | null;
  issues_count_trend?: number | null;
  redirects_count_trend?: number | null;
  audit?: {
    id?: string;
    score?: number;
    state?: string;
    last_run?: string | null;
    issues?: {
      critical?: number;
      high?: number;
      medium?: number;
      low?: number;
      total?: number;
    };
    history?: Array<{
      id?: string;
      score?: number;
      completed_at?: string | null;
      issues_total?: number | null;
    }>;
  };
  performance?: {
    cache?: Record<string, any>;
    assets?: Record<string, any>;
    images?: Record<string, any>;
    rum?: Record<string, any>;
  };
  [key: string]: any;
}

interface Tab {
  name: string;
  title: string;
  icon: React.ComponentType<React.SVGProps<SVGSVGElement>>;
  content: React.ReactElement;
  badge?: number;
}

const Dashboard: React.FC = () => {
  const [loading, setLoading] = useState<boolean>(true);
  const [stats, setStats] = useState<DashboardStats | null>(null);
  const [activeTab, setActiveTab] = useState<number>(0);

  useEffect(() => {
    document.title =
      __('Dashboard', 'prorank-seo') + ' ‹ ' + __('ProRank SEO', 'prorank-seo') + ' — WordPress';
    loadDashboardData();
  }, []);

  const loadDashboardData = async (forceRefresh: boolean = false): Promise<void> => {
    setLoading(true);
    try {
      // Load dashboard statistics from REST API
      const path = forceRefresh
        ? '/prorank-seo/v1/dashboard/stats?refresh=1'
        : '/prorank-seo/v1/dashboard/stats';
      const response = await (window as any).wp.apiFetch({
        path,
      }) as any;
      const payload = (response && typeof response === 'object' && 'data' in response) ? (response as any).data : response;
      setStats(payload);
    } catch (error) {
      // Show error state instead of fallback data
      setStats(null);
    } finally {
      setLoading(false);
    }
  };

  const tabs: Tab[] = [
    {
      name: 'overview',
      title: __('Overview', 'prorank-seo'),
      icon: ChartBarIcon,
      content: <OverviewTab stats={stats} onRefresh={() => loadDashboardData(true)} />,
    },
    {
      name: 'module-manager',
      title: __('Module Manager', 'prorank-seo'),
      icon: Cog6ToothIcon,
      content: <ModuleManagerTab />,
    },
    {
      name: 'quick-setup',
      title: __('Quick Setup', 'prorank-seo'),
      icon: RocketLaunchIcon,
      content: <QuickSetupWizard />,
    },
  ];

  if (loading && !stats) {
    return (
      <div className="prorank-dashboard-loading">
        <div className="prorank-spinner" />
        <p>{__('Loading dashboard…', 'prorank-seo')}</p>
      </div>
    );
  }

  return (
    <div className="prorank-page">
      <PageHeader
        title={__('Dashboard', 'prorank-seo')}
        subtitle={__('Monitor your SEO performance and manage your optimization tasks', 'prorank-seo')}
        badges={headerBadges.dashboard}
      />

      {/* Main Content */}
      <div className="prorank-content">
        {/* Tab Navigation */}
        <div className="prorank-tab-panel">
          <div className="prorank-tab-panel__tabs" role="tablist">
            {tabs.map((tab, index) => {
              const IconComponent = tab.icon;
              const isActive = index === activeTab;

              return (
                <button
                  key={tab.name}
                  role="tab"
                  aria-selected={isActive}
                  aria-controls={`panel-${tab.name}`}
                  onClick={() => setActiveTab(index)}
                  className={`prorank-tab-panel__tab ${isActive ? 'is-active' : ''}`}
                >
                  <IconComponent className="pr-w-5 pr-h-5 pr-mr-2" aria-hidden="true" />
                  <span>{tab.title}</span>
                  {tab.badge !== undefined && tab.badge > 0 && (
                    <span
                      className="pr-ml-2 pr-px-2 pr-py-0.5 pr-text-xs pr-font-medium pr-bg-orange-500 pr-text-white pr-rounded-full"
                      aria-label={`${tab.badge} notifications`}
                    >
                      {tab.badge > 99 ? '99+' : tab.badge}
                    </span>
                  )}
                </button>
              );
            })}
          </div>
        </div>

        {/* Tab Content */}
        <div
          className="prorank-tab-panel__tab-content"
          role="tabpanel"
          id={`panel-${tabs[activeTab]?.name}`}
        >
          {tabs[activeTab]?.content}
        </div>
      </div>
    </div>
  );
};

export default Dashboard;
