/**
 * Quick Setup Wizard Component (Free)
 *
 * Modern, polished wizard with blue/orange brand colors
 */

import { useState, lazy, Suspense } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import { useWizardState } from './wizard/hooks/useWizardState';
import WizardProgress from './wizard/components/WizardProgress';
import WizardNavigation from './wizard/components/WizardNavigation';
import LoadingState from '../../components/LoadingState';

// Lazy load wizard steps for better performance
const WelcomeStep = lazy(() => import(/* webpackMode: "eager" */ './wizard/steps/WelcomeStep'));
const CompetitorImportStep = lazy(() => import(/* webpackMode: "eager" */ './wizard/steps/CompetitorImportStep'));
const SiteBasicsStep = lazy(() => import(/* webpackMode: "eager" */ './wizard/steps/SiteBasicsStep'));
const ModulesStep = lazy(() => import(/* webpackMode: "eager" */ './wizard/steps/ModulesStep'));
const OptimizationStep = lazy(() => import(/* webpackMode: "eager" */ './wizard/steps/OptimizationStep'));
const ReadyStep = lazy(() => import(/* webpackMode: "eager" */ './wizard/steps/ReadyStep'));

// ProRank Logo SVG component
const ProrankLogo = () => (
  <svg width="40" height="40" viewBox="0 0 1909 2048" className="pr-flex-shrink-0">
    <path transform="translate(959,1)" d="m0 0h10l6 3 5 8 12 32 8 19 9 24 7 20 16 40 10 26 3 10 13 34 4 9 12 31 7 20 10 25 9 23 7 18 6 18 8 21 7 16 11 29 3 10 14 36 5 12 11 28 4 13 12 36 3 16 1 13v13l-2 16-2 4-2 1-14 1-424 11h-14l-6 4-6 7-2 6-1 30-1 149v195l-3 8-7 6-35 12-28 11-27 10-37 13-16 6-12 5-30 11-11-2-11-9-16-13-11-8-9-8-11-9-5-4-5-5-14-12-10-8-12-9-13-11-11-9-5-4-5-5-13-11-14-11-9-7-13-11-14-11-8-8-14-11-9-8-6-5-5-3-13-11-11-9-10-9-13-11-11-9-8-6-10-8-13-11-11-9-12-11-10-8-11-9-11-8-15-13-10-8-12-11-22-18-11-8-15-13-10-8-12-11-11-9-4-5-1-2v-12l4-6 8-4 27-1 36-1 11-1 24-1 47-1 42-2 47-1 35-2 47-1 26-1 9-1 73-2 42-2h19l44-2 19-1 47-1 25-1 10-1 35-1h12l12-1 10-5 4-7 7-20 9-24 5-11 10-27 6-18 11-29 4-8 13-35 7-20 11-28 10-26 4-10 4-13 12-31 8-20 8-21 6-18 10-26 4-9 11-29 5-15 13-34 5-12 11-29 7-20 9-24 4-9 12-31 4-13 4-8z" fill="#0B121F" />
    <path transform="translate(728,646)" d="m0 0h324l24 2 24 4 20 5 20 7 16 7 19 10 15 10 14 11 13 12 11 12 10 13 10 15 11 21 7 18 7 25 4 25 1 11v62l-3 24-4 19-9 27-10 21-11 18-10 13-11 13-14 14-11 9-14 10-15 9-17 9-23 9-26 7-23 4-22 2h-185v275h-142z" fill="#FE6900" />
    <path transform="translate(1156,637)" d="m0 0h26l567 19 143 5 9 3 5 5 3 9-3 8-8 8-11 9-8 6-11 9-13 11-9 7-5 5-11 10-11 9-8 6-11 9-13 11-10 8-5 5-10 9-30 24-13 11-11 9-7 7-11 9-13 11-13 10-26 22-11 10-11 9-12 10-8 6-13 11-11 9-4 2-2 4-13 11-11 9-8 6-11 9-13 11-11 9-12 11-10 8-13 10-14 11-14 12-10 9-16 13-14 11-8 6-11 9-16 13-10 9-14 11-11 9-5 3-16 13-14 11-10 9-14 11-24 18-25 20-13 11-16 12-11 8-13 10-17 13-5 5-16 12-14 11-7 4-15 12-15 11-5 5-19 14-11 8-10 6-7 1-4-2h-12l-12 1-25 1-50 1-16 1h-17l-1-4v-31l1-24 1-59v-142l-3-1v-3h3l1 3 22 1 9 1 29 1h15l60 3h29l28-3 23-4 29-9 25-12 19-10 12-7 15-11 12-10 16-16 9-11 11-13 10-18 6-13 4-13 8-28 3-13 4-30 1-25v-39l-3-29-7-28-7-19-4-10-8-15-8-12-7-8-8-10-11-12-7-7v-2l-4-2-13-11-15-10-24-13-9-5-5-6-1-3v-11l3-6z" fill="#0B121F" />
    <path transform="translate(1369,1159)" d="m0 0 9 1 6 4 4 6 4 15 10 40 5 22 6 24 5 17 6 25 3 14 8 33 3 9 8 32 4 20 7 27 5 18 6 25 4 18 7 27 9 35 4 18 8 33 4 13 7 29 6 26 5 20 5 18 6 24 6 27 7 25 9 36v10l-3 6-6 5-9 2-8-3-16-8-19-10-18-11-29-16-12-6-21-12-21-13-24-14-12-6-22-13-15-10-27-16-16-9-21-13-13-9-24-15-14-8-23-15-16-11-17-11-11-7-9-5-20-13-10-7-23-15-14-9-9-5-17-11-12-8-10-7-25-16-14-8-18-12-5-6-3-9v-8l4-9 4-5 13-10 8-7 17-13 9-7 5-3 26-20 13-11 17-13 11-8 16-12 13-10 7-5 5-5 14-11 13-10 13-9 14-11 13-10 9-8 12-9 13-10 9-6 14-11 13-10 9-8 28-21 11-8 18-14 5-4h2l2-4 12-9 18-14 6-3z" fill="#0B121F" />
    <path transform="translate(714,1135)" d="m0 0h3l1 41v73l1 138-1 24h142l55 1h71l21 1-5 5-11 8-11 9-14 10-8 6-7 4-13 10-15 11h-2l-1 3-19 14-14 10-17 12-18 13-5 5-18 13-11 8-9 6-19 14-9 7-13 10-14 10-9 6-17 12-16 12-10 8-17 12-15 10-28 20h-2l-1 3-19 13-14 10-7 4-17 12-14 10-15 11-18 12-7 4-24 16-15 11-20 13-15 9-17 11-12 8-10 7-18 11-8 5-8 4-11 7-5-1-4-7-1-5v-13l7-30 4-13 7-29 3-15 7-28 4-14 6-25 3-15 7-28 4-15 6-24 3-15 7-28 4-15 6-24 3-15 7-28 4-15 6-24 5-23 5-20 4-15 6-24 3-15 7-28 4-15 6-24 3-15 7-28 5-18 2-5 25-10 48-17 10-3 37-13 15-6 37-13z" fill="#0B121F" />
    <path transform="translate(915,789)" d="m0 0h44l53 1 22 2 12 3 19 9 10 8 12 12 8 14 5 14 2 14v28l-3 19-7 16-7 12-4 2-2 4-10 9-13 8-11 5-12 3-8 1-31 1-36 1h-25l-29-1-12-1-1-23v-45l1-115z" fill="#0B121F" />
    <path transform="translate(870,776)" d="m0 0h170l22 4 15 5 15 8 13 10 11 11 10 16 6 15 4 20v41l-4 19-6 15-7 12-9 10-5 5-8 7-15 9-19 7-20 4-19 1h-154zm45 13-23 1-1 115v45l1 23 12 1 29 1h25l47-1 20-1 16-3 13-5 12-7 11-9 4-4v-2l4-2 8-14 6-14 3-19v-28l-2-14-5-14-8-14-15-15-11-7-18-8-16-3-24-1-44-1z" fill="#fff" />
  </svg>
);

const QuickSetupWizard = () => {
  const {
    currentStep,
    completedSteps,
    wizardData,
    isWizardCompleted,
    goToStep,
    nextStep,
    previousStep,
    markStepCompleted,
    updateWizardData,
    completeWizard,
    canNavigateToStep,
    saving,
    lastSaved,
  } = useWizardState();

  const [isProcessing, setIsProcessing] = useState(false);
  const [canGoNext, setCanGoNext] = useState(true);

  // Define wizard steps (free only)
  const wizardSteps = [
    {
      id: 'welcome',
      title: __('Welcome', 'prorank-seo'),
      description: __('Welcome and compatibility check', 'prorank-seo'),
      component: WelcomeStep,
    },
    {
      id: 'import',
      title: __('Import Data', 'prorank-seo'),
      description: __('Import from other SEO plugins', 'prorank-seo'),
      component: CompetitorImportStep,
    },
    {
      id: 'site-basics',
      title: __('Site Basics', 'prorank-seo'),
      description: __('Configure basic site information', 'prorank-seo'),
      component: SiteBasicsStep,
    },
    {
      id: 'modules',
      title: __('Modules', 'prorank-seo'),
      description: __('Choose SEO modules to activate', 'prorank-seo'),
      component: ModulesStep,
    },
    {
      id: 'optimization',
      title: __('Optimization', 'prorank-seo'),
      description: __('Quick optimization settings', 'prorank-seo'),
      component: OptimizationStep,
    },
    {
      id: 'ready',
      title: __('Ready!', 'prorank-seo'),
      description: __('Setup complete', 'prorank-seo'),
      component: ReadyStep,
    },
  ];

  const currentStepData = wizardSteps[currentStep];
  const StepComponent = currentStepData?.component;

  const handleNext = () => {
    if (!canGoNext) return;
    markStepCompleted(currentStep);
    if (currentStep < wizardSteps.length - 1) {
      nextStep();
      setCanGoNext(true);
    }
  };

  const handlePrevious = () => {
    previousStep();
  };

  const handleSkip = () => {
    const confirmSkip = window.confirm(__('Skip this step? You can always configure it later.', 'prorank-seo'));
    if (!confirmSkip) return;
    updateWizardData(`step_${currentStep}_skipped`, true);
    handleNext();
  };

  const handleComplete = async () => {
    setIsProcessing(true);
    try {
      await completeWizard();
      window.location.href = '/wp-admin/admin.php?page=prorank-seo';
    } finally {
      setIsProcessing(false);
    }
  };

  const handleStepClick = (stepIndex) => {
    if (canNavigateToStep(stepIndex)) {
      goToStep(stepIndex);
      setCanGoNext(true);
    }
  };

  if (isWizardCompleted) {
    return (
      <div className="pr-min-h-screen pr-flex pr-items-center pr-justify-center pr-p-8">
        <div className="pr-text-center pr-max-w-md">
          <div className="pr-inline-flex pr-items-center pr-justify-center pr-w-20 pr-h-20 pr-rounded-2xl pr-bg-emerald-100 pr-mb-6">
            <svg className="pr-w-10 pr-h-10 pr-text-emerald-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
            </svg>
          </div>
          <h2 className="pr-text-2xl pr-font-bold pr-text-gray-900 pr-mb-3">
            {__('Setup Already Complete', 'prorank-seo')}
          </h2>
          <p className="pr-text-gray-600 pr-mb-6">
            {__('The setup wizard has already been completed. You can configure individual settings from the menu.', 'prorank-seo')}
          </p>
          <a
            href="/wp-admin/admin.php?page=prorank-seo"
            className="pr-inline-flex pr-items-center pr-gap-2 pr-px-6 pr-py-3 pr-bg-blue-600 pr-text-white pr-font-medium pr-rounded-xl hover:pr-bg-blue-700 pr-transition-colors"
          >
            {__('Go to Dashboard', 'prorank-seo')}
          </a>
        </div>
      </div>
    );
  }

  return (
    <div className="pr-relative">
      {/* Background gradient - use absolute positioning within container */}
      <div
        className="pr-absolute pr-inset-0 pr-z-0 pr-pointer-events-none pr-overflow-hidden pr-rounded-xl"
        style={{
          background: 'linear-gradient(135deg, #f8fafc 0%, #f1f5f9 50%, #e2e8f0 100%)',
        }}
      >
        {/* Decorative elements */}
        <div className="pr-absolute pr-top-0 pr-right-0 pr-w-96 pr-h-96 pr-opacity-30">
          <div
            className="pr-absolute pr-inset-0 pr-rounded-full pr-blur-3xl"
            style={{ background: 'radial-gradient(circle, rgba(59, 130, 246, 0.3) 0%, transparent 70%)' }}
          />
        </div>
        <div className="pr-absolute pr-bottom-0 pr-left-0 pr-w-96 pr-h-96 pr-opacity-30">
          <div
            className="pr-absolute pr-inset-0 pr-rounded-full pr-blur-3xl"
            style={{ background: 'radial-gradient(circle, rgba(255, 107, 53, 0.3) 0%, transparent 70%)' }}
          />
        </div>
      </div>

      {/* Main content */}
      <div className="pr-relative pr-z-10 pr-p-4 md:pr-p-6 pr-max-w-4xl pr-mx-auto">
        {/* Header */}
        <div className="pr-mb-4">
          <div className="pr-flex pr-items-center pr-justify-center pr-gap-2 pr-mb-1">
            <ProrankLogo />
            <h1 className="pr-text-xl md:pr-text-2xl pr-font-bold pr-text-gray-900">
              {__('ProRank SEO', 'prorank-seo')}
            </h1>
          </div>
          <p className="pr-text-center pr-text-gray-400 pr-text-xs">
            {__('Quick Setup Wizard', 'prorank-seo')}
          </p>
        </div>

        {/* Wizard Card */}
        <div
          className="pr-bg-white pr-rounded-xl pr-shadow-lg pr-p-4 md:pr-p-6"
          style={{
            boxShadow: '0 10px 40px -10px rgba(0, 0, 0, 0.1), 0 0 0 1px rgba(0, 0, 0, 0.03)',
          }}
        >
          {/* Progress */}
          <WizardProgress
            steps={wizardSteps}
            currentStep={currentStep}
            completedSteps={completedSteps}
            onStepClick={handleStepClick}
          />

          {/* Step Content */}
          <div className="pr-min-h-[380px]">
            <Suspense fallback={<LoadingState message={__('Loading...', 'prorank-seo')} />}>
              {StepComponent && (
                <StepComponent
                  wizardData={wizardData}
                  updateWizardData={updateWizardData}
                  onNext={handleNext}
                  onValidationChange={setCanGoNext}
                />
              )}
            </Suspense>
          </div>

          {/* Navigation */}
          <WizardNavigation
            currentStep={currentStep}
            totalSteps={wizardSteps.length}
            onPrevious={handlePrevious}
            onNext={handleNext}
            onSkip={currentStep < wizardSteps.length - 1 ? handleSkip : null}
            onComplete={handleComplete}
            canGoNext={canGoNext}
            canGoPrevious={currentStep > 0}
            isProcessing={isProcessing}
            isLastStep={currentStep === wizardSteps.length - 1}
            saving={saving}
            lastSaved={lastSaved}
          />
        </div>

        {/* Footer */}
        <div className="pr-mt-4 pr-text-center">
          <p className="pr-text-[11px] pr-text-gray-400">
            {__('Need help?', 'prorank-seo')}{' '}
            <a
              href="https://docs.prorank.io"
              target="_blank"
              rel="noopener noreferrer"
              className="pr-text-blue-500 hover:pr-text-blue-600 hover:pr-underline"
            >
              {__('View docs', 'prorank-seo')}
            </a>
          </p>
        </div>
      </div>
    </div>
  );
};

export default QuickSetupWizard;
