/**
 * Dashboard Overview Tab
 *
 * Modern, clean dashboard with real data from API
 * Brand colors: Blue, Orange, Green (NO purple/indigo)
 */

import { __ } from '@wordpress/i18n';
import { useMemo } from '@wordpress/element';
import {
  CheckCircleIcon,
  DocumentTextIcon,
  ExclamationTriangleIcon,
  ArrowPathIcon,
  GlobeAltIcon,
  BoltIcon,
  ShieldCheckIcon,
  PhotoIcon,
  ArrowTrendingUpIcon,
  ArrowTrendingDownIcon,
  ServerIcon,
  CpuChipIcon,
  CircleStackIcon,
} from '@heroicons/react/24/outline';
import { Line, Bar } from 'react-chartjs-2';
import {
  Chart as ChartJS,
  CategoryScale,
  LinearScale,
  PointElement,
  LineElement,
  BarElement,
  Tooltip,
  Legend,
  Filler,
} from 'chart.js';

ChartJS.register(CategoryScale, LinearScale, PointElement, LineElement, BarElement, Tooltip, Legend, Filler);

const OverviewTab = ({ stats, onRefresh }) => {
  const audit = stats?.audit ?? null;
  const performance = stats?.performance ?? {};
  const severity = audit?.issues ?? null;
  const rum = performance?.rum ?? {};
  const images = performance?.images ?? {};
  const cache = performance?.cache ?? {};
  const assets = performance?.assets ?? {};

  const hasAudit = Boolean(audit && (audit.state === 'completed' || audit.last_run));
  const totalIssues = hasAudit ? (severity?.total ?? stats?.issues_count ?? null) : null;
  const seoScore = hasAudit ? (audit?.score ?? stats?.seo_score ?? null) : null;
  const seoScoreValue = seoScore ?? 0;
  const pagesCount = stats?.published_pages ?? null;
  const rumSamples = Number(rum?.samples ?? 0);
  const rumLastSampleTs = rum?.last_sample?.ts ? Number(rum.last_sample.ts) : null;
  const rumLastUpdated = rumLastSampleTs ? new Date(rumLastSampleTs * 1000) : null;
  const adminBase = window.prorankSeo?.adminUrl || window.proRankSEO?.adminUrl || window.prorankSEO?.adminUrl || '/wp-admin/';
  const adminUrl = adminBase.endsWith('/') ? adminBase : `${adminBase}/`;
  const pagespeedUrl = `https://pagespeed.web.dev/report?url=${encodeURIComponent(window.prorankSeoAdmin?.siteUrl || window.location.origin)}`;

  const formatBytes = (bytes) => {
    if (!bytes || bytes <= 0) return '0 B';
    const units = ['B', 'KB', 'MB', 'GB'];
    const exponent = Math.min(Math.floor(Math.log(bytes) / Math.log(1024)), units.length - 1);
    const value = bytes / Math.pow(1024, exponent);
    return `${value.toFixed(exponent === 0 ? 0 : 1)} ${units[exponent]}`;
  };

  const formatNumber = (value) => {
    if (value === null || value === undefined) return '—';
    return Number(value).toLocaleString();
  };

  const scoreTrendDelta = useMemo(() => {
    if (!audit?.history || audit.history.length < 2) return null;
    const [latest, previous] = audit.history;
    if (latest?.score === undefined || previous?.score === undefined) return null;
    return Math.round((latest.score - previous.score));
  }, [audit?.history]);

  const getScoreColor = (score) => {
    if (score === null || score === undefined) return { text: 'pr-text-gray-400', bg: 'pr-bg-gray-400' };
    if (score >= 80) return { text: 'pr-text-emerald-500', bg: 'pr-bg-emerald-500' };
    if (score >= 50) return { text: 'pr-text-orange-500', bg: 'pr-bg-orange-500' };
    return { text: 'pr-text-red-500', bg: 'pr-bg-red-500' };
  };

  const getVitalStatus = (metric, value) => {
    if (value === null || value === undefined) return 'unknown';
    const thresholds = {
      lcp: { good: 2500, poor: 4000 },
      inp: { good: 200, poor: 500 },
      cls: { good: 0.1, poor: 0.25 },
    };
    const t = thresholds[metric];
    if (!t) return 'unknown';
    if (value <= t.good) return 'good';
    if (value <= t.poor) return 'needs-improvement';
    return 'poor';
  };

  const severityData = useMemo(() => {
    const values = [
      severity?.critical || 0,
      severity?.high || 0,
      severity?.medium || 0,
      severity?.low || 0,
    ];
    return {
      labels: [__('Critical', 'prorank-seo'), __('High', 'prorank-seo'), __('Medium', 'prorank-seo'), __('Low', 'prorank-seo')],
      datasets: [
        {
          label: __('Issues', 'prorank-seo'),
          data: values,
          backgroundColor: ['#ef4444', '#f97316', '#eab308', '#22c55e'],
          borderRadius: 8,
          borderWidth: 0,
          barThickness: 32,
        },
      ],
    };
  }, [severity]);

  const auditHistoryData = useMemo(() => {
    const history = Array.isArray(audit?.history) ? audit.history.slice().reverse() : [];
    if (!history.length) return null;
    return {
      labels: history.map((run) => run?.completed_at ? new Date(run.completed_at).toLocaleDateString() : __('Run', 'prorank-seo')),
      datasets: [
        {
          label: __('Audit Score', 'prorank-seo'),
          data: history.map((run) => run?.score ?? 0),
          borderColor: '#3b82f6',
          backgroundColor: 'rgba(59, 130, 246, 0.1)',
          tension: 0.4,
          pointRadius: 5,
          pointHoverRadius: 7,
          pointBackgroundColor: '#3b82f6',
          fill: true,
        },
      ],
    };
  }, [audit?.history]);

  const scoreColor = getScoreColor(seoScore);

  return (
    <div className="pr-space-y-6 pr-p-6 pr-bg-gray-50 pr-min-h-screen">
      {/* Top Row: Score + Quick Stats */}
      <div className="pr-grid pr-grid-cols-1 lg:pr-grid-cols-4 pr-gap-6">
        {/* SEO Score Circle */}
        <div className="pr-bg-white pr-rounded-2xl pr-shadow-sm pr-border pr-border-gray-100 pr-p-6 pr-flex pr-flex-col pr-items-center pr-justify-center">
          <h3 className="pr-text-sm pr-font-semibold pr-text-gray-500 pr-uppercase pr-tracking-wider pr-mb-4">
            {__('SEO Score', 'prorank-seo')}
          </h3>
          <div className="pr-relative">
            <svg className="pr-w-32 pr-h-32 pr-transform -pr-rotate-90">
              <circle cx="64" cy="64" r="56" fill="none" stroke="#f3f4f6" strokeWidth="10" />
              <circle
                cx="64" cy="64" r="56" fill="none"
                stroke="url(#seoScoreGradient)"
                strokeWidth="10"
                strokeLinecap="round"
                strokeDasharray={`${(seoScoreValue / 100) * 352} 352`}
              />
              <defs>
                <linearGradient id="seoScoreGradient" x1="0%" y1="0%" x2="100%" y2="0%">
                  <stop offset="0%" stopColor="#3b82f6" />
                  <stop offset="100%" stopColor="#22c55e" />
                </linearGradient>
              </defs>
            </svg>
            <div className="pr-absolute pr-inset-0 pr-flex pr-flex-col pr-items-center pr-justify-center">
              <span className={`pr-text-3xl pr-font-bold ${scoreColor.text}`}>
                {seoScore === null || seoScore === undefined ? '—' : `${seoScore}%`}
              </span>
            </div>
          </div>
          {hasAudit && scoreTrendDelta !== null && (
            <div className={`pr-flex pr-items-center pr-gap-1 pr-mt-3 ${scoreTrendDelta >= 0 ? 'pr-text-emerald-600' : 'pr-text-red-600'}`}>
              {scoreTrendDelta >= 0 ? (
                <ArrowTrendingUpIcon className="pr-w-4 pr-h-4" />
              ) : (
                <ArrowTrendingDownIcon className="pr-w-4 pr-h-4" />
              )}
              <span className="pr-text-sm pr-font-medium">{scoreTrendDelta > 0 ? '+' : ''}{scoreTrendDelta}%</span>
            </div>
          )}
          <a
            href={`${adminUrl}admin.php?page=prorank-site-audit`}
            className="pr-mt-3 pr-text-sm pr-text-blue-600 hover:pr-underline"
          >
            {hasAudit ? __('View Audit →', 'prorank-seo') : __('Run Audit →', 'prorank-seo')}
          </a>
        </div>

        {/* Quick Stats Grid */}
        <div className="lg:pr-col-span-3 pr-grid pr-grid-cols-2 md:pr-grid-cols-3 pr-gap-4">
          {/* SEO Issues */}
          <div className="pr-bg-white pr-rounded-xl pr-shadow-sm pr-border pr-border-gray-100 pr-p-5">
            <div className="pr-flex pr-items-center pr-gap-3">
              <div className={`pr-p-2 pr-rounded-lg ${totalIssues > 0 ? 'pr-bg-orange-100' : 'pr-bg-emerald-100'}`}>
                <ExclamationTriangleIcon className={`pr-w-5 pr-h-5 ${totalIssues > 0 ? 'pr-text-orange-600' : 'pr-text-emerald-600'}`} />
              </div>
              <div>
                <p className="pr-text-2xl pr-font-bold pr-text-gray-900">{formatNumber(totalIssues)}</p>
                <p className="pr-text-sm pr-text-gray-500">{__('SEO Issues', 'prorank-seo')}</p>
              </div>
            </div>
          </div>

          {/* Published Pages */}
          <div className="pr-bg-white pr-rounded-xl pr-shadow-sm pr-border pr-border-gray-100 pr-p-5">
            <div className="pr-flex pr-items-center pr-gap-3">
              <div className="pr-p-2 pr-rounded-lg pr-bg-blue-100">
                <DocumentTextIcon className="pr-w-5 pr-h-5 pr-text-blue-600" />
              </div>
              <div>
                <p className="pr-text-2xl pr-font-bold pr-text-gray-900">{formatNumber(pagesCount)}</p>
                <p className="pr-text-sm pr-text-gray-500">{__('Published Pages', 'prorank-seo')}</p>
              </div>
            </div>
          </div>

          {/* Images Optimized */}
          <div className="pr-bg-white pr-rounded-xl pr-shadow-sm pr-border pr-border-gray-100 pr-p-5">
            <div className="pr-flex pr-items-center pr-gap-3">
              <div className="pr-p-2 pr-rounded-lg pr-bg-emerald-100">
                <PhotoIcon className="pr-w-5 pr-h-5 pr-text-emerald-600" />
              </div>
              <div>
                <p className="pr-text-2xl pr-font-bold pr-text-gray-900">
                  {formatNumber(images?.stats?.optimized_images ?? 0)}/{formatNumber(images?.stats?.total_images ?? 0)}
                </p>
                <p className="pr-text-sm pr-text-gray-500">{__('Images Optimized', 'prorank-seo')}</p>
              </div>
            </div>
            {images?.stats?.saved_bytes > 0 && (
              <p className="pr-mt-2 pr-text-xs pr-text-emerald-600 pr-font-medium">
                {formatBytes(images.stats.saved_bytes)} {__('saved', 'prorank-seo')}
              </p>
            )}
          </div>

          {/* Core Web Vitals */}
          <div className="pr-bg-white pr-rounded-xl pr-shadow-sm pr-border pr-border-gray-100 pr-p-5 md:pr-col-span-3">
            <div className="pr-flex pr-items-start pr-justify-between pr-gap-3 pr-mb-2">
              <div className="pr-flex pr-items-start pr-gap-2">
                <GlobeAltIcon className="pr-w-5 pr-h-5 pr-text-blue-600 pr-mt-0.5" />
                <div>
                  <h4 className="pr-font-semibold pr-text-gray-900">{__('Core Web Vitals', 'prorank-seo')}</h4>
                  <p className="pr-text-xs pr-text-gray-500">
                    {__('Field data (real users)', 'prorank-seo')}
                    {rumSamples > 0
                      ? ` · ${formatNumber(rumSamples)} ${__('samples', 'prorank-seo')}`
                      : ` · ${__('No field data yet', 'prorank-seo')}`}
                    {rumLastUpdated ? ` · ${__('Last updated', 'prorank-seo')} ${rumLastUpdated.toLocaleString()}` : ''}
                  </p>
                </div>
              </div>
              <a
                href={pagespeedUrl}
                target="_blank"
                rel="noreferrer"
                className="pr-inline-flex pr-items-center pr-gap-1 pr-text-xs pr-font-semibold pr-text-blue-600 hover:pr-text-blue-700"
              >
                <BoltIcon className="pr-w-4 pr-h-4" />
                {__('Run PageSpeed', 'prorank-seo')}
              </a>
            </div>
            <p className="pr-text-xs pr-text-gray-400 pr-mb-3">
              {__('Lab data from PageSpeed can differ from field data due to device and network conditions.', 'prorank-seo')}
            </p>
            <div className="pr-grid pr-grid-cols-3 pr-gap-4">
              {/* LCP */}
              <div className="pr-text-center">
                <p className="pr-text-xs pr-text-gray-500 pr-uppercase pr-tracking-wide">LCP</p>
                <p className={`pr-text-lg pr-font-bold ${
                  getVitalStatus('lcp', rum?.averages?.lcp) === 'good' ? 'pr-text-emerald-600' :
                  getVitalStatus('lcp', rum?.averages?.lcp) === 'needs-improvement' ? 'pr-text-orange-600' : 'pr-text-red-600'
                }`}>
                  {rum?.averages?.lcp === null || rum?.averages?.lcp === undefined ? '—' : `${(rum.averages.lcp / 1000).toFixed(2)}s`}
                </p>
              </div>
              {/* INP */}
              <div className="pr-text-center">
                <p className="pr-text-xs pr-text-gray-500 pr-uppercase pr-tracking-wide">INP</p>
                <p className={`pr-text-lg pr-font-bold ${
                  getVitalStatus('inp', rum?.averages?.inp) === 'good' ? 'pr-text-emerald-600' :
                  getVitalStatus('inp', rum?.averages?.inp) === 'needs-improvement' ? 'pr-text-orange-600' : 'pr-text-red-600'
                }`}>
                  {rum?.averages?.inp === null || rum?.averages?.inp === undefined ? '—' : `${Math.round(rum.averages.inp)}ms`}
                </p>
              </div>
              {/* CLS */}
              <div className="pr-text-center">
                <p className="pr-text-xs pr-text-gray-500 pr-uppercase pr-tracking-wide">CLS</p>
                <p className={`pr-text-lg pr-font-bold ${
                  getVitalStatus('cls', rum?.averages?.cls) === 'good' ? 'pr-text-emerald-600' :
                  getVitalStatus('cls', rum?.averages?.cls) === 'needs-improvement' ? 'pr-text-orange-600' : 'pr-text-red-600'
                }`}>
                  {rum?.averages?.cls === null || rum?.averages?.cls === undefined
                    ? '—'
                    : rum.averages.cls.toFixed(3)}
                </p>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Charts Row */}
      <div className="pr-grid pr-grid-cols-1 lg:pr-grid-cols-2 pr-gap-6">
        {/* Score Trend Chart */}
        <div className="pr-bg-white pr-rounded-2xl pr-shadow-sm pr-border pr-border-gray-100 pr-p-6">
          <div className="pr-flex pr-items-center pr-justify-between pr-mb-4">
            <h3 className="pr-font-semibold pr-text-gray-900">{__('Score Trend', 'prorank-seo')}</h3>
            {onRefresh && (
              <button
                onClick={onRefresh}
                className="pr-flex pr-items-center pr-gap-1 pr-text-sm pr-text-blue-600 hover:pr-text-blue-700 pr-font-medium"
              >
                <ArrowPathIcon className="pr-w-4 pr-h-4" />
                {__('Refresh', 'prorank-seo')}
              </button>
            )}
          </div>
          {auditHistoryData ? (
            <div className="pr-h-48">
              <Line
                data={auditHistoryData}
                options={{
                  responsive: true,
                  maintainAspectRatio: false,
                  plugins: { legend: { display: false } },
                  scales: {
                    y: { suggestedMin: 0, suggestedMax: 100, grid: { color: 'rgba(0,0,0,0.05)' } },
                    x: { grid: { display: false } },
                  },
                }}
              />
            </div>
          ) : (
            <div className="pr-h-48 pr-flex pr-items-center pr-justify-center pr-bg-gray-50 pr-rounded-xl">
              <p className="pr-text-gray-500">{__('Run a site audit to see trends', 'prorank-seo')}</p>
            </div>
          )}
        </div>

        {/* Issues by Severity Chart */}
        <div className="pr-bg-white pr-rounded-2xl pr-shadow-sm pr-border pr-border-gray-100 pr-p-6">
          <h3 className="pr-font-semibold pr-text-gray-900 pr-mb-4">{__('Issues by Severity', 'prorank-seo')}</h3>
          {!hasAudit ? (
            <div className="pr-h-48 pr-flex pr-items-center pr-justify-center pr-bg-gray-50 pr-rounded-xl">
              <p className="pr-text-gray-500">{__('Run a site audit to see issues', 'prorank-seo')}</p>
            </div>
          ) : totalIssues > 0 ? (
            <div className="pr-h-48">
              <Bar
                data={severityData}
                options={{
                  responsive: true,
                  maintainAspectRatio: false,
                  plugins: { legend: { display: false } },
                  scales: {
                    y: { beginAtZero: true, grid: { color: 'rgba(0,0,0,0.05)' } },
                    x: { grid: { display: false } },
                  },
                }}
              />
            </div>
          ) : (
            <div className="pr-h-48 pr-flex pr-items-center pr-justify-center pr-bg-emerald-50 pr-rounded-xl">
              <div className="pr-text-center">
                <CheckCircleIcon className="pr-w-10 pr-h-10 pr-text-emerald-500 pr-mx-auto pr-mb-2" />
                <p className="pr-text-emerald-700 pr-font-medium">{__('No issues detected', 'prorank-seo')}</p>
              </div>
            </div>
          )}
        </div>
      </div>

      {/* Performance Settings Cards */}
      <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-3 pr-gap-6">
        {/* Caching & Delivery */}
        <div className="pr-bg-white pr-rounded-2xl pr-shadow-sm pr-border pr-border-gray-100 pr-p-6">
          <div className="pr-flex pr-items-center pr-gap-3 pr-mb-4">
            <div className="pr-p-2 pr-bg-blue-100 pr-rounded-lg">
              <ServerIcon className="pr-w-5 pr-h-5 pr-text-blue-600" />
            </div>
            <h3 className="pr-font-semibold pr-text-gray-900">{__('Caching & Delivery', 'prorank-seo')}</h3>
          </div>
          <div className="pr-space-y-3">
            <div className="pr-flex pr-justify-between pr-items-center">
              <span className="pr-text-gray-600">{__('Browser Cache', 'prorank-seo')}</span>
              <span className={`pr-px-2 pr-py-1 pr-rounded-full pr-text-xs pr-font-medium ${
                cache?.browser_cache_enabled ? 'pr-bg-emerald-100 pr-text-emerald-700' : 'pr-bg-gray-100 pr-text-gray-600'
              }`}>
                {cache?.browser_cache_enabled ? __('On', 'prorank-seo') : __('Off', 'prorank-seo')}
              </span>
            </div>
            <div className="pr-flex pr-justify-between pr-items-center">
              <span className="pr-text-gray-600">{__('Resource Hints', 'prorank-seo')}</span>
              <span className={`pr-px-2 pr-py-1 pr-rounded-full pr-text-xs pr-font-medium ${
                cache?.link_preload_enabled ? 'pr-bg-emerald-100 pr-text-emerald-700' : 'pr-bg-gray-100 pr-text-gray-600'
              }`}>
                {cache?.link_preload_enabled ? __('On', 'prorank-seo') : __('Off', 'prorank-seo')}
              </span>
            </div>
            <div className="pr-flex pr-justify-between pr-items-center">
              <span className="pr-text-gray-600">{__('JS Minify', 'prorank-seo')}</span>
              <span className={`pr-px-2 pr-py-1 pr-rounded-full pr-text-xs pr-font-medium ${
                assets?.js_minify ? 'pr-bg-emerald-100 pr-text-emerald-700' : 'pr-bg-gray-100 pr-text-gray-600'
              }`}>
                {assets?.js_minify ? __('On', 'prorank-seo') : __('Off', 'prorank-seo')}
              </span>
            </div>
          </div>
        </div>

        {/* Assets & Fonts */}
        <div className="pr-bg-white pr-rounded-2xl pr-shadow-sm pr-border pr-border-gray-100 pr-p-6">
          <div className="pr-flex pr-items-center pr-gap-3 pr-mb-4">
            <div className="pr-p-2 pr-bg-orange-100 pr-rounded-lg">
              <CpuChipIcon className="pr-w-5 pr-h-5 pr-text-orange-600" />
            </div>
            <h3 className="pr-font-semibold pr-text-gray-900">{__('Assets & Fonts', 'prorank-seo')}</h3>
          </div>
          <div className="pr-space-y-3">
            <div className="pr-flex pr-justify-between pr-items-center">
              <span className="pr-text-gray-600">{__('JS Minify', 'prorank-seo')}</span>
              <span className={`pr-px-2 pr-py-1 pr-rounded-full pr-text-xs pr-font-medium ${
                assets?.js_minify ? 'pr-bg-emerald-100 pr-text-emerald-700' : 'pr-bg-gray-100 pr-text-gray-600'
              }`}>
                {assets?.js_minify ? __('On', 'prorank-seo') : __('Off', 'prorank-seo')}
              </span>
            </div>
            <div className="pr-flex pr-justify-between pr-items-center">
              <span className="pr-text-gray-600">{__('Font Optimization', 'prorank-seo')}</span>
              <span className={`pr-px-2 pr-py-1 pr-rounded-full pr-text-xs pr-font-medium ${
                assets?.font_optimization ? 'pr-bg-emerald-100 pr-text-emerald-700' : 'pr-bg-gray-100 pr-text-gray-600'
              }`}>
                {assets?.font_optimization ? __('On', 'prorank-seo') : __('Off', 'prorank-seo')}
              </span>
            </div>
          </div>
        </div>

        {/* Images */}
        <div className="pr-bg-white pr-rounded-2xl pr-shadow-sm pr-border pr-border-gray-100 pr-p-6">
          <div className="pr-flex pr-items-center pr-gap-3 pr-mb-4">
            <div className="pr-p-2 pr-bg-emerald-100 pr-rounded-lg">
              <CircleStackIcon className="pr-w-5 pr-h-5 pr-text-emerald-600" />
            </div>
            <h3 className="pr-font-semibold pr-text-gray-900">{__('Images', 'prorank-seo')}</h3>
          </div>
          <div className="pr-space-y-3">
            <div className="pr-flex pr-justify-between pr-items-center">
              <span className="pr-text-gray-600">{__('Space Saved', 'prorank-seo')}</span>
              <span className="pr-text-sm pr-font-medium pr-text-emerald-600">
                {formatBytes(images?.stats?.saved_bytes || 0)}
              </span>
            </div>
            <div className="pr-flex pr-justify-between pr-items-center">
              <span className="pr-text-gray-600">{__('Formats', 'prorank-seo')}</span>
              <span className="pr-text-sm pr-font-medium pr-text-gray-900">
                {[images?.webp_enabled && 'WebP', images?.avif_enabled && 'AVIF'].filter(Boolean).join(' ') || __('Standard', 'prorank-seo')}
              </span>
            </div>
            <div className="pr-flex pr-justify-between pr-items-center">
              <span className="pr-text-gray-600">{__('Auto Optimize Uploads', 'prorank-seo')}</span>
              <span className={`pr-px-2 pr-py-1 pr-rounded-full pr-text-xs pr-font-medium ${
                images?.optimize_on_upload ? 'pr-bg-emerald-100 pr-text-emerald-700' : 'pr-bg-gray-100 pr-text-gray-600'
              }`}>
                {images?.optimize_on_upload ? __('On', 'prorank-seo') : __('Off', 'prorank-seo')}
              </span>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default OverviewTab;
