/**
 * Module Manager Tab (Free)
 *
 * Manages top-level module groups without any upgrade gating.
 */

import { useState, useEffect } from '@wordpress/element';
import { __, sprintf } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { Card, Button, Toggle, Alert } from '../../components/ui';
import {
  ShieldCheckIcon,
  PencilIcon,
  RocketLaunchIcon,
  StarIcon as StarSolidIcon,
} from '@heroicons/react/24/solid';
import { StarIcon as StarOutlineIcon } from '@heroicons/react/24/outline';
import { useNotification } from '../../contexts/NotificationContext';
import { useUserPreferences } from '../../hooks/useUserPreferences';
import LoadingState from '../../components/LoadingState';

// Icon mapping for module groups
const moduleIcons = {
  'on-page-seo': PencilIcon,
  'technical-seo': ShieldCheckIcon,
  performance: RocketLaunchIcon,
};

const ModuleManagerTab = () => {
  const [moduleGroups, setModuleGroups] = useState([]);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState({});
  const [error, setError] = useState(null);
  const [updatingPin, setUpdatingPin] = useState({});

  const { showNotification } = useNotification();
  const { pinnedModules, togglePinnedModule } = useUserPreferences();

  useEffect(() => {
    fetchModuleGroups();
  }, []);

  const fetchModuleGroups = async () => {
    setLoading(true);
    setError(null);

    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/modules/groups',
      });

      setModuleGroups(response.groups || []);
    } catch (err) {
      setError(err.message || __('Failed to load module groups', 'prorank-seo'));
    } finally {
      setLoading(false);
    }
  };

  const handleToggleModule = async (groupSlug, newState) => {
    setSaving({ ...saving, [groupSlug]: true });

    try {
      await apiFetch({
        path: `/prorank-seo/v1/modules/groups/${groupSlug}/status`,
        method: 'POST',
        data: { enabled: newState },
      });

      setModuleGroups((groups) =>
        groups.map((g) => (g.slug === groupSlug ? { ...g, enabled: newState } : g))
      );

      const group = moduleGroups.find((g) => g.slug === groupSlug);
      showNotification(
        sprintf(
          __('Module group "%1$s" has been %2$s', 'prorank-seo'),
          group?.name || groupSlug,
          newState ? __('enabled', 'prorank-seo') : __('disabled', 'prorank-seo')
        ),
        'success'
      );
    } catch (err) {
      showNotification(err.message || __('Failed to update module status', 'prorank-seo'), 'error');
    } finally {
      setSaving({ ...saving, [groupSlug]: false });
    }
  };

  const getModulePageUrl = (slug) => `admin.php?page=prorank-${slug}`;

  const handleTogglePin = async (moduleSlug) => {
    setUpdatingPin({ ...updatingPin, [moduleSlug]: true });

    try {
      const result = await togglePinnedModule(moduleSlug);

      showNotification(
        result.is_pinned
          ? __('Module pinned to top', 'prorank-seo')
          : __('Module unpinned', 'prorank-seo'),
        'success'
      );
    } catch (err) {
      showNotification(__('Failed to update pinned status', 'prorank-seo'), 'error');
    } finally {
      setUpdatingPin({ ...updatingPin, [moduleSlug]: false });
    }
  };

  const renderModuleCard = (group) => {
    const isSaving = saving[group.slug];
    const isPinned = pinnedModules.includes(group.slug);
    const isPinning = updatingPin[group.slug];
    const IconComponent = moduleIcons[group.slug] || ShieldCheckIcon;

    return (
      <Card
        key={group.slug}
        className="pr-relative"
        variant="elevated"
      >
        <div className="pr-flex pr-items-start pr-gap-4 pr-mb-4">
          <div className={`pr-p-3 pr-rounded-lg ${group.enabled ? 'pr-bg-primary-100' : 'pr-bg-gray-100'}`}>
            <IconComponent className={`pr-w-6 pr-h-6 ${group.enabled ? 'pr-text-primary-600' : 'pr-text-gray-600'}`} />
          </div>
          <div className="pr-flex-1">
            <div className="pr-flex pr-items-center pr-gap-2 pr-mb-1">
              <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">{group.name}</h3>
            </div>
            <p className="pr-text-sm pr-text-gray-600">{group.description}</p>
          </div>
          <button
            onClick={() => handleTogglePin(group.slug)}
            disabled={isPinning}
            className="pr-p-2 pr-rounded-md hover:pr-bg-gray-100 pr-transition-colors disabled:pr-opacity-50"
            aria-label={isPinned ? __('Unpin module', 'prorank-seo') : __('Pin module', 'prorank-seo')}
          >
            {isPinned ? (
              <StarSolidIcon className="pr-w-5 pr-h-5 pr-text-warning-500" />
            ) : (
              <StarOutlineIcon className="pr-w-5 pr-h-5 pr-text-gray-400" />
            )}
          </button>
        </div>

        <div className="pr-flex pr-items-center pr-justify-between pr-pt-4 pr-border-t pr-border-gray-200">
          <Toggle
            checked={Boolean(group.enabled)}
            onChange={(value) => handleToggleModule(group.slug, value)}
            disabled={isSaving}
            label={group.enabled ? __('Enabled', 'prorank-seo') : __('Disabled', 'prorank-seo')}
          />

          <Button
            variant="primary"
            size="sm"
            onClick={() => {
              if (group.enabled) {
                window.location.href = getModulePageUrl(group.slug);
              }
            }}
            disabled={isSaving || !group.enabled}
          >
            {__('Configure', 'prorank-seo')}
          </Button>
        </div>

        {isSaving && (
          <div className="pr-absolute pr-inset-0 pr-bg-white pr-bg-opacity-75 pr-flex pr-items-center pr-justify-center pr-rounded-lg">
            <LoadingState message={__('Updating…', 'prorank-seo')} />
          </div>
        )}
      </Card>
    );
  };

  if (loading) {
    return <LoadingState message={__('Loading modules…', 'prorank-seo')} />;
  }

  if (error) {
    return (
      <Alert variant="error" title={__('Error', 'prorank-seo')}>
        {error}
      </Alert>
    );
  }

  return (
    <div className="pr-space-y-6">
      <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-2 xl:pr-grid-cols-3 pr-gap-6">
        {moduleGroups.map(renderModuleCard)}
      </div>
    </div>
  );
};

export default ModuleManagerTab;
