/**
 * Bulk Edit Tab - Titles & Meta Settings
 *
 * Allows bulk editing of SEO titles and descriptions.
 *
 * @package
 * @since   1.0.0
 */

import { useState, useEffect, useCallback } from '@wordpress/element';
import { __, sprintf } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { debounce } from '../../../shims/wp-utils-shim';
import { Card, Button, Select, Input, Alert } from '../../components/ui';
import LoadingState from '../../components/LoadingState';
import { useNotification } from '../../contexts/NotificationContext';

// Helper component for radio buttons
const RadioGroup = ({ label, options, selected, onChange }) => (
  <div className="pr-space-y-2">
    <label className="pr-block pr-text-sm pr-font-medium pr-text-gray-700">{label}</label>
    <div className="pr-space-y-2">
      {options.map((option) => (
        <label key={option.value} className="pr-flex pr-items-center pr-gap-2 pr-cursor-pointer">
          <input
            type="radio"
            value={option.value}
            checked={selected === option.value}
            onChange={(e) => onChange(e.target.value)}
            className="pr-w-4 pr-h-4 pr-text-primary-600 pr-border-gray-300 focus:pr-ring-primary-500"
          />
          <span className="pr-text-sm pr-text-gray-700">{option.label}</span>
        </label>
      ))}
    </div>
  </div>
);

// Helper component for checkbox
const Checkbox = ({ label, checked, onChange }) => (
  <label className="pr-flex pr-items-center pr-gap-2 pr-cursor-pointer">
    <input
      type="checkbox"
      checked={checked}
      onChange={(value) => {
        const nextChecked = typeof value === 'boolean' ? value : !!value?.target?.checked;
        onChange(nextChecked);
      }}
      className="pr-w-4 pr-h-4 pr-text-primary-600 pr-border-gray-300 pr-rounded focus:pr-ring-primary-500"
    />
    <span className="pr-text-sm pr-text-gray-700">{label}</span>
  </label>
);

const BulkEditTab = () => {
  const [loading, setLoading] = useState(false);
  const [saving, setSaving] = useState(false);
  const [posts, setPosts] = useState([]);
  const [selectedPosts, setSelectedPosts] = useState([]);
  const [postType, setPostType] = useState('post');
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [totalPosts, setTotalPosts] = useState(0);
  const [searchTerm, setSearchTerm] = useState('');
  const [editedData, setEditedData] = useState({});
  const [error, setError] = useState(null);
  const [success, setSuccess] = useState(null);
  const [availablePostTypes, setAvailablePostTypes] = useState([]);
  const [settings, setSettings] = useState({});
  const { showNotification } = useNotification();

  // Debounced search
  const debouncedSearch = useCallback(
    debounce((term) => {
      loadPosts(1, term);
    }, 500),
    [postType]
  );

  useEffect(() => {
    // Load settings and available post types
    loadSettings();
  }, []);

  useEffect(() => {
    if (postType) {
      loadPosts();
    }
  }, [postType]);

  useEffect(() => {
    if (searchTerm) {
      debouncedSearch(searchTerm);
    } else if (postType) {
      loadPosts();
    }
  }, [searchTerm]);

  const loadSettings = async () => {
    try {
      // Load titles-meta settings to check which post types have bulk editing enabled
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/titles-meta',
      });
      
      setSettings(response || {});
      
      // Get all post types from WordPress
      const typesResponse = await apiFetch({
        path: '/wp/v2/types?context=edit',
      });
      
      // Filter to only show post types that have bulk editing enabled
      const types = [];
      Object.entries(typesResponse).forEach(([key, type]) => {
        // Check if bulk editing is enabled for this post type
        const postTypeSettings = response[`post_type_${key}`] || {};
        // Always show posts and pages, check setting for others
        if (key === 'post' || key === 'page' || postTypeSettings.bulk_editing === true) {
          types.push({
            value: key,
            label: type.labels.name,
          });
        }
      });
      
      setAvailablePostTypes(types);

      // Set initial post type if available
      if (types.length > 0 && !postType) {
        setPostType(types[0].value);
      }
    } catch (err) {
      // Fallback to default post types
      setAvailablePostTypes([
        { label: __('Posts', 'prorank-seo'), value: 'post' },
        { label: __('Pages', 'prorank-seo'), value: 'page' },
      ]);
    }
  };

  const loadPosts = async (page = 1, search = '') => {
    setLoading(true);
    setError(null);
    
    try {
      const response = await apiFetch({
        path: `/prorank-seo/v1/titles-meta/bulk-edit?post_type=${postType}&paged=${page}&per_page=20&search=${search}`,
      });

      if (response.success && response.data) {
        setPosts(response.data.posts);
        setTotalPages(response.data.pages);
        setTotalPosts(response.data.total);
        setCurrentPage(response.data.current_page);
        setSelectedPosts([]);
        setEditedData({});
      }
    } catch (err) {
      setError(__('Failed to load posts.', 'prorank-seo'));
    } finally {
      setLoading(false);
    }
  };

  const handleSelectAll = (checked) => {
    if (checked) {
      setSelectedPosts(posts.map(post => post.id));
    } else {
      setSelectedPosts([]);
    }
  };

  const handleSelectPost = (postId, checked) => {
    if (checked) {
      setSelectedPosts([...selectedPosts, postId]);
    } else {
      setSelectedPosts(selectedPosts.filter(id => id !== postId));
    }
  };

  const handleFieldEdit = (postId, field, value) => {
    setEditedData(prev => ({
      ...prev,
      [postId]: {
        ...prev[postId],
        id: postId,
        [field]: value,
      }
    }));
  };

  const saveChanges = async () => {
    const updates = Object.values(editedData).filter(data => 
      selectedPosts.includes(data.id)
    );

    if (updates.length === 0) {
      setError(__('No changes to save.', 'prorank-seo'));
      return;
    }

    setSaving(true);
    setError(null);
    setSuccess(null);

    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/titles-meta/bulk-edit',
        method: 'POST',
        data: { updates },
      });

      if (response.success) {
        showNotification(response.message, 'success');
        setEditedData({});
        loadPosts(currentPage, searchTerm);
      } else {
        throw new Error(response.message || __('Failed to save changes.', 'prorank-seo'));
      }
    } catch (err) {
      setError(err.message);
      showNotification(__('Failed to save changes.', 'prorank-seo'), 'error');
    } finally {
      setSaving(false);
    }
  };

  const getFieldValue = (post, field) => {
    if (editedData[post.id] && editedData[post.id][field] !== undefined) {
      return editedData[post.id][field];
    }
    return post[field] || '';
  };

  // Enhanced Character Progress with color-coded progress bar
  const CharacterProgress = ({ text, type = 'title' }) => {
    const length = text?.length || 0;

    // Define ranges based on type
    const ranges = type === 'title'
      ? { min: 30, optMin: 50, optMax: 60, warnMax: 65, max: 70 }
      : { min: 70, optMin: 120, optMax: 155, warnMax: 160, max: 160 };

    // Calculate status
    const getStatus = () => {
      if (length === 0) return { color: 'gray' };
      if (length < ranges.min) return { color: 'red' };
      if (length < ranges.optMin) return { color: 'yellow' };
      if (length <= ranges.optMax) return { color: 'green' };
      if (length <= ranges.warnMax) return { color: 'yellow' };
      return { color: 'red' };
    };

    const status = getStatus();
    const percent = length === 0 ? 0 : Math.min(100, (length / ranges.max) * 100);

    const barColors = {
      gray: 'pr-bg-gray-300',
      red: 'pr-bg-red-500',
      yellow: 'pr-bg-yellow-500',
      green: 'pr-bg-green-500',
    };

    const textColors = {
      gray: 'pr-text-gray-500',
      red: 'pr-text-red-600',
      yellow: 'pr-text-yellow-600',
      green: 'pr-text-green-600',
    };

    return (
      <div className="pr-mt-1.5">
        {/* Progress bar */}
        <div className="pr-h-1.5 pr-bg-gray-200 pr-rounded-full pr-overflow-hidden">
          <div
            className={`pr-h-full pr-transition-all pr-duration-300 pr-rounded-full ${barColors[status.color]}`}
            style={{ width: `${percent}%` }}
          />
        </div>
        {/* Counter text */}
        <div className="pr-flex pr-justify-between pr-items-center pr-mt-1">
          <span className={`pr-text-xs pr-font-medium ${textColors[status.color]}`}>
            {length}
          </span>
          <span className="pr-text-xs pr-text-gray-400">
            {type === 'title' ? '50-60' : '120-155'}
          </span>
        </div>
      </div>
    );
  };

  return (
    <div className="pr-space-y-6">
      {error && (
        <Alert variant="error" dismissible onDismiss={() => setError(null)}>
          {error}
        </Alert>
      )}

      {success && (
        <Alert variant="success" dismissible onDismiss={() => setSuccess(null)}>
          {success}
        </Alert>
      )}

      {availablePostTypes.length === 0 && !loading && (
        <Alert variant="info">
          <p>
            <strong>{__('No post types available for bulk editing.', 'prorank-seo')}</strong>
          </p>
          <p>
            {__('To enable bulk editing for a post type:', 'prorank-seo')}
          </p>
          <ol className="pr-ml-5 pr-mt-2 pr-list-decimal">
            <li>{__('Go to the Post Types tab', 'prorank-seo')}</li>
            <li>{__('Select the post type you want to edit (Posts, Pages, etc.)', 'prorank-seo')}</li>
            <li>{__('Enable the "Enable Bulk Editing" option', 'prorank-seo')}</li>
            <li>{__('Save your changes', 'prorank-seo')}</li>
          </ol>
          <p className="pr-mt-2">
            <em>{__('Note: Posts and Pages are always available for bulk editing by default.', 'prorank-seo')}</em>
          </p>
        </Alert>
      )}

      <Card>
        <div className="pr-p-6 pr-border-b pr-border-gray-200">
          <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-3 pr-gap-4">
            <div>
              <label className="pr-block pr-text-sm pr-font-medium pr-text-gray-700 pr-mb-2">
                {__('Post Type', 'prorank-seo')}
              </label>
              <Select
                value={postType}
                options={availablePostTypes}
                onChange={(e) => setPostType(e.target.value)}
              />
              {availablePostTypes.length === 0 && (
                <p className="pr-text-xs pr-text-gray-500 pr-mt-1">
                  {__('No post types available. Enable bulk editing in Post Types settings.', 'prorank-seo')}
                </p>
              )}
            </div>

            <div>
              <label className="pr-block pr-text-sm pr-font-medium pr-text-gray-700 pr-mb-2">
                {__('Search', 'prorank-seo')}
              </label>
              <Input
                type="text"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                placeholder={__('Search posts...', 'prorank-seo')}
              />
            </div>

            <div className="pr-mt-6 pr-flex pr-gap-2 pr-transition-all pr-duration-300">
              <Button
                variant="primary"
                onClick={saveChanges}
                disabled={saving || selectedPosts.length === 0}
                isBusy={saving}
              >
                {saving ? __('Saving...', 'prorank-seo') : __('Save Changes', 'prorank-seo')}
              </Button>
            </div>
          </div>
        </div>
        <div className="pr-p-6">
          {loading ? (
            <div className="pr-text-center pr-py-10 pr-transition-all pr-duration-300">
              <LoadingState />
              <p className="pr-mt-4 pr-text-gray-600">{__('Loading posts...', 'prorank-seo')}</p>
            </div>
          ) : posts.length === 0 ? (
            <Alert variant="info">
              {__('No posts found.', 'prorank-seo')}
            </Alert>
          ) : (
            <>
              <div className="pr-mb-5 pr-transition-all pr-duration-300">
                <Checkbox
                  label={__('Select All', 'prorank-seo')}
                  checked={selectedPosts.length === posts.length && posts.length > 0}
                  onChange={handleSelectAll}
                />
                <p className="pr-mt-2 pr-text-gray-500 pr-transition-all pr-duration-300">
                  {selectedPosts.length > 0
                    ? sprintf(__('%d items selected', 'prorank-seo'), selectedPosts.length)
                    : __('Select items to edit', 'prorank-seo')
                  }
                </p>
              </div>

              <div className="bulk-edit-table pr-overflow-x-auto">
                <table className="wp-list-table widefat fixed striped">
                  <thead>
                    <tr>
                      <th className="pr-w-8 pr-transition-all pr-duration-300"></th>
                      <th>{__('Title', 'prorank-seo')}</th>
                      <th>
                        {__('SEO Title', 'prorank-seo')}
                        <span className="pr-text-xs pr-text-gray-400 pr-font-normal pr-ml-1">(50-60)</span>
                      </th>
                      <th>
                        {__('Meta Description', 'prorank-seo')}
                        <span className="pr-text-xs pr-text-gray-400 pr-font-normal pr-ml-1">(120-155)</span>
                      </th>
                      <th>{__('Focus Keyword', 'prorank-seo')}</th>
                    </tr>
                  </thead>
                  <tbody>
                    {posts.map(post => (
                      <tr key={post.id} className={selectedPosts.includes(post.id) ? 'selected' : ''}>
                        <td>
                          <Checkbox
                            checked={selectedPosts.includes(post.id)}
                            onChange={(checked) => handleSelectPost(post.id, checked)}
                          />
                        </td>
                        <td>
                          <strong>{post.title}</strong>
                          <br />
                          <span className="pr-text-xs pr-text-gray-500 pr-transition-all pr-duration-300">
                            <a href={post.url} target="_blank" rel="noopener noreferrer" className="hover:pr-text-primary-600">
                              {__('View', 'prorank-seo')}
                            </a>
                          </span>
                        </td>
                        <td>
                          <div className="pr-relative">
                            <Input
                              type="text"
                              value={getFieldValue(post, 'seo_title')}
                              onChange={(e) => handleFieldEdit(post.id, 'seo_title', e.target.value)}
                              placeholder={post.title}
                              disabled={!selectedPosts.includes(post.id)}
                            />
                          </div>
                          <CharacterProgress
                            text={getFieldValue(post, 'seo_title') || post.title}
                            type="title"
                          />
                        </td>
                        <td>
                          <div className="pr-relative">
                            <textarea
                              className="pr-w-full pr-px-3 pr-py-2 pr-border pr-border-gray-300 pr-rounded-md focus:pr-outline-none focus:pr-ring-2 focus:pr-ring-primary-500 pr-transition-all pr-duration-300"
                              value={getFieldValue(post, 'seo_description')}
                              onChange={(e) => handleFieldEdit(post.id, 'seo_description', e.target.value)}
                              placeholder={post.excerpt || __('Auto-generated from content', 'prorank-seo')}
                              disabled={!selectedPosts.includes(post.id)}
                              rows={2}
                            />
                          </div>
                          <CharacterProgress
                            text={getFieldValue(post, 'seo_description') || post.excerpt || ''}
                            type="description"
                          />
                        </td>
                        <td>
                          <Input
                            type="text"
                            value={getFieldValue(post, 'focus_keyword')}
                            onChange={(e) => handleFieldEdit(post.id, 'focus_keyword', e.target.value)}
                            placeholder={__('Focus keyword', 'prorank-seo')}
                            disabled={!selectedPosts.includes(post.id)}
                          />
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>

              {totalPages > 1 && (
                <div className="pr-mt-5 pr-text-center pr-flex pr-items-center pr-justify-center pr-gap-4 pr-transition-all pr-duration-300">
                  <Button
                    variant="secondary"
                    onClick={() => loadPosts(currentPage - 1, searchTerm)}
                    disabled={currentPage === 1}
                  >
                    {__('Previous', 'prorank-seo')}
                  </Button>
                  <span className="pr-text-gray-600 pr-transition-all pr-duration-300">
                    {sprintf(__('Page %d of %d', 'prorank-seo'), currentPage, totalPages)}
                  </span>
                  <Button
                    variant="secondary"
                    onClick={() => loadPosts(currentPage + 1, searchTerm)}
                    disabled={currentPage === totalPages}
                  >
                    {__('Next', 'prorank-seo')}
                  </Button>
                </div>
              )}
            </>
          )}
        </div>
      </Card>

    </div>
  );
};

export default BulkEditTab;
