/**
 * useToast Hook
 *
 * React hook for displaying toast notifications in the ProRank SEO admin interface.
 *
 * @since 0.1.0
 */

import { useCallback } from 'react';

/**
 * Toast notification hook
 *
 * @return {Object} Toast methods for showing different types of notifications
 */
const useToast = () => {
  /**
   * Show a toast notification
   *
   * @param {string} message - The message to display
   * @param {string} type    - The type of notification ('info', 'success', 'error', 'warning')
   * @param {Object} options - Additional options for the toast
   * @return {void}
   */
  const showToast = useCallback((message, type = 'info', options = {}) => {
    if (
      window.proRankSEO !== null &&
      window.proRankSEO !== undefined &&
      window.proRankSEO.showToast !== null &&
      window.proRankSEO.showToast !== undefined
    ) {
      window.proRankSEO.showToast(message, type, options);
    } else {
      // Fallback - toast system not available
    }
  }, []);

  /**
   * Show a success toast
   *
   * @param {string} message - The success message to display
   * @param {Object} options - Additional options for the toast
   * @return {void}
   */
  const success = useCallback(
    (message, options = {}) => {
      showToast(message, 'success', options);
    },
    [showToast]
  );

  /**
   * Show an error toast
   *
   * @param {string} message - The error message to display
   * @param {Object} options - Additional options for the toast
   * @return {void}
   */
  const error = useCallback(
    (message, options = {}) => {
      showToast(message, 'error', options);
    },
    [showToast]
  );

  /**
   * Show a warning toast
   *
   * @param {string} message - The warning message to display
   * @param {Object} options - Additional options for the toast
   * @return {void}
   */
  const warning = useCallback(
    (message, options = {}) => {
      showToast(message, 'warning', options);
    },
    [showToast]
  );

  /**
   * Show an info toast
   *
   * @param {string} message - The info message to display
   * @param {Object} options - Additional options for the toast
   * @return {void}
   */
  const info = useCallback(
    (message, options = {}) => {
      showToast(message, 'info', options);
    },
    [showToast]
  );

  return {
    showToast,
    success,
    error,
    warning,
    info,
  };
};

export default useToast;
