import React, { createContext, useContext, useState, useEffect, useCallback } from 'react';
import apiFetch from '@wordpress/api-fetch';
import { __ } from '@wordpress/i18n';

const SupportAccountLicenseContext = createContext({
  license: null,
  refreshLicense: () => {},
  loading: true,
});

export const SupportAccountLicenseProvider = ({ children }) => {
  const [license, setLicense] = useState(null);
  const [loading, setLoading] = useState(true);

  const refreshLicense = useCallback(async () => {
    try {
      setLoading(true);
      const response = await apiFetch({
        path: '/prorank-seo/v1/license/status',
      });

      if (response.success) {
        const nextLicense = response.license || null;
        if (nextLicense && nextLicense.tier === 'core') {
          nextLicense.tier = 'pro';
        }
        setLicense(nextLicense);
      } else {
        // If no license, set to inactive state
        setLicense({
          status: 'inactive',
          tier: 'free',
        });
      }
    } catch (error) {
      // On error, set to inactive state
      setLicense({
        status: 'inactive',
        tier: 'free',
      });
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    refreshLicense();
  }, [refreshLicense]);

  return (
    <SupportAccountLicenseContext.Provider value={{ license, refreshLicense, loading }}>
      {children}
    </SupportAccountLicenseContext.Provider>
  );
};

export const useLicense = () => {
  const context = useContext(SupportAccountLicenseContext);
  if (!context) {
    throw new Error('useLicense must be used within a SupportAccountLicenseProvider');
  }
  return context;
};
