/**
 * Toggle Component - ProRank SEO
 *
 * Premium toggle switch component for boolean settings.
 * Modern alternative to checkboxes for on/off states.
 *
 * @example
 * <Toggle label="Enable Feature" checked={enabled} onChange={setEnabled} />
 * <Toggle
 *   label="Advanced Mode"
 *   description="Enable advanced features"
 *   checked={advanced}
 *   onChange={setAdvanced}
 *   size="lg"
 * />
 */

import React from 'react';

// Size configurations
const sizeConfig = {
  sm: { track: { width: 44, height: 26 }, thumb: 20 },
  md: { track: { width: 50, height: 28 }, thumb: 22 },
  lg: { track: { width: 56, height: 32 }, thumb: 26 },
};

const Toggle = React.forwardRef(({
  label,
  description,
  id,
  name,
  checked = false,
  onChange,
  disabled = false,
  error,
  className = '',
  size = 'md',
  ...props
}, ref) => {
  const toggleId = id || name || `toggle-${Math.random().toString(36).substr(2, 9)}`;
  const hasError = Boolean(error);
  const config = sizeConfig[size] || sizeConfig.md;

  // Handle toggle change
  const handleToggle = () => {
    if (!disabled && onChange) {
      onChange(!checked);
    }
  };

  // Handle keyboard events
  const handleKeyDown = (e) => {
    if (e.key === ' ' || e.key === 'Enter') {
      e.preventDefault();
      handleToggle();
    }
  };

  // Track styles (the pill-shaped background)
  const trackStyle = {
    position: 'relative',
    display: 'inline-flex',
    alignItems: 'center',
    width: config.track.width,
    height: config.track.height,
    borderRadius: 9999,
    backgroundColor: checked ? '#3b82f6' : '#d1d5db',
    cursor: disabled ? 'not-allowed' : 'pointer',
    opacity: disabled ? 0.5 : 1,
    transition: 'background-color 0.2s',
    border: 'none',
    padding: 0,
    outline: 'none',
  };

  // Thumb styles (the sliding circle)
  const thumbStyle = {
    position: 'absolute',
    top: 3,
    left: checked ? config.track.width - config.thumb - 3 : 3,
    width: config.thumb,
    height: config.thumb,
    borderRadius: '50%',
    backgroundColor: '#ffffff',
    boxShadow: 'rgba(0, 0, 0, 0.2) 0px 2px 4px',
    transition: 'left 0.2s',
  };

  return (
    <div>
      <div
        className={className}
        style={{
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'space-between',
          gap: 16,
          opacity: disabled ? 0.5 : 1,
        }}
      >
        {/* Label and description */}
        {(label || description) && (
          <div
            style={{ flex: 1, cursor: disabled ? 'default' : 'pointer' }}
            onClick={!disabled ? handleToggle : undefined}
          >
            {label && (
              <span style={{
                display: 'block',
                fontSize: 14,
                fontWeight: 500,
                color: hasError ? '#b91c1c' : '#111827',
              }}>
                {label}
              </span>
            )}
            {description && (
              <span
                id={`${toggleId}-description`}
                style={{
                  display: 'block',
                  marginTop: 4,
                  fontSize: 12,
                  color: hasError ? '#dc2626' : '#6b7280',
                }}
              >
                {description}
              </span>
            )}
          </div>
        )}

        {/* Toggle switch */}
        <button
          ref={ref}
          id={toggleId}
          type="button"
          role="switch"
          aria-checked={checked}
          aria-invalid={hasError}
          aria-describedby={
            error ? `${toggleId}-error` : description ? `${toggleId}-description` : undefined
          }
          disabled={disabled}
          onClick={handleToggle}
          onKeyDown={handleKeyDown}
          style={trackStyle}
          {...props}
        >
          <span className="prorank-toggle-slider__thumb" style={thumbStyle} />
        </button>
      </div>

      {/* Error message */}
      {error && (
        <p
          id={`${toggleId}-error`}
          style={{
            marginTop: 8,
            fontSize: 12,
            color: '#dc2626',
          }}
          role="alert"
        >
          {error}
        </p>
      )}
    </div>
  );
});

Toggle.displayName = 'Toggle';

export default Toggle;
