/**
 * Textarea Component
 *
 * A unified textarea component following Tailwind-first architecture.
 *
 * @example
 * <Textarea
 *   label="Description"
 *   value={description}
 *   onChange={(e) => setDescription(e.target.value)}
 *   rows={4}
 *   placeholder="Enter description..."
 * />
 */

import React from 'react';

const Textarea = ({
  label,
  value,
  onChange,
  placeholder = '',
  rows = 4,
  disabled = false,
  error = '',
  helperText = '',
  className = '',
  ...props
}) => {
  const textareaId = `textarea-${Math.random().toString(36).substr(2, 9)}`;

  return (
    <div className={`pr-w-full ${className}`}>
      {label && (
        <label
          htmlFor={textareaId}
          className="pr-block pr-text-sm pr-font-medium pr-text-gray-700 pr-mb-2"
        >
          {label}
        </label>
      )}
      <textarea
        id={textareaId}
        value={value}
        onChange={onChange}
        placeholder={placeholder}
        rows={rows}
        disabled={disabled}
        className={`
          pr-w-full pr-px-4 pr-py-2 pr-border pr-rounded-lg pr-text-sm
          pr-transition-colors pr-duration-200
          focus:pr-outline-none focus:pr-ring-2 focus:pr-ring-blue-500 focus:pr-border-transparent
          ${disabled ? 'pr-bg-gray-100 pr-cursor-not-allowed pr-text-gray-500' : 'pr-bg-white'}
          ${error ? 'pr-border-red-500' : 'pr-border-gray-300'}
        `}
        {...props}
      />
      {error && (
        <p className="pr-mt-1 pr-text-sm pr-text-red-600">{error}</p>
      )}
      {helperText && !error && (
        <p className="pr-mt-1 pr-text-sm pr-text-gray-500">{helperText}</p>
      )}
    </div>
  );
};

export default Textarea;
