/**
 * Select Component - ProRank SEO
 *
 * Premium select dropdown component with validation states.
 * Consistent styling with Input component.
 *
 * @example
 * <Select label="Post Type" options={postTypes} value={selected} onChange={handleChange} />
 * <Select label="Category" error="Please select a category">
 *   <option value="">Select...</option>
 *   <option value="1">Category 1</option>
 * </Select>
 */

import React from 'react';
import clsx from 'clsx';
import { ChevronDownIcon } from '@heroicons/react/24/outline';

const Select = React.forwardRef(({
  label,
  id,
  name,
  value,
  onChange,
  onBlur,
  error,
  helperText,
  disabled = false,
  required = false,
  className = '',
  selectClassName = '',
  fullWidth = true,
  size = 'md',
  children,
  options, // Alternative to children
  placeholder = 'Select an option',
  ...props
}, ref) => {
  const selectId = id || name || `select-${Math.random().toString(36).substr(2, 9)}`;
  const hasError = Boolean(error);

  // Container styles
  const containerStyles = clsx(
    'pr-space-y-2',
    {
      'pr-w-full': fullWidth,
      'pr-opacity-50': disabled,
    },
    className
  );

  // Size variants
  const sizeStyles = {
    sm: 'pr-px-3 pr-py-1.5 pr-text-sm pr-pr-10',
    md: 'pr-px-3 pr-py-2 pr-text-sm pr-pr-10',
    lg: 'pr-px-4 pr-py-2.5 pr-text-base pr-pr-12',
  };

  // Select styles
  const selectStyles = clsx(
    'pr-block',
    'pr-w-full',
    'pr-border',
    'pr-rounded-md',
    'pr-transition-colors',
    'pr-duration-150',
    'pr-appearance-none',
    'pr-bg-white',
    'focus:pr-outline-none',
    'focus:pr-ring-2',
    'focus:pr-border-transparent',
    'disabled:pr-bg-gray-50',
    'disabled:pr-text-gray-500',
    'disabled:pr-cursor-not-allowed',
    sizeStyles[size],
    {
      'pr-border-gray-300 focus:pr-ring-primary-500': !hasError,
      'pr-border-error-300 pr-text-error-900 focus:pr-ring-error-500 focus:pr-border-error-500': hasError,
    },
    selectClassName
  );

  // Label styles
  const labelStyles = clsx(
    'pr-block',
    'pr-text-sm',
    'pr-font-medium',
    {
      'pr-text-gray-700': !hasError,
      'pr-text-error-700': hasError,
    }
  );

  // Helper/Error text styles
  const helperStyles = clsx(
    'pr-text-xs',
    {
      'pr-text-gray-500': !hasError,
      'pr-text-error-600': hasError,
    }
  );

  return (
    <div className={containerStyles}>
      {/* Label */}
      {label && (
        <label htmlFor={selectId} className={labelStyles}>
          {label}
          {required && <span className="pr-text-error-500 pr-ml-1">*</span>}
        </label>
      )}

      {/* Select wrapper */}
      <div className="pr-relative">
        {/* Select field */}
        <select
          ref={ref}
          id={selectId}
          name={name}
          value={value}
          onChange={onChange}
          onBlur={onBlur}
          disabled={disabled}
          required={required}
          aria-invalid={hasError}
          aria-describedby={
            error ? `${selectId}-error` : helperText ? `${selectId}-helper` : undefined
          }
          className={selectStyles}
          {...props}
        >
          {/* Options from array */}
          {options ? (
            <>
              {placeholder && (
                <option value="" disabled>
                  {placeholder}
                </option>
              )}
              {options.map((option) => (
                <option
                  key={option.value}
                  value={option.value}
                  disabled={option.disabled}
                >
                  {option.label}
                </option>
              ))}
            </>
          ) : (
            children
          )}
        </select>

        {/* Chevron icon */}
        <div className="pr-pointer-events-none pr-absolute pr-inset-y-0 pr-right-0 pr-flex pr-items-center pr-pr-3">
          <ChevronDownIcon
            className={clsx('pr-w-5 pr-h-5', {
              'pr-text-gray-400': !hasError,
              'pr-text-error-500': hasError,
            })}
          />
        </div>
      </div>

      {/* Error message */}
      {error && (
        <p id={`${selectId}-error`} className={helperStyles} role="alert">
          {error}
        </p>
      )}

      {/* Helper text */}
      {helperText && !error && (
        <p id={`${selectId}-helper`} className={helperStyles}>
          {helperText}
        </p>
      )}
    </div>
  );
});

Select.displayName = 'Select';

export default Select;
