/**
 * PageHeader Component - ProRank SEO
 *
 * Premium page header component for admin pages.
 * Provides consistent header layout with title, description, and actions.
 *
 * @example
 * <PageHeader
 *   title="Dashboard"
 *   description="Overview of your SEO performance"
 *   actions={<Button variant="primary">Add New</Button>}
 * />
 *
 * <PageHeader
 *   title="Settings"
 *   breadcrumbs={[
 *     { label: 'Home', href: '/' },
 *     { label: 'Settings' }
 *   ]}
 * />
 */

import React from 'react';
import clsx from 'clsx';
import { ChevronRightIcon } from '@heroicons/react/24/outline';

const PageHeader = ({
  title,
  description,
  breadcrumbs,
  actions,
  stats,
  className = '',
  ...props
}) => {
  // Container styles
  const containerStyles = clsx(
    'pr-border-b',
    'pr-border-gray-200',
    'pr-bg-white',
    'pr-px-8',
    'pr-py-6',
    className
  );

  return (
    <div className={containerStyles} {...props}>
      {/* Breadcrumbs */}
      {breadcrumbs && breadcrumbs.length > 0 && (
        <nav className="pr-flex pr-items-center pr-gap-2 pr-mb-4" aria-label="Breadcrumb">
          <ol className="pr-flex pr-items-center pr-gap-2">
            {breadcrumbs.map((crumb, index) => (
              <li key={index} className="pr-flex pr-items-center pr-gap-2">
                {index > 0 && (
                  <ChevronRightIcon className="pr-w-4 pr-h-4 pr-text-gray-400" />
                )}
                {crumb.href ? (
                  <a
                    href={crumb.href}
                    className="pr-text-sm pr-text-gray-500 hover:pr-text-gray-700 pr-transition-colors"
                  >
                    {crumb.label}
                  </a>
                ) : (
                  <span className="pr-text-sm pr-text-gray-900 pr-font-medium">
                    {crumb.label}
                  </span>
                )}
              </li>
            ))}
          </ol>
        </nav>
      )}

      {/* Main header content */}
      <div className="pr-flex pr-items-center pr-justify-between pr-gap-4">
        {/* Left side - Title and description */}
        <div className="pr-flex-1 pr-min-w-0">
          <h1 className="pr-text-2xl pr-font-semibold pr-text-gray-900">
            {title}
          </h1>
          {description && (
            <p className="pr-mt-1 pr-text-sm pr-text-gray-500">
              {description}
            </p>
          )}
        </div>

        {/* Right side - Stats and actions */}
        <div className="pr-flex pr-items-center pr-gap-6">
          {/* Stats */}
          {stats && (
            <div className="pr-flex pr-items-center pr-gap-6">
              {stats}
            </div>
          )}

          {/* Actions */}
          {actions && (
            <div className="pr-flex pr-items-center pr-gap-3">
              {actions}
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

PageHeader.displayName = 'PageHeader';

export default PageHeader;
