/**
 * Input Component - ProRank SEO
 *
 * Premium input field component with validation states and helper text.
 * Supports text, email, password, number, and other input types.
 *
 * @example
 * <Input label="Site Title" placeholder="Enter title" />
 * <Input label="Email" type="email" error="Invalid email" />
 * <Input label="Password" type="password" helperText="Min 8 characters" />
 */

import React from 'react';
import clsx from 'clsx';

const Input = React.forwardRef(({
  label,
  id,
  name,
  type = 'text',
  placeholder,
  value,
  onChange,
  onBlur,
  error,
  helperText,
  disabled = false,
  required = false,
  className = '',
  inputClassName = '',
  leftIcon: LeftIcon,
  rightIcon: RightIcon,
  fullWidth = true,
  size = 'md',
  ...props
}, ref) => {
  const inputId = id || name || `input-${Math.random().toString(36).substr(2, 9)}`;
  const hasError = Boolean(error);

  // Container styles
  const containerStyles = clsx(
    'pr-space-y-2',
    {
      'pr-w-full': fullWidth,
      'pr-opacity-50': disabled,
    },
    className
  );

  // Size variants
  const sizeStyles = {
    sm: 'pr-px-3 pr-py-1.5 pr-text-sm',
    md: 'pr-px-3 pr-py-2 pr-text-sm',
    lg: 'pr-px-4 pr-py-2.5 pr-text-base',
  };

  // Input styles
  const inputStyles = clsx(
    'pr-block',
    'pr-w-full',
    'pr-border',
    'pr-rounded-md',
    'pr-transition-colors',
    'pr-duration-150',
    'placeholder:pr-text-gray-400',
    'focus:pr-outline-none',
    'focus:pr-ring-2',
    'focus:pr-border-transparent',
    'disabled:pr-bg-gray-50',
    'disabled:pr-text-gray-500',
    'disabled:pr-cursor-not-allowed',
    sizeStyles[size],
    {
      'pr-border-gray-300 focus:pr-ring-primary-500': !hasError,
      'pr-border-error-300 pr-text-error-900 focus:pr-ring-error-500 focus:pr-border-error-500': hasError,
      'pr-pl-10': LeftIcon,
      'pr-pr-10': RightIcon,
    },
    inputClassName
  );

  // Label styles
  const labelStyles = clsx(
    'pr-block',
    'pr-text-sm',
    'pr-font-medium',
    {
      'pr-text-gray-700': !hasError,
      'pr-text-error-700': hasError,
    }
  );

  // Helper/Error text styles
  const helperStyles = clsx(
    'pr-text-xs',
    {
      'pr-text-gray-500': !hasError,
      'pr-text-error-600': hasError,
    }
  );

  // Icon wrapper styles
  const iconWrapperStyles = clsx(
    'pr-absolute',
    'pr-top-1/2',
    'pr--translate-y-1/2',
    'pr-flex',
    'pr-items-center',
    'pr-pointer-events-none'
  );

  const iconStyles = clsx(
    'pr-w-5',
    'pr-h-5',
    {
      'pr-text-gray-400': !hasError,
      'pr-text-error-500': hasError,
    }
  );

  return (
    <div className={containerStyles}>
      {/* Label */}
      {label && (
        <label htmlFor={inputId} className={labelStyles}>
          {label}
          {required && <span className="pr-text-error-500 pr-ml-1">*</span>}
        </label>
      )}

      {/* Input wrapper */}
      <div className="pr-relative">
        {/* Left Icon */}
        {LeftIcon && (
          <div className={clsx(iconWrapperStyles, 'pr-left-3')}>
            <LeftIcon className={iconStyles} />
          </div>
        )}

        {/* Input field */}
        <input
          ref={ref}
          id={inputId}
          name={name}
          type={type}
          value={value}
          onChange={onChange}
          onBlur={onBlur}
          placeholder={placeholder}
          disabled={disabled}
          required={required}
          aria-invalid={hasError}
          aria-describedby={
            error ? `${inputId}-error` : helperText ? `${inputId}-helper` : undefined
          }
          className={inputStyles}
          {...props}
        />

        {/* Right Icon */}
        {RightIcon && (
          <div className={clsx(iconWrapperStyles, 'pr-right-3')}>
            <RightIcon className={iconStyles} />
          </div>
        )}
      </div>

      {/* Error message */}
      {error && (
        <p id={`${inputId}-error`} className={helperStyles} role="alert">
          {error}
        </p>
      )}

      {/* Helper text */}
      {helperText && !error && (
        <p id={`${inputId}-helper`} className={helperStyles}>
          {helperText}
        </p>
      )}
    </div>
  );
});

Input.displayName = 'Input';

export default Input;
