/**
 * Checkbox Component - ProRank SEO
 *
 * Premium checkbox component with label and description support.
 * Follows accessibility best practices.
 *
 * @example
 * <Checkbox label="Enable XML Sitemap" checked={enabled} onChange={setEnabled} />
 * <Checkbox
 *   label="Advanced Options"
 *   description="Enable advanced SEO features"
 *   checked={advanced}
 *   onChange={setAdvanced}
 * />
 */

import React from 'react';
import clsx from 'clsx';

const Checkbox = React.forwardRef(({
  label,
  description,
  id,
  name,
  checked = false,
  onChange,
  disabled = false,
  error,
  className = '',
  size = 'md',
  ...props
}, ref) => {
  const checkboxId = id || name || `checkbox-${Math.random().toString(36).substr(2, 9)}`;
  const hasError = Boolean(error);

  // Container styles
  const containerStyles = clsx(
    'pr-flex pr-items-start pr-gap-3',
    {
      'pr-opacity-50 pr-cursor-not-allowed': disabled,
      'pr-cursor-pointer': !disabled,
    },
    className
  );

  // Size variants for checkbox
  const sizeStyles = {
    sm: 'pr-w-4 pr-h-4',
    md: 'pr-w-4 pr-h-4',
    lg: 'pr-w-5 pr-h-5',
  };

  // Checkbox styles
  const checkboxStyles = clsx(
    'pr-mt-0.5',
    'pr-rounded',
    'pr-border',
    'pr-transition-colors',
    'pr-duration-150',
    'focus:pr-outline-none',
    'focus:pr-ring-2',
    'focus:pr-ring-offset-2',
    'disabled:pr-opacity-50',
    'disabled:pr-cursor-not-allowed',
    sizeStyles[size],
    {
      'pr-border-gray-300 pr-text-primary-500 focus:pr-ring-primary-500': !hasError,
      'pr-border-error-300 pr-text-error-500 focus:pr-ring-error-500': hasError,
    }
  );

  // Label styles
  const labelStyles = clsx(
    'pr-flex-1',
    {
      'pr-cursor-pointer': !disabled,
    }
  );

  const labelTextStyles = clsx(
    'pr-block pr-text-sm pr-font-medium',
    {
      'pr-text-gray-900': !hasError,
      'pr-text-error-700': hasError,
    }
  );

  const descriptionStyles = clsx(
    'pr-mt-1 pr-text-xs',
    {
      'pr-text-gray-500': !hasError,
      'pr-text-error-600': hasError,
    }
  );

  return (
    <div>
      <label htmlFor={checkboxId} className={containerStyles}>
        <input
          ref={ref}
          id={checkboxId}
          name={name}
          type="checkbox"
          checked={checked}
          onChange={onChange}
          disabled={disabled}
          aria-invalid={hasError}
          aria-describedby={
            error ? `${checkboxId}-error` : description ? `${checkboxId}-description` : undefined
          }
          className={checkboxStyles}
          {...props}
        />

        {(label || description) && (
          <div className={labelStyles}>
            {label && (
              <span className={labelTextStyles}>
                {label}
              </span>
            )}
            {description && (
              <span
                id={`${checkboxId}-description`}
                className={descriptionStyles}
              >
                {description}
              </span>
            )}
          </div>
        )}
      </label>

      {/* Error message */}
      {error && (
        <p
          id={`${checkboxId}-error`}
          className="pr-mt-2 pr-ml-7 pr-text-xs pr-text-error-600"
          role="alert"
        >
          {error}
        </p>
      )}
    </div>
  );
});

Checkbox.displayName = 'Checkbox';

export default Checkbox;
