/**
 * Card Component - ProRank SEO
 *
 * Premium card component for content grouping.
 * Supports variants with headers, footers, and hover effects.
 *
 * @example
 * <Card>
 *   <h3>Card Title</h3>
 *   <p>Card content</p>
 * </Card>
 *
 * <Card
 *   header={<h3>Title</h3>}
 *   footer={<Button>Action</Button>}
 * >
 *   Content
 * </Card>
 *
 * <Card
 *   variant="performance"
 *   header={<h3>Performance Settings</h3>}
 * >
 *   Settings content
 * </Card>
 */

import React from 'react';
import clsx from 'clsx';

const Card = ({
  children,
  header,
  footer,
  variant = 'default',
  padding = 'default',
  hoverable = false,
  className = '',
  headerClassName = '',
  footerClassName = '',
  bodyClassName = '',
  ...props
}) => {
  // Padding variants
  const paddingVariants = {
    none: '',
    sm: 'pr-p-4',
    default: 'pr-p-6',
    lg: 'pr-p-8',
  };

  // Variant styles
  const variantStyles = {
    default: 'pr-bg-white pr-border pr-border-gray-200',
    outlined: 'pr-bg-white pr-border-2 pr-border-gray-300',
    elevated: 'pr-bg-white pr-shadow-card',
    flat: 'pr-bg-gray-50',
    performance: 'prorank-card', // Uses custom CSS from admin.css
  };

  // Use performance variant with custom CSS classes
  const isPerformanceVariant = variant === 'performance';

  // Card container styles
  const cardStyles = clsx(
    {
      'pr-rounded-lg pr-transition-shadow pr-duration-200': !isPerformanceVariant,
      'hover:pr-shadow-card-hover pr-cursor-pointer': hoverable && !isPerformanceVariant,
    },
    variantStyles[variant],
    className
  );

  // Header styles
  const headerStyles = clsx(
    isPerformanceVariant ? 'prorank-card-header' : 'pr-px-6 pr-py-4 pr-border-b pr-border-gray-200 pr-bg-gray-50',
    headerClassName
  );

  // Body styles
  const bodyStyles = clsx(
    isPerformanceVariant ? 'prorank-card-body pr-p-6' : paddingVariants[padding],
    bodyClassName
  );

  // Footer styles
  const footerStyles = clsx(
    'pr-px-6 pr-py-4',
    'pr-border-t pr-border-gray-200',
    'pr-bg-gray-50',
    footerClassName
  );

  return (
    <div className={cardStyles} {...props}>
      {/* Header */}
      {header && (
        <div className={headerStyles}>
          {header}
        </div>
      )}

      {/* Body */}
      <div className={bodyStyles}>
        {children}
      </div>

      {/* Footer */}
      {footer && (
        <div className={footerStyles}>
          {footer}
        </div>
      )}
    </div>
  );
};

Card.displayName = 'Card';

/**
 * CardBody Component - Simple wrapper for card content
 * Used for backward compatibility with @wordpress/components style
 */
export const CardBody = ({ children, className = '', style = {} }) => (
  <div className={clsx('pr-p-4', className)} style={style}>
    {children}
  </div>
);

CardBody.displayName = 'CardBody';

/**
 * CardHeader Component - Simple wrapper for card header
 */
export const CardHeader = ({ children, className = '', style = {} }) => (
  <div className={clsx('pr-px-4 pr-py-3 pr-border-b pr-border-gray-200 pr-font-medium', className)} style={style}>
    {children}
  </div>
);

CardHeader.displayName = 'CardHeader';

export default Card;
