/**
 * Button Component - ProRank SEO
 *
 * Premium button component with multiple variants following Tailwind-first architecture.
 * Supports all standard button types with consistent styling across the plugin.
 *
 * @example
 * <Button variant="primary" onClick={handleSave}>Save Changes</Button>
 * <Button variant="secondary" size="sm">Cancel</Button>
 * <Button variant="danger" loading>Delete</Button>
 */

import React from 'react';
import clsx from 'clsx';

const Button = React.forwardRef(({
  children,
  variant = 'primary',
  size = 'md',
  loading = false,
  disabled = false,
  fullWidth = false,
  type = 'button',
  className = '',
  icon: Icon,
  iconPosition = 'left',
  ...props
}, ref) => {
  // Base styles - shared across all variants
  const baseStyles = clsx(
    'pr-inline-flex',
    'pr-items-center',
    'pr-justify-center',
    'pr-font-medium',
    'pr-rounded-md',
    'pr-transition-colors',
    'pr-duration-150',
    'focus:pr-outline-none',
    'focus:pr-ring-2',
    'focus:pr-ring-offset-2',
    'disabled:pr-opacity-50',
    'disabled:pr-cursor-not-allowed',
    {
      'pr-w-full': fullWidth,
      'pr-cursor-not-allowed pr-opacity-50': loading || disabled,
    }
  );

  // Size variants
  const sizeStyles = {
    xs: clsx('pr-px-2.5', 'pr-py-1.5', 'pr-text-xs', Icon && 'pr-gap-1.5'),
    sm: clsx('pr-px-3', 'pr-py-2', 'pr-text-sm', Icon && 'pr-gap-2'),
    md: clsx('pr-px-4', 'pr-py-2', 'pr-text-sm', Icon && 'pr-gap-2'),
    lg: clsx('pr-px-5', 'pr-py-2.5', 'pr-text-base', Icon && 'pr-gap-2.5'),
    xl: clsx('pr-px-6', 'pr-py-3', 'pr-text-base', Icon && 'pr-gap-3'),
  };

  // Variant styles
  const variantStyles = {
    primary: clsx(
      'pr-bg-primary-500',
      'pr-text-white',
      'pr-border pr-border-transparent',
      'hover:pr-bg-primary-600',
      'focus:pr-ring-primary-500',
      'active:pr-bg-primary-700'
    ),
    secondary: clsx(
      'pr-bg-white',
      'pr-text-gray-700',
      'pr-border pr-border-gray-300',
      'hover:pr-bg-gray-50',
      'focus:pr-ring-primary-500',
      'active:pr-bg-gray-100'
    ),
    success: clsx(
      'pr-bg-success-500',
      'pr-text-white',
      'pr-border pr-border-transparent',
      'hover:pr-bg-success-600',
      'focus:pr-ring-success-500',
      'active:pr-bg-success-700'
    ),
    danger: clsx(
      'pr-bg-error-500',
      'pr-text-white',
      'pr-border pr-border-transparent',
      'hover:pr-bg-error-600',
      'focus:pr-ring-error-500',
      'active:pr-bg-error-700'
    ),
    warning: clsx(
      'pr-bg-warning-500',
      'pr-text-white',
      'pr-border pr-border-transparent',
      'hover:pr-bg-warning-600',
      'focus:pr-ring-warning-500',
      'active:pr-bg-warning-700'
    ),
    ghost: clsx(
      'pr-bg-transparent',
      'pr-text-gray-700',
      'pr-border pr-border-transparent',
      'hover:pr-bg-gray-100',
      'focus:pr-ring-primary-500',
      'active:pr-bg-gray-200'
    ),
    link: clsx(
      'pr-bg-transparent',
      'pr-text-primary-500',
      'pr-border pr-border-transparent',
      'hover:pr-text-primary-600',
      'hover:pr-underline',
      'focus:pr-ring-primary-500',
      'active:pr-text-primary-700',
      'pr-px-0'
    ),
  };

  // Icon size mapping
  const iconSizes = {
    xs: 'pr-w-3.5 pr-h-3.5',
    sm: 'pr-w-4 pr-h-4',
    md: 'pr-w-5 pr-h-5',
    lg: 'pr-w-5 pr-h-5',
    xl: 'pr-w-6 pr-h-6',
  };

  // Loading spinner
  const LoadingSpinner = () => (
    <svg
      className={clsx('pr-animate-spin', iconSizes[size])}
      xmlns="http://www.w3.org/2000/svg"
      fill="none"
      viewBox="0 0 24 24"
    >
      <circle
        className="pr-opacity-25"
        cx="12"
        cy="12"
        r="10"
        stroke="currentColor"
        strokeWidth="4"
      />
      <path
        className="pr-opacity-75"
        fill="currentColor"
        d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
      />
    </svg>
  );

  return (
    <button
      ref={ref}
      type={type}
      disabled={disabled || loading}
      className={clsx(
        baseStyles,
        sizeStyles[size],
        variantStyles[variant],
        className
      )}
      {...props}
    >
      {/* Loading spinner or left icon */}
      {loading ? (
        <LoadingSpinner />
      ) : (
        Icon && iconPosition === 'left' && <Icon className={iconSizes[size]} />
      )}

      {/* Button text */}
      {children}

      {/* Right icon */}
      {!loading && Icon && iconPosition === 'right' && (
        <Icon className={iconSizes[size]} />
      )}
    </button>
  );
});

Button.displayName = 'Button';

export default Button;
