/**
 * Alert Component - ProRank SEO
 *
 * Premium alert component for displaying important messages.
 * Supports multiple variants with icons and dismiss functionality.
 *
 * @example
 * <Alert variant="success" title="Success">
 *   Your changes have been saved successfully
 * </Alert>
 *
 * <Alert variant="error" dismissible onDismiss={handleDismiss}>
 *   Something went wrong
 * </Alert>
 */

import React from 'react';
import clsx from 'clsx';
import {
  CheckCircleIcon,
  ExclamationTriangleIcon,
  XCircleIcon,
  InformationCircleIcon,
  XMarkIcon,
} from '@heroicons/react/24/outline';

const Alert = ({
  children,
  title,
  variant = 'info',
  icon: CustomIcon,
  dismissible = false,
  onDismiss,
  className = '',
  ...props
}) => {
  // Default icons for each variant
  const defaultIcons = {
    success: CheckCircleIcon,
    warning: ExclamationTriangleIcon,
    error: XCircleIcon,
    info: InformationCircleIcon,
  };

  // Ensure variant is valid
  const validVariant = ['success', 'warning', 'error', 'info'].includes(variant) ? variant : 'info';

  // Get icon with fallback - ensure it's a valid component
  let Icon = CustomIcon || defaultIcons[validVariant] || null;

  // Extra safety: if Icon is truthy but not a function/object, set to null
  if (Icon && typeof Icon !== 'function' && typeof Icon !== 'object') {
    Icon = null;
  }

  // Variant styles
  const variantStyles = {
    success: {
      container: 'pr-bg-success-50 pr-border-success-200',
      icon: 'pr-text-success-500',
      title: 'pr-text-success-900',
      text: 'pr-text-success-700',
      dismiss: 'pr-text-success-500 hover:pr-text-success-700',
    },
    warning: {
      container: 'pr-bg-warning-50 pr-border-warning-200',
      icon: 'pr-text-warning-500',
      title: 'pr-text-warning-900',
      text: 'pr-text-warning-700',
      dismiss: 'pr-text-warning-500 hover:pr-text-warning-700',
    },
    error: {
      container: 'pr-bg-error-50 pr-border-error-200',
      icon: 'pr-text-error-500',
      title: 'pr-text-error-900',
      text: 'pr-text-error-700',
      dismiss: 'pr-text-error-500 hover:pr-text-error-700',
    },
    info: {
      container: 'pr-bg-info-50 pr-border-info-200',
      icon: 'pr-text-info-500',
      title: 'pr-text-info-900',
      text: 'pr-text-info-700',
      dismiss: 'pr-text-info-500 hover:pr-text-info-700',
    },
  };

  const styles = variantStyles[validVariant];

  // Alert container styles
  const alertStyles = clsx(
    'pr-flex',
    'pr-items-start',
    'pr-gap-3',
    'pr-p-4',
    'pr-border',
    'pr-rounded-lg',
    styles.container,
    className
  );

  return (
    <div className={alertStyles} role="alert" {...props}>
      {/* Icon */}
      {Icon && typeof Icon === 'function' && (
        <Icon
          className={clsx(
            'pr-w-5',
            'pr-h-5',
            'pr-flex-shrink-0',
            'pr-mt-0.5',
            styles.icon
          )}
        />
      )}

      {/* Content */}
      <div className="pr-flex-1">
        {title && (
          <h4 className={clsx('pr-text-sm', 'pr-font-medium', 'pr-mb-1', styles.title)}>
            {title || ''}
          </h4>
        )}
        <div className={clsx('pr-text-sm', styles.text)}>
          {children || ''}
        </div>
      </div>

      {/* Dismiss button */}
      {dismissible && (
        <button
          type="button"
          onClick={onDismiss}
          className={clsx(
            'pr-flex-shrink-0',
            'pr-transition-colors',
            'focus:pr-outline-none',
            'focus:pr-ring-2',
            'focus:pr-ring-offset-2',
            'focus:pr-ring-primary-500',
            'pr-rounded-md',
            'pr-p-1',
            styles.dismiss
          )}
          aria-label="Dismiss alert"
        >
          <XMarkIcon className="pr-w-5 pr-h-5" />
        </button>
      )}
    </div>
  );
};

Alert.displayName = 'Alert';

export default Alert;
