/**
 * Flex Layout Components
 *
 * Simple flexbox layout components to replace WordPress experimental Flex components
 */

import * as React from 'react';

interface FlexProps {
  children: React.ReactNode;
  direction?: 'row' | 'column';
  align?: 'start' | 'center' | 'end' | 'stretch' | 'space-between';
  justify?: 'start' | 'center' | 'end' | 'space-between' | 'space-around' | 'space-evenly';
  gap?: number | string;
  wrap?: boolean;
  className?: string;
  style?: React.CSSProperties;
}

interface FlexItemProps {
  children: React.ReactNode;
  grow?: number;
  shrink?: number;
  basis?: string | number;
  className?: string;
  style?: React.CSSProperties;
}

const alignmentMap: Record<string, string> = {
  start: 'flex-start',
  center: 'center',
  end: 'flex-end',
  stretch: 'stretch',
  'space-between': 'space-between',
};

const justifyMap: Record<string, string> = {
  start: 'flex-start',
  center: 'center',
  end: 'flex-end',
  'space-between': 'space-between',
  'space-around': 'space-around',
  'space-evenly': 'space-evenly',
};

export const Flex: React.FC<FlexProps> = ({
  children,
  direction = 'row',
  align = 'stretch',
  justify = 'start',
  gap = 0,
  wrap = false,
  className = '',
  style = {},
}) => {
  const flexStyle: React.CSSProperties = {
    display: 'flex',
    flexDirection: direction,
    alignItems: alignmentMap[align] || align,
    justifyContent: justifyMap[justify] || justify,
    gap: typeof gap === 'number' ? `${gap * 4}px` : gap,
    flexWrap: wrap ? 'wrap' : 'nowrap',
    ...style,
  };

  return (
    <div className={`prorank-flex ${className}`} style={flexStyle}>
      {children}
    </div>
  );
};

export const FlexItem: React.FC<FlexItemProps> = ({
  children,
  grow = 0,
  shrink = 1,
  basis = 'auto',
  className = '',
  style = {},
}) => {
  const itemStyle: React.CSSProperties = {
    flexGrow: grow,
    flexShrink: shrink,
    flexBasis: basis,
    ...style,
  };

  return (
    <div className={`prorank-flex-item ${className}`} style={itemStyle}>
      {children}
    </div>
  );
};

export const FlexBlock: React.FC<FlexItemProps> = ({
  children,
  className = '',
  style = {},
  ...props
}) => {
  return (
    <FlexItem grow={1} className={className} style={style} {...props}>
      {children}
    </FlexItem>
  );
};

export default Flex;
