/**
 * Typography Components
 *
 * Stable replacements for WordPress experimental typography components.
 * These components follow ProRank SEO's design system and CSS Modules architecture.
 *
 * @package
 */

// Temporarily disable CSS module import to debug
// import styles from './Typography.module.css';
const styles = {
  divider: 'prorank-divider',
  heading: 'prorank-heading',
  'heading--size-small': 'prorank-heading--size-small',
  'heading--size-medium': 'prorank-heading--size-medium',
  'heading--size-large': 'prorank-heading--size-large',
  'heading--weight-normal': 'prorank-heading--weight-normal',
  'heading--weight-medium': 'prorank-heading--weight-medium',
  'heading--weight-semibold': 'prorank-heading--weight-semibold',
  'heading--weight-bold': 'prorank-heading--weight-bold',
  text: 'prorank-text',
  'text--body': 'prorank-text--body',
  'text--muted': 'prorank-text--muted',
  'text--caption': 'prorank-text--caption',
  'text--error': 'prorank-text--error',
  'text--success': 'prorank-text--success',
  'text--size-small': 'prorank-text--size-small',
  'text--size-medium': 'prorank-text--size-medium',
  'text--size-large': 'prorank-text--size-large',
  'text--weight-normal': 'prorank-text--weight-normal',
  'text--weight-medium': 'prorank-text--weight-medium',
  'text--weight-semibold': 'prorank-text--weight-semibold',
  'text--weight-bold': 'prorank-text--weight-bold',
};

/**
 * Divider component - horizontal rule with consistent styling
 *
 * @param {Object} props
 * @param {string} props.className - Additional CSS classes
 */
export const Divider = ({ className = '', ...props }) => (
  <hr className={`${styles.divider} ${className}`.trim()} {...props} />
);

/**
 * Heading component - semantic headings with design system styling
 *
 * @param {Object}                    props
 * @param {number}                    props.level     - Heading level (1-6), defaults to 2
 * @param {string}                    props.size      - Size variant: 'small', 'medium', 'large'
 * @param {string}                    props.weight    - Font weight: 'normal', 'medium', 'semibold', 'bold'
 * @param {string}                    props.className - Additional CSS classes
 * @param {import('react').ReactNode} props.children  - Content
 */
export const Heading = ({
  level = 2,
  size = 'medium',
  weight = 'semibold',
  className = '',
  children,
  ...props
}) => {
  // Ensure valid heading level
  const validLevel = Math.min(Math.max(1, parseInt(level)), 6);
  const Tag = `h${validLevel}`;

  const classes = [
    styles.heading,
    styles[`heading--size-${size}`],
    styles[`heading--weight-${weight}`],
    className,
  ]
    .filter(Boolean)
    .join(' ');

  return (
    <Tag className={classes} {...props}>
      {children}
    </Tag>
  );
};

/**
 * Text component - paragraph text with design system styling
 *
 * @param {Object}                    props
 * @param {string}                    props.variant   - Text variant: 'body', 'muted', 'caption', 'error', 'success'
 * @param {string}                    props.size      - Size variant: 'small', 'medium', 'large'
 * @param {string}                    props.weight    - Font weight: 'normal', 'medium', 'semibold', 'bold'
 * @param {boolean}                   props.as        - HTML element to render as (p, span, div)
 * @param {string}                    props.className - Additional CSS classes
 * @param {import('react').ReactNode} props.children  - Content
 */
export const Text = ({
  variant = 'body',
  size = 'medium',
  weight = 'normal',
  as = 'p',
  className = '',
  children,
  ...props
}) => {
  const Tag = as;

  const classes = [
    styles.text,
    styles[`text--${variant}`],
    styles[`text--size-${size}`],
    styles[`text--weight-${weight}`],
    className,
  ]
    .filter(Boolean)
    .join(' ');

  return (
    <Tag className={classes} {...props}>
      {children}
    </Tag>
  );
};
