/* global sprintf */
import { __, sprintf } from '@wordpress/i18n';
import { Button } from '@wordpress/components';

const Pagination = ({ currentPage, totalPages, totalItems, perPage, onPageChange }) => {
  if (totalPages <= 1) {
    return null;
  }

  const startItem = (currentPage - 1) * perPage + 1;
  const endItem = Math.min(currentPage * perPage, totalItems);

  const goToPage = (page) => {
    if (page >= 1 && page <= totalPages && page !== currentPage) {
      onPageChange(page);
    }
  };

  const renderPageNumbers = () => {
    const pages = [];
    const maxVisible = 7;
    const halfVisible = Math.floor(maxVisible / 2);

    let startPage = Math.max(1, currentPage - halfVisible);
    let endPage = Math.min(totalPages, currentPage + halfVisible);

    if (currentPage <= halfVisible) {
      endPage = Math.min(totalPages, maxVisible);
    } else if (currentPage >= totalPages - halfVisible) {
      startPage = Math.max(1, totalPages - maxVisible + 1);
    }

    if (startPage > 1) {
      pages.push(
        <Button
          key={1}
          variant="tertiary"
          onClick={() => goToPage(1)}
          className="prorank-page-number"
        >
          1
        </Button>
      );
      if (startPage > 2) {
        pages.push(
          <span key="dots1" className="prorank-page-dots">
            ...
          </span>
        );
      }
    }

    for (let i = startPage; i <= endPage; i++) {
      pages.push(
        <Button
          key={i}
          variant={i === currentPage ? 'primary' : 'tertiary'}
          onClick={() => goToPage(i)}
          className={`prorank-page-number ${i === currentPage ? 'is-current' : ''}`}
        >
          {i}
        </Button>
      );
    }

    if (endPage < totalPages) {
      if (endPage < totalPages - 1) {
        pages.push(
          <span key="dots2" className="prorank-page-dots">
            ...
          </span>
        );
      }
      pages.push(
        <Button
          key={totalPages}
          variant="tertiary"
          onClick={() => goToPage(totalPages)}
          className="prorank-page-number"
        >
          {totalPages}
        </Button>
      );
    }

    return pages;
  };

  return (
    <div className="prorank-pagination">
      <div className="prorank-pagination-info">
        {sprintf(
          /* translators: %1$d: first item number, %2$d: last item number, %3$d: total number of items */
          __('Showing %1$d-%2$d of %3$d', 'prorank-seo'),
          startItem,
          endItem,
          totalItems
        )}
      </div>

      <div className="prorank-pagination-controls">
        <Button
          variant="tertiary"
          onClick={() => goToPage(currentPage - 1)}
          disabled={currentPage === 1}
          className="prorank-page-prev"
          aria-label={__('Previous page', 'prorank-seo')}
        >
          {'←'}
        </Button>

        {renderPageNumbers()}

        <Button
          variant="tertiary"
          onClick={() => goToPage(currentPage + 1)}
          disabled={currentPage === totalPages}
          className="prorank-page-next"
          aria-label={__('Next page', 'prorank-seo')}
        >
          {'→'}
        </Button>
      </div>
    </div>
  );
};

export default Pagination;
