/**
 * Visual Sitemap Builder Component
 * 
 * Drag-and-drop interface for organizing sitemap hierarchy
 * Allows custom priority and changefreq settings per page
 * 
 * @package ProRank\SEO
 * @since 1.0.0
 */

import { useState, useEffect } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import {
  DndContext,
  DragOverlay,
  closestCenter,
  KeyboardSensor,
  PointerSensor,
  useSensor,
  useSensors,
} from '@dnd-kit/core';
import {
  SortableContext,
  verticalListSortingStrategy,
  arrayMove,
} from '@dnd-kit/sortable';
import { useSortable } from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';
import ProrankButton from '../ProrankButton';
import { SelectControl, RangeControl } from '@wordpress/components';

interface SitemapItem {
  id: string;
  title: string;
  url: string;
  type: 'post' | 'page' | 'category' | 'tag' | 'custom';
  priority: number;
  changefreq: string;
  lastmod: string;
  children?: SitemapItem[];
  parent?: string;
  excluded?: boolean;
}

interface VisualSitemapBuilderProps {
  onSave?: (items: SitemapItem[]) => void;
}

const SortableItem = ({ item, onUpdate, onExclude, depth = 0 }) => {
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({ id: item.id });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.5 : 1,
    marginLeft: `${depth * 20}px`,
  };

  const priorityColors = {
    1.0: '#10b981',
    0.8: '#3b82f6',
    0.6: '#fbbf24',
    0.4: '#fb923c',
    0.2: '#ef4444',
  };

  return (
    <div
      ref={setNodeRef}
      style={style}
      className={`prorank-sitemap-item ${item.excluded ? 'excluded' : ''}`}
    >
      <div className="prorank-item-handle" {...attributes} {...listeners}>
        <svg width="20" height="20" viewBox="0 0 20 20">
          <path
            d="M7 2a2 2 0 1 1-4 0 2 2 0 0 1 4 0zM7 10a2 2 0 1 1-4 0 2 2 0 0 1 4 0zM7 18a2 2 0 1 1-4 0 2 2 0 0 1 4 0zM17 2a2 2 0 1 1-4 0 2 2 0 0 1 4 0zM17 10a2 2 0 1 1-4 0 2 2 0 0 1 4 0zM17 18a2 2 0 1 1-4 0 2 2 0 0 1 4 0z"
            fill="currentColor"
          />
        </svg>
      </div>

      <div className="prorank-item-content">
        <div className="prorank-item-info">
          <span className={`prorank-item-type prorank-type-${item.type}`}>
            {item.type}
          </span>
          <strong className="prorank-item-title">{item.title}</strong>
          <span className="prorank-item-url">{item.url}</span>
        </div>

        <div className="prorank-item-controls">
          <div className="prorank-priority-control">
            <label>{__('Priority:', 'prorank-seo')}</label>
            <input
              type="range"
              min="0"
              max="1"
              step="0.1"
              value={item.priority}
              onChange={(e) => onUpdate(item.id, 'priority', parseFloat(e.target.value))}
              style={{
                background: `linear-gradient(to right, #ef4444 0%, #fbbf24 50%, #10b981 100%)`,
              }}
            />
            <span
              className="prorank-priority-value"
              style={{ color: priorityColors[item.priority] || '#6b7280' }}
            >
              {item.priority}
            </span>
          </div>

          <div className="prorank-changefreq-control">
            <label>{__('Change Freq:', 'prorank-seo')}</label>
            <select
              value={item.changefreq}
              onChange={(e) => onUpdate(item.id, 'changefreq', e.target.value)}
            >
              <option value="always">{__('Always', 'prorank-seo')}</option>
              <option value="hourly">{__('Hourly', 'prorank-seo')}</option>
              <option value="daily">{__('Daily', 'prorank-seo')}</option>
              <option value="weekly">{__('Weekly', 'prorank-seo')}</option>
              <option value="monthly">{__('Monthly', 'prorank-seo')}</option>
              <option value="yearly">{__('Yearly', 'prorank-seo')}</option>
              <option value="never">{__('Never', 'prorank-seo')}</option>
            </select>
          </div>

          <button
            className={`prorank-exclude-btn ${item.excluded ? 'excluded' : ''}`}
            onClick={() => onExclude(item.id)}
            title={item.excluded ? __('Include in sitemap', 'prorank-seo') : __('Exclude from sitemap', 'prorank-seo')}
          >
            {item.excluded ? '➕' : '❌'}
          </button>
        </div>
      </div>

      {item.children && item.children.length > 0 && (
        <div className="prorank-item-children">
          {item.children.map((child) => (
            <SortableItem
              key={child.id}
              item={child}
              onUpdate={onUpdate}
              onExclude={onExclude}
              depth={depth + 1}
            />
          ))}
        </div>
      )}
    </div>
  );
};

const VisualSitemapBuilder: React.FC<VisualSitemapBuilderProps> = ({ onSave }) => {
  const [items, setItems] = useState<SitemapItem[]>([]);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [activeId, setActiveId] = useState(null);
  const [filter, setFilter] = useState('all');
  const [searchTerm, setSearchTerm] = useState('');

  const sensors = useSensors(
    useSensor(PointerSensor),
    useSensor(KeyboardSensor)
  );

  useEffect(() => {
    loadSitemapData();
  }, []);

  const loadSitemapData = async () => {
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/sitemap/items'
      });
      
      if (response && response.items) {
        setItems(response.items);
      } else {
        // Generate sample data for demo
        setItems(generateSampleData());
      }
    } catch (error) {
      // Use sample data as fallback
      setItems(generateSampleData());
    } finally {
      setLoading(false);
    }
  };

  const generateSampleData = (): SitemapItem[] => {
    return [
      {
        id: 'home',
        title: 'Home',
        url: '/',
        type: 'page',
        priority: 1.0,
        changefreq: 'daily',
        lastmod: new Date().toISOString(),
      },
      {
        id: 'about',
        title: 'About Us',
        url: '/about/',
        type: 'page',
        priority: 0.8,
        changefreq: 'monthly',
        lastmod: new Date().toISOString(),
      },
      {
        id: 'blog',
        title: 'Blog',
        url: '/blog/',
        type: 'page',
        priority: 0.9,
        changefreq: 'daily',
        lastmod: new Date().toISOString(),
        children: [
          {
            id: 'post-1',
            title: 'Latest News',
            url: '/blog/latest-news/',
            type: 'post',
            priority: 0.7,
            changefreq: 'weekly',
            lastmod: new Date().toISOString(),
          },
        ],
      },
      {
        id: 'products',
        title: 'Products',
        url: '/products/',
        type: 'page',
        priority: 0.9,
        changefreq: 'weekly',
        lastmod: new Date().toISOString(),
      },
    ];
  };

  const handleDragEnd = (event) => {
    const { active, over } = event;

    if (active.id !== over.id) {
      setItems((items) => {
        const oldIndex = items.findIndex((item) => item.id === active.id);
        const newIndex = items.findIndex((item) => item.id === over.id);
        return arrayMove(items, oldIndex, newIndex);
      });
    }

    setActiveId(null);
  };

  const handleDragStart = (event) => {
    setActiveId(event.active.id);
  };

  const updateItem = (id: string, field: string, value: any) => {
    setItems((prevItems) =>
      prevItems.map((item) =>
        item.id === id ? { ...item, [field]: value } : item
      )
    );
  };

  const toggleExclude = (id: string) => {
    setItems((prevItems) =>
      prevItems.map((item) =>
        item.id === id ? { ...item, excluded: !item.excluded } : item
      )
    );
  };

  const applyBulkSettings = () => {
    const priority = prompt(__('Enter priority (0.0 - 1.0):', 'prorank-seo'), '0.5');
    if (priority) {
      setItems((prevItems) =>
        prevItems.map((item) => ({
          ...item,
          priority: parseFloat(priority),
        }))
      );
    }
  };

  const saveChanges = async () => {
    setSaving(true);
    try {
      await apiFetch({
        path: '/prorank-seo/v1/sitemap/items',
        method: 'POST',
        data: { items }
      });
      
      if (onSave) {
        onSave(items);
      }
    } catch (error) {
      // Handle error
    } finally {
      setSaving(false);
    }
  };

  const filteredItems = items.filter((item) => {
    if (filter !== 'all' && item.type !== filter) return false;
    if (searchTerm && !item.title.toLowerCase().includes(searchTerm.toLowerCase())) return false;
    return true;
  });

  if (loading) {
    return <div className="prorank-loading">{__('Loading sitemap data...', 'prorank-seo')}</div>;
  }

  return (
    <div className="prorank-visual-sitemap-builder">
      <div className="prorank-builder-header">
        <h3>{__('Visual Sitemap Builder', 'prorank-seo')}</h3>
        <p className="description">
          {__('Drag and drop to reorganize your sitemap. Set custom priorities and change frequencies for better SEO control.', 'prorank-seo')}
        </p>
      </div>

      <div className="prorank-builder-toolbar">
        <div className="prorank-toolbar-left">
          <input
            type="text"
            placeholder={__('Search pages...', 'prorank-seo')}
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="prorank-search-input"
          />
          <select
            value={filter}
            onChange={(e) => setFilter(e.target.value)}
            className="prorank-filter-select"
          >
            <option value="all">{__('All Types', 'prorank-seo')}</option>
            <option value="page">{__('Pages', 'prorank-seo')}</option>
            <option value="post">{__('Posts', 'prorank-seo')}</option>
            <option value="category">{__('Categories', 'prorank-seo')}</option>
            <option value="tag">{__('Tags', 'prorank-seo')}</option>
          </select>
        </div>
        <div className="prorank-toolbar-right">
          <ProrankButton
            variant="secondary"
            size="small"
            onClick={applyBulkSettings}
          >
            {__('Bulk Settings', 'prorank-seo')}
          </ProrankButton>
        </div>
      </div>

      <div className="prorank-builder-content">
        <DndContext
          sensors={sensors}
          collisionDetection={closestCenter}
          onDragStart={handleDragStart}
          onDragEnd={handleDragEnd}
        >
          <SortableContext
            items={filteredItems.map((item) => item.id)}
            strategy={verticalListSortingStrategy}
          >
            {filteredItems.map((item) => (
              <SortableItem
                key={item.id}
                item={item}
                onUpdate={updateItem}
                onExclude={toggleExclude}
              />
            ))}
          </SortableContext>
          <DragOverlay>
            {activeId ? (
              <div className="prorank-drag-overlay">
                {items.find((item) => item.id === activeId)?.title}
              </div>
            ) : null}
          </DragOverlay>
        </DndContext>
      </div>

      <div className="prorank-builder-footer">
        <div className="prorank-stats">
          <span>{__('Total items:', 'prorank-seo')} {items.length}</span>
          <span>{__('Excluded:', 'prorank-seo')} {items.filter(i => i.excluded).length}</span>
        </div>
        <ProrankButton
          variant="primary"
          onClick={saveChanges}
          disabled={saving}
        >
          {saving ? __('Saving...', 'prorank-seo') : __('Save Changes', 'prorank-seo')}
        </ProrankButton>
      </div>

    </div>
  );
};

export default VisualSitemapBuilder;