/**
 * Toast Notification Component
 *
 * Displays toast-style notifications in the ProRank SEO admin interface.
 * Supports success, error, warning, and info notification types.
 *
 * @since 0.1.0
 */

import React, { useEffect, useState, useCallback } from 'react';
import { Icon } from '@wordpress/components';
import { check, warning, info, closeSmall } from '@wordpress/icons';
import { motion, AnimatePresence } from 'framer-motion';

const toastConfig = {
  success: {
    icon: check,
    bgClass: 'bg-green-50 border-green-200',
    textClass: 'text-green-800',
    focusRingClass: 'focus:ring-green-500',
  },
  error: {
    icon: closeSmall,
    bgClass: 'bg-red-50 border-red-200',
    textClass: 'text-red-800',
    focusRingClass: 'focus:ring-red-500',
  },
  warning: {
    icon: warning,
    bgClass: 'bg-yellow-50 border-yellow-200',
    textClass: 'text-yellow-800',
    focusRingClass: 'focus:ring-yellow-500',
  },
  info: {
    icon: info,
    bgClass: 'bg-blue-50 border-blue-200',
    textClass: 'text-blue-800',
    focusRingClass: 'focus:ring-blue-500',
  },
};

const positionClasses = {
  'top-right': 'top-4 right-4',
  'top-left': 'top-4 left-4',
  'bottom-right': 'bottom-4 right-4',
  'bottom-left': 'bottom-4 left-4',
  'top-center': 'top-4 left-1/2 transform -translate-x-1/2',
  'bottom-center': 'bottom-4 left-1/2 transform -translate-x-1/2',
};

const ToastNotification = ({
  id,
  message,
  type = 'info',
  duration = 5000,
  dismissible = true,
  onDismiss,
  position = 'bottom-right',
  actions = [],
}) => {
  const [isVisible, setIsVisible] = useState(true);

  const handleDismiss = useCallback(() => {
    setIsVisible(false);
    if (typeof onDismiss === 'function') {
      setTimeout(() => onDismiss(id), 300); // Wait for animation
    }
  }, [id, onDismiss]);

  useEffect(() => {
    if (duration > 0) {
      const timer = setTimeout(handleDismiss, duration);
      return () => clearTimeout(timer);
    }
  }, [duration, handleDismiss]);

  const { icon, bgClass, textClass, focusRingClass } = toastConfig[type] || toastConfig.info;
  const currentPositionClass = positionClasses[position] || positionClasses['bottom-right'];

  return (
    <AnimatePresence>
      {isVisible === true && (
        <motion.div
          initial={{ opacity: 0, y: 50, scale: 0.3 }}
          animate={{ opacity: 1, y: 0, scale: 1 }}
          exit={{ opacity: 0, scale: 0.5, transition: { duration: 0.2 } }}
          className={`prorank-toast prorank-toast--${type} fixed z-50 p-4 rounded-lg shadow-lg border ${bgClass} ${currentPositionClass} max-w-md`}
          role="alert"
          aria-live="assertive"
          aria-atomic="true"
        >
          <div className="flex items-start">
            <div className="flex-shrink-0">
              <Icon icon={icon} size={20} className={textClass} />
            </div>
            <div className="ml-3 flex-1">
              <p className={`text-sm font-medium ${textClass}`}>{message || ''}</p>
              {actions.length > 0 && (
                <div className="mt-3 flex space-x-2">
                  {actions.map((action, index) => (
                    <button
                      key={index}
                      onClick={action.onClick}
                      className={`text-sm font-medium rounded-md px-2 py-1 ${
                        action.variant === 'primary'
                          ? `${textClass} bg-white hover:bg-gray-50`
                          : 'text-gray-700 hover:text-gray-500'
                      }`}
                    >
                      {action.label}
                    </button>
                  ))}
                </div>
              )}
            </div>
            {dismissible === true && (
              <div className="ml-4 flex-shrink-0 flex">
                <button
                  className={`inline-flex rounded-md hover:bg-white hover:bg-opacity-20 focus:outline-none focus:ring-2 focus:ring-offset-2 ${focusRingClass}`}
                  onClick={handleDismiss}
                  aria-label="Dismiss notification"
                >
                  <Icon icon={closeSmall} size={20} className={textClass} />
                </button>
              </div>
            )}
          </div>
        </motion.div>
      )}
    </AnimatePresence>
  );
};

export default ToastNotification;
