/**
 * Unified Stats Card Component
 *
 * Enhanced with tooltips, clickable cards, and accessibility features.
 * Replaces all old stat card designs with consistent Tailwind version.
 */

import { __ } from '@wordpress/i18n';

const StatsCard = ({
  icon: Icon,
  value,
  label,
  trend,
  variant = 'default',
  tooltip,
  href,
  onClick,
  'aria-label': ariaLabel,
}) => {
  const variantClasses = {
    default: 'pr-border-gray-200',
    success: 'pr-border-green-200 pr-bg-green-50',
    warning: 'pr-border-yellow-200 pr-bg-yellow-50',
    danger: 'pr-border-red-200 pr-bg-red-50',
    info: 'pr-border-blue-200 pr-bg-blue-50',
  };

  const iconClasses = {
    default: 'pr-text-gray-600',
    success: 'pr-text-green-600',
    warning: 'pr-text-yellow-600',
    danger: 'pr-text-red-600',
    info: 'pr-text-blue-600',
  };

  // Determine if card is interactive
  const isClickable = href || onClick;

  // Render trend indicator
  const renderTrend = () => {
    if (trend === undefined || trend === null) return null;

    const isPositive = trend >= 0;
    // For issues, negative trend is good (fewer issues)
    const trendColor = isPositive ? 'pr-text-green-600' : 'pr-text-red-600';

    return (
      <div className={`pr-text-xs pr-mt-2 pr-flex pr-items-center pr-gap-1 ${trendColor}`}>
        <span aria-hidden="true">{isPositive ? '↑' : '↓'}</span>
        <span className="pr-font-semibold">{Math.abs(trend)}%</span>
        <span className="pr-text-gray-500">
          {isPositive ? __('increase', 'prorank-seo') : __('decrease', 'prorank-seo')}
        </span>
      </div>
    );
  };

  // Card content (shared between link and non-link versions)
  const cardContent = (
    <>
      <div className="pr-flex pr-items-start pr-gap-4">
        {Icon && (
          <div
            className={`pr-flex-shrink-0 pr-w-12 pr-h-12 pr-flex pr-items-center pr-justify-center pr-rounded-lg pr-bg-gray-50 ${iconClasses[variant]}`}
          >
            <Icon className="pr-w-6 pr-h-6" aria-hidden="true" />
          </div>
        )}
        <div className="pr-flex-1 pr-min-w-0">
          <div className="pr-flex pr-items-center pr-gap-2">
            <div className="pr-text-4xl pr-font-bold pr-text-gray-900 pr-leading-tight">
              {value !== undefined && value !== null ? value : '—'}
            </div>
          </div>
          <div className="pr-text-sm pr-text-gray-600 pr-font-medium">{label || ''}</div>
          {renderTrend()}
        </div>
      </div>
      {/* Click indicator for interactive cards */}
      {isClickable && (
        <div className="pr-absolute pr-bottom-3 pr-right-3 pr-text-xs pr-text-gray-400 pr-opacity-0 group-hover:pr-opacity-100 pr-transition-opacity">
          {__('Click for details →', 'prorank-seo')}
        </div>
      )}
    </>
  );

  // Base classes for the card
  const baseClasses = `pr-relative pr-bg-white pr-rounded-lg pr-shadow-sm pr-border pr-p-6 pr-transition-all ${variantClasses[variant]}`;
  const interactiveClasses = isClickable
    ? 'pr-cursor-pointer hover:pr-shadow-md hover:pr-scale-[1.02] focus:pr-outline-none focus:pr-ring-2 focus:pr-ring-primary-500 focus:pr-ring-offset-2 pr-group'
    : 'hover:pr-shadow-md';

  // Render as link if href provided
  if (href) {
    return (
      <a
        href={href}
        className={`${baseClasses} ${interactiveClasses} pr-block pr-no-underline`}
        aria-label={ariaLabel}
      >
        {cardContent}
      </a>
    );
  }

  // Render as button if onClick provided
  if (onClick) {
    return (
      <button
        type="button"
        onClick={onClick}
        className={`${baseClasses} ${interactiveClasses} pr-w-full pr-text-left`}
        aria-label={ariaLabel}
      >
        {cardContent}
      </button>
    );
  }

  // Render as static div
  return (
    <div className={`${baseClasses} ${interactiveClasses}`} aria-label={ariaLabel}>
      {cardContent}
    </div>
  );
};

export default StatsCard;
