/**
 * Settings Lint Message Component
 *
 * Displays validation messages for settings fields.
 *
 * @package
 * @since   0.1.0
 */

import { Notice, Icon } from '@wordpress/components';
import { warning, info, closeSmall } from '@wordpress/icons';

/**
 * Get icon for message type
 *
 * @param {string} type Message type
 * @return {Object} Icon component
 */
const getMessageIcon = (type) => {
  switch (type) {
    case 'error':
      return closeSmall;
    case 'warning':
      return warning;
    case 'info':
    default:
      return info;
  }
};

/**
 * Settings Lint Message component
 *
 * @param {Object } props          Component props
 * @param {Array  } props.messages Validation messages
 * @param {boolean} props.inline   Whether to display inline
 * @return {JSX.Element|null} Lint message or null
 */
const SettingsLintMessage = ({ messages = [], inline = true }) => {
  if (!messages || messages.length === 0) {
    return null;
  }

  // Group messages by type
  const groupedMessages = messages.reduce((acc, msg) => {
    if (!acc[msg.type]) {
      acc[msg.type] = [];
    }
    acc[msg.type].push(msg);
    return acc;
  }, {});

  // Priority order for display
  const typeOrder = ['error', 'warning', 'info'];

  if (inline === true) {
    // Inline display (next to field)
    return (
      <div className="prorank-settings-lint-inline" role="alert">
        {typeOrder.map((type) => {
          if (groupedMessages[type] === undefined) {
            return null;
          }

          return groupedMessages[type].map((msg, index) => (
            <div
              key={`${type}-${index}`}
              className={`prorank-settings-lint-message prorank-settings-lint-message--${type}`}
            >
              <Icon icon={getMessageIcon(type)} size={16} className="prorank-settings-lint-icon" />
              <span className="prorank-settings-lint-text">{msg.message}</span>
            </div>
          ));
        })}
      </div>
    );
  }

  // Block display (as notices)
  return (
    <div className="prorank-settings-lint-block">
      {typeOrder.map((type) => {
        if (groupedMessages[type] === undefined) {
          return null;
        }

        return groupedMessages[type].map((msg, index) => (
          <Notice
            key={`${type}-${index}`}
            status={type === 'error' ? 'error' : type}
            isDismissible={false}
            className="prorank-settings-lint-notice"
          >
            {msg.message}
          </Notice>
        ));
      })}
    </div>
  );
};

export default SettingsLintMessage;
