import { useState, useEffect } from '@wordpress/element';
import * as React from 'react';

/**
 * ProRank Toggle Slider Component
 * Settings toggle without text display
 * Used for individual feature toggles within settings
 */

interface ProrankToggleSliderProps {
  checked?: boolean;
  onChange?: (checked: boolean) => void;
  disabled?: boolean;
  className?: string;
  size?: 'small' | 'medium';
}

const ProrankToggleSlider: React.FC<ProrankToggleSliderProps> = ({
  checked = false,
  onChange,
  disabled = false,
  className = '',
  size = 'medium'
}) => {
  const [isChecked, setIsChecked] = useState(checked);

  useEffect(() => {
    setIsChecked(checked);
  }, [checked]);

  const handleToggle = () => {
    if (!disabled) {
      const newValue = !isChecked;
      setIsChecked(newValue);
      onChange?.(newValue);
    }
  };

  const sizeConfig = size === 'small'
    ? { width: 40, height: 22, padding: 2, radius: 11, thumb: 18 }
    : { width: 48, height: 26, padding: 3, radius: 13, thumb: 20 };

  const thumbLeft = isChecked
    ? sizeConfig.width - sizeConfig.thumb - sizeConfig.padding
    : sizeConfig.padding;

  const buttonStyle: React.CSSProperties = {
    position: 'relative',
    display: 'inline-flex',
    alignItems: 'center',
    width: `${sizeConfig.width}px`,
    height: `${sizeConfig.height}px`,
    padding: 0,
    border: 'none',
    borderRadius: `${sizeConfig.radius}px`,
    backgroundColor: isChecked ? '#ff6b35' : '#d1d5db',
    cursor: disabled ? 'not-allowed' : 'pointer',
    transition: 'background-color 0.2s ease',
    opacity: disabled ? 0.5 : 1,
    flexShrink: 0,
  };

  const thumbStyle: React.CSSProperties = {
    position: 'absolute',
    top: `${sizeConfig.padding}px`,
    left: `${thumbLeft}px`,
    width: `${sizeConfig.thumb}px`,
    height: `${sizeConfig.thumb}px`,
    borderRadius: '50%',
    backgroundColor: '#fff',
    boxShadow: '0 2px 4px rgba(0, 0, 0, 0.2)',
    transition: 'left 0.2s ease',
  };

  return (
    <button
      type="button"
      role="switch"
      aria-checked={isChecked}
      className={`prorank-toggle-slider prorank-toggle-slider--${size} ${isChecked ? 'prorank-toggle-slider--checked' : ''} ${disabled ? 'prorank-toggle-slider--disabled' : ''} ${className}`}
      onClick={handleToggle}
      disabled={disabled}
      style={buttonStyle}
    >
      <span className="prorank-toggle-slider__thumb" style={thumbStyle} />
    </button>
  );
};

export default ProrankToggleSlider;
