import React from 'react';

/**
 * ProRank Number Input Component
 * Styled number input matching ProRank design standards
 */

interface ProrankNumberInputProps {
  value: number;
  onChange: (value: number) => void;
  min?: number;
  max?: number;
  step?: number;
  label?: string;
  help?: string;
  className?: string;
  disabled?: boolean;
  inline?: boolean;
  width?: 'small' | 'medium' | 'large' | 'full';
  suffix?: string;
}

const ProrankNumberInput: React.FC<ProrankNumberInputProps> = ({
  value,
  onChange,
  min,
  max,
  step = 1,
  label,
  help,
  className = '',
  disabled = false,
  inline = false,
  width = 'small',
  suffix
}) => {
  const widthClasses = {
    small: 'prorank-input-small',
    medium: 'prorank-input-medium',
    large: 'prorank-input-large',
    full: 'prorank-input-full'
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const newValue = parseInt(e.target.value, 10);
    if (!isNaN(newValue)) {
      onChange(newValue);
    }
  };

  if (inline && label) {
    return (
      <div className={`prorank-input-inline ${className}`}>
        <label className="prorank-input-label">{label}</label>
        <div className="prorank-input-wrapper">
          <input
            type="number"
            className={`prorank-input prorank-input-number ${widthClasses[width]}`}
            value={value}
            onChange={handleChange}
            min={min}
            max={max}
            step={step}
            disabled={disabled}
          />
          {suffix && <span className="prorank-input-suffix">{suffix}</span>}
        </div>
        {help && <span className="prorank-input-help">{help}</span>}
      </div>
    );
  }

  return (
    <div className={`prorank-form-group ${className}`}>
      {label && <label className="prorank-form-label">{label}</label>}
      {help && <p className="prorank-form-help">{help}</p>}
      <div className="prorank-input-wrapper">
        <input
          type="number"
          className={`prorank-input prorank-input-number ${widthClasses[width]}`}
          value={value}
          onChange={handleChange}
          min={min}
          max={max}
          step={step}
          disabled={disabled}
        />
        {suffix && <span className="prorank-input-suffix">{suffix}</span>}
      </div>
    </div>
  );
};

export default ProrankNumberInput;