import React from 'react';

/**
 * PageHeader Component
 *
 * Unified header for all ProRank SEO admin pages
 * Provides the clean, neutral header style
 * Supports premium toggles positioned under subtitle
 *
 * @param {string}    title    - Page title
 * @param {string}    subtitle - Page subtitle/description
 * @param {ReactNode} actions  - Action buttons (optional)
 * @param {ReactNode} toggles  - Toggle controls (optional, positioned under subtitle)
 * @param {Array}     badges   - Badge chips displayed under subtitle (optional)
 */
const PageHeader = ({ title, subtitle, actions, toggles, badges }) => {
  const normalizedBadges = Array.isArray(badges)
    ? badges.map((badge) => {
      if (typeof badge === 'string') {
        return { label: badge };
      }
      return badge || null;
    }).filter(Boolean)
    : [];

  return (
    <div className="prorank-header">
      <div className="prorank-header-accent-bar" />
      <div className="prorank-header-wave" />
      <div className="prorank-header-content">
        <div className="prorank-header-left">
          <h1>{title || ''}</h1>
          {subtitle && <p>{subtitle}</p>}
          {normalizedBadges.length > 0 && (
            <div className="prorank-header-badges">
              {normalizedBadges.map((badge, index) => (
                <span
                  key={`${badge.label}-${index}`}
                  className={`prorank-header-badge${badge.variant ? ` prorank-header-badge--${badge.variant}` : ''}`}
                >
                  {badge.label}
                </span>
              ))}
            </div>
          )}
        </div>
        {actions && (
          <div className="prorank-header-right">
            <div className="prorank-header-actions">{actions}</div>
          </div>
        )}
      </div>

      {toggles && (
        <div className="prorank-header-toggles">
          <div className="prorank-header-toggles-container">
            {toggles}
          </div>
        </div>
      )}
    </div>
  );
};

export default PageHeader;
