/**
 * ModuleNav Component
 *
 * Unified Tailwind navigation for module sections.
 * Compact tabs with optional badges, counts, and toggles.
 */

import React from 'react';
import ProrankToggleSlider from './ProrankToggleSlider';

const ModuleNav = ({ modules, activeModule, onModuleChange, onToggleChange }) => {
  const canToggle = typeof onToggleChange === 'function';

  return (
    <div className="prorank-tab-navigation">
      <ul className="prorank-tab-list" role="tablist" aria-label="Module navigation">
        {modules.map((module) => {
          const isActive = activeModule === module.id;
          const isDisabled = Boolean(module.disabled);
          const showToggle = Boolean(module.hasToggle);

          return (
            <li key={module.id} className="prorank-tab-item">
              <button
                type="button"
                onClick={() => onModuleChange(module.id)}
                className={`prorank-tab-button ${isActive ? 'active' : ''} ${isDisabled ? 'disabled' : ''}`}
                aria-current={isActive ? 'page' : undefined}
                disabled={isDisabled}
              >
                <span className="prorank-tab-label">
                  <span className="prorank-tab-name">{module.label || ''}</span>
                  {module.count !== undefined && module.count !== null && (
                    <span className="pr-ml-2 pr-inline-flex pr-items-center pr-justify-center pr-min-w-[18px] pr-h-4 pr-px-1.5 pr-rounded-full pr-bg-gray-100 pr-text-[10px] pr-font-semibold pr-text-gray-600">
                      {typeof module.count === 'number' || typeof module.count === 'string'
                        ? String(module.count)
                        : ''}
                    </span>
                  )}
                </span>
              </button>

              {showToggle && (
                <div
                  className="prorank-tab-toggle"
                  onClick={(event) => event.stopPropagation()}
                >
                  <ProrankToggleSlider
                    size="small"
                    checked={Boolean(module.toggleValue)}
                    onChange={(checked) => {
                      if (canToggle) {
                        onToggleChange(module.id, checked);
                      }
                    }}
                    disabled={Boolean(module.toggleDisabled || module.disabled || !canToggle)}
                  />
                </div>
              )}
            </li>
          );
        })}
      </ul>
    </div>
  );
};

export default ModuleNav;
