/**
 * IndexNow Integration Component
 * 
 * Provides instant indexing through IndexNow protocol
 * Supports Bing, Yandex, Seznam and other search engines
 * 
 * @package ProRank\SEO
 * @since 1.0.0
 */

import { useState, useEffect } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { Notice, Spinner, TextControl, ExternalLink } from '@wordpress/components';
import ProrankButton from '../ProrankButton';
import ProrankToggle from '../ProrankToggle';

interface IndexNowSettings {
  enabled: boolean;
  api_key: string;
  auto_submit: boolean;
  submit_on_publish: boolean;
  submit_on_update: boolean;
  engines: string[];
}

const IndexNowIntegration = () => {
  const [settings, setSettings] = useState<IndexNowSettings>({
    enabled: false,
    api_key: '',
    auto_submit: true,
    submit_on_publish: true,
    submit_on_update: true,
    engines: ['bing', 'yandex', 'seznam', 'yep']
  });
  
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [testing, setTesting] = useState(false);
  const [generateKeyLoading, setGenerateKeyLoading] = useState(false);
  const [submissionHistory, setSubmissionHistory] = useState<any[]>([]);
  const [testResult, setTestResult] = useState<{ success: boolean; message: string } | null>(null);

  const searchEngines = [
    { id: 'bing', name: 'Bing', url: 'https://www.bing.com/indexnow' },
    { id: 'yandex', name: 'Yandex', url: 'https://yandex.com/indexnow' },
    { id: 'seznam', name: 'Seznam', url: 'https://search.seznam.cz/indexnow' },
    { id: 'yep', name: 'Yep', url: 'https://indexnow.yep.com' },
    { id: 'naver', name: 'Naver', url: 'https://searchadvisor.naver.com/indexnow' }
  ];

  useEffect(() => {
    loadSettings();
    loadSubmissionHistory();
  }, []);

  const loadSettings = async () => {
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/indexnow'
      });
      
      if (response && response.settings) {
        setSettings({
          enabled: response.settings.enabled || false,
          api_key: response.settings.api_key || '',
          auto_submit: response.settings.auto_submit !== undefined ? response.settings.auto_submit : true,
          submit_on_publish: response.settings.submit_on_publish !== undefined ? response.settings.submit_on_publish : true,
          submit_on_update: response.settings.submit_on_update !== undefined ? response.settings.submit_on_update : true,
          engines: response.settings.engines || ['bing', 'yandex', 'seznam', 'yep']
        });
      }
    } catch (error) {
      // Use defaults - API might not exist yet
    } finally {
      setLoading(false);
    }
  };

  const loadSubmissionHistory = async () => {
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/indexnow/history'
      });
      
      if (response && response.submissions) {
        setSubmissionHistory(response.submissions);
      }
    } catch (error) {
      // Silently fail
    }
  };

  const saveSettings = async () => {
    setSaving(true);
    try {
      await apiFetch({
        path: '/prorank-seo/v1/settings/indexnow',
        method: 'POST',
        data: { settings }
      });
      
      // Place API key file in root
      if (settings.api_key) {
        await apiFetch({
          path: '/prorank-seo/v1/indexnow/deploy-key',
          method: 'POST',
          data: { api_key: settings.api_key }
        });
      }
    } catch (error) {
      // Handle error
    } finally {
      setSaving(false);
    }
  };

  const generateApiKey = async () => {
    setGenerateKeyLoading(true);
    try {
      // Generate a cryptographically secure random key
      const array = new Uint8Array(16);
      crypto.getRandomValues(array);
      const key = Array.from(array, byte => byte.toString(16).padStart(2, '0')).join('');
      
      setSettings({ ...settings, api_key: key });
    } finally {
      setGenerateKeyLoading(false);
    }
  };

  const testSubmission = async () => {
    setTesting(true);
    setTestResult(null);
    
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/indexnow/test',
        method: 'POST',
        data: {
          url: window.location.origin,
          api_key: settings.api_key
        }
      });
      
      setTestResult({
        success: response.success,
        message: response.message || __('Test submission successful!', 'prorank-seo')
      });
    } catch (error) {
      setTestResult({
        success: false,
        message: error.message || __('Test submission failed', 'prorank-seo')
      });
    } finally {
      setTesting(false);
    }
  };

  const submitBulkUrls = async () => {
    const urls = prompt(__('Enter URLs to submit (one per line):', 'prorank-seo'));
    if (!urls) return;
    
    const urlList = urls.split('\n').filter(url => url.trim());
    
    try {
      await apiFetch({
        path: '/prorank-seo/v1/indexnow/bulk',
        method: 'POST',
        data: {
          urls: urlList,
          api_key: settings.api_key
        }
      });
      
      loadSubmissionHistory();
    } catch (error) {
      // Handle error
    }
  };

  if (loading) {
    return (
      <div className="prorank-loading">
        <Spinner />
      </div>
    );
  }

  return (
    <div className="prorank-indexnow-integration">
      <div className="prorank-indexnow-header">
        <h3>{__('IndexNow Integration', 'prorank-seo')}</h3>
        <p className="description">
          {__('Submit URLs instantly to search engines using the IndexNow protocol. Get your content indexed in minutes instead of days.', 'prorank-seo')}
        </p>
      </div>

      <div className="prorank-setting-section">
        <ProrankToggle
          label={__('Enable IndexNow', 'prorank-seo')}
          help={__('Instantly notify search engines when content is created or updated', 'prorank-seo')}
          checked={settings.enabled}
          onChange={(enabled) => setSettings({ ...settings, enabled })}
        />
      </div>

      {settings.enabled && (
        <>
          <div className="prorank-setting-section">
            <div className="prorank-api-key-section">
              <TextControl
                label={__('IndexNow API Key', 'prorank-seo')}
                value={settings.api_key}
                onChange={(api_key) => setSettings({ ...settings, api_key })}
                help={__('Your unique API key for IndexNow authentication', 'prorank-seo')}
              />
              <div className="prorank-key-actions">
                <ProrankButton
                  variant="secondary"
                  size="small"
                  onClick={generateApiKey}
                  disabled={generateKeyLoading}
                >
                  {generateKeyLoading ? __('Generating...', 'prorank-seo') : __('Generate Key', 'prorank-seo')}
                </ProrankButton>
                {settings.api_key && (
                  <ProrankButton
                    variant="secondary"
                    size="small"
                    onClick={testSubmission}
                    disabled={testing}
                  >
                    {testing ? __('Testing...', 'prorank-seo') : __('Test Connection', 'prorank-seo')}
                  </ProrankButton>
                )}
              </div>
            </div>

            {testResult && (
              <Notice
                status={testResult.success ? 'success' : 'error'}
                isDismissible
                onRemove={() => setTestResult(null)}
              >
                {testResult.message}
              </Notice>
            )}
          </div>

          <div className="prorank-setting-section">
            <h4>{__('Automatic Submission', 'prorank-seo')}</h4>
            
            <ProrankToggle
              label={__('Submit on Publish', 'prorank-seo')}
              help={__('Automatically submit new content when published', 'prorank-seo')}
              checked={settings.submit_on_publish}
              onChange={(submit_on_publish) => setSettings({ ...settings, submit_on_publish })}
            />

            <ProrankToggle
              label={__('Submit on Update', 'prorank-seo')}
              help={__('Automatically submit content when updated', 'prorank-seo')}
              checked={settings.submit_on_update}
              onChange={(submit_on_update) => setSettings({ ...settings, submit_on_update })}
            />
          </div>

          <div className="prorank-setting-section">
            <h4>{__('Search Engines', 'prorank-seo')}</h4>
            <p className="description">
              {__('Select which search engines to notify:', 'prorank-seo')}
            </p>
            
            <div className="prorank-engines-list">
              {searchEngines.map(engine => (
                <label key={engine.id} className="prorank-checkbox-label">
                  <input
                    type="checkbox"
                    checked={settings.engines && settings.engines.includes(engine.id)}
                    onChange={(value) => {
                      const isChecked = typeof value === 'boolean' ? value : value?.target?.checked;
                      const currentEngines = settings.engines || [];
                      const engines = isChecked
                        ? [...currentEngines, engine.id]
                        : currentEngines.filter(id => id !== engine.id);
                      setSettings({ ...settings, engines });
                    }}
                  />
                  <span>{engine.name}</span>
                  <ExternalLink href={engine.url}>
                    {__('Info', 'prorank-seo')}
                  </ExternalLink>
                </label>
              ))}
            </div>
          </div>

          <div className="prorank-setting-section">
            <h4>{__('Bulk Submission', 'prorank-seo')}</h4>
            <p className="description">
              {__('Submit multiple URLs at once to IndexNow:', 'prorank-seo')}
            </p>
            <ProrankButton
              variant="secondary"
              onClick={submitBulkUrls}
            >
              {__('Submit URLs', 'prorank-seo')}
            </ProrankButton>
          </div>

          {submissionHistory.length > 0 && (
            <div className="prorank-setting-section">
              <h4>{__('Recent Submissions', 'prorank-seo')}</h4>
              <div className="prorank-submission-history">
                <table className="prorank-table">
                  <thead>
                    <tr>
                      <th>{__('URL', 'prorank-seo')}</th>
                      <th>{__('Status', 'prorank-seo')}</th>
                      <th>{__('Date', 'prorank-seo')}</th>
                    </tr>
                  </thead>
                  <tbody>
                    {submissionHistory.slice(0, 5).map((submission, index) => (
                      <tr key={index}>
                        <td>{submission.url}</td>
                        <td>
                          <span className={`prorank-status prorank-status-${submission.status}`}>
                            {submission.status}
                          </span>
                        </td>
                        <td>{new Date(submission.date).toLocaleString()}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </div>
          )}
        </>
      )}

      <div className="prorank-actions">
        <ProrankButton
          variant="primary"
          onClick={saveSettings}
          disabled={saving}
        >
          {saving ? __('Saving...', 'prorank-seo') : __('Save Settings', 'prorank-seo')}
        </ProrankButton>
      </div>

    </div>
  );
};

export default IndexNowIntegration;
