/* global wp */
/**
 * ProRank SEO Free - Main Admin App
 *
 * This is the main entry point for the ProRank SEO Free admin interface.
 * This version only includes FREE features for WordPress.org compliance.
 */

import React, { useState, useEffect, lazy, Suspense } from 'react';
import { __ } from '@wordpress/i18n';
import { Spinner } from '@wordpress/components';

// Initialize WordPress URL helpers (must be loaded early)
import './utils/urls';

// Setup apiFetch middleware
import setupApiFetch from './utils/setupApiFetch';

// Context Providers
import { NotificationProvider } from './contexts/NotificationContext';
import { UserPreferencesProvider } from './contexts/UserPreferencesContext';

// Global Components
import ToastContainer from './components/ToastContainer';
import ErrorBoundary from './components/ErrorBoundary';
import CommandPalette from './components/CommandPalette';
// UpgradeBanner removed - using page-specific CTAs instead

// Always loaded pages
import Dashboard from './pages/Dashboard';

// Lazy load pages to reduce initial bundle size
const OnPageSEO = lazy(() => import('./pages/OnPageSEO'));
const Performance = lazy(() => import('./pages/Performance'));
const TechnicalSEO = lazy(() => import('./pages/TechnicalSEO'));
const ImportExport = lazy(() => import('./pages/ImportExport'));
const InternalLinking = lazy(() => import('./pages/InternalLinking'));
const SiteAudit = lazy(() => import('./pages/SiteAudit/index.jsx'));
const SupportAccount = lazy(() => import('./pages/SupportAccount'));
// Direct import for BulkEdit to avoid lazy loading issues
import BulkEditDirect from './pages/BulkEdit';

// Styles - Import Tailwind CSS
import './styles/main.css';

// Loading component for lazy loaded pages
const PageLoader = () => (
  <div
    style={{
      display: 'flex',
      alignItems: 'center',
      justifyContent: 'center',
      minHeight: '400px',
      flexDirection: 'column',
      gap: '16px',
    }}
  >
    <Spinner />
    <p>{__('Loading page…', 'prorank-seo')}</p>
  </div>
);

/**
 * Main App Component - Free Version
 */
const App = () => {
  const [currentPage, setCurrentPage] = useState('');
  const [isReady, setIsReady] = useState(false);

  // Setup apiFetch on mount
  useEffect(() => {
    setupApiFetch();
  }, []);

  useEffect(() => {
    // Get the current page from URL parameters
    const urlParams = new URLSearchParams(window.location.search);
    const page = urlParams.get('page');
    const tab = urlParams.get('tab');
    const pageValue = page || '';
    const root = document.getElementById('prorank-admin-root');
    const dataPage = root?.getAttribute('data-page');

    if (dataPage) {
      setCurrentPage(dataPage);
      setIsReady(true);
      return;
    }

    // Check if we have a tab parameter (new menu structure)
    if (tab) {
      setCurrentPage(tab);
    } else if (pageValue === 'prorank-seo') {
      // Main dashboard page
      setCurrentPage('dashboard');
    } else if (pageValue.startsWith('prorank-')) {
      // Legacy format: Remove 'prorank-' prefix to get the sub-page
      const subPage = pageValue.replace('prorank-', '');
      setCurrentPage(subPage === 'seo-site-audit' ? 'site-audit' : subPage);
    } else {
      setCurrentPage('dashboard');
    }

    setIsReady(true);
  }, []);

  /**
   * Render the appropriate page component based on current page
   * Free version only - no premium pages
   */
  const renderPage = () => {
    switch (currentPage) {
      case 'dashboard':
      case '':
        return <Dashboard />;

      case 'on-page-seo':
        // Check if we're on the internal-linking tab
        const onPageUrlParams = new URLSearchParams(window.location.search);
        const onPageTab = onPageUrlParams.get('tab');
        if (onPageTab === 'internal-linking') {
          return <InternalLinking />;
        }
        return <OnPageSEO />;

      case 'performance':
        return <Performance />;

      case 'technical-seo':
        return <TechnicalSEO />;

      case 'site-audit':
        // Check if we're on the import-export tab
        const urlParams = new URLSearchParams(window.location.search);
        const tab = urlParams.get('tab');
        if (tab === 'import-export') {
          return <ImportExport />;
        }
        return <SiteAudit />;

      case 'tools':
      case 'import-export':
        return <ImportExport />;

      case 'internal-linking':
        return <InternalLinking />;

      case 'bulk-edit':
        return <BulkEditDirect />;

      case 'support-account':
        return <SupportAccount />;

      default:
        // If page not found, show dashboard
        return <Dashboard />;
    }
  };

  if (!isReady) {
    return (
      <div className="prorank-loading">
        <span className="spinner is-active"></span>
        <span className="prorank-loading-text">{__('Loading ProRank SEO…', 'prorank-seo')}</span>
      </div>
    );
  }

  return (
    <ErrorBoundary>
      <NotificationProvider>
        <UserPreferencesProvider>
          <div className="prorank-admin-app prorankAdmin" data-page={currentPage}>
            <div className="prorank-admin-content">
              <Suspense fallback={<PageLoader />}>{renderPage()}</Suspense>
            </div>

            <ToastContainer />

            {/* Global Command Palette */}
            <CommandPalette />
          </div>
        </UserPreferencesProvider>
      </NotificationProvider>
    </ErrorBoundary>
  );
};

export default App;
