<?php
declare(strict_types=1);

/**
 * ProRank SEO Helper Functions
 *
 * @package ProRank
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Get plugin instance
 *
 * @return \ProRank\SEO\Plugin
 */
if (!function_exists('prorank')) {
    function prorank() {
        return \ProRank\SEO\Plugin::get_instance();
    }
}

/**
 * Get plugin version
 *
 * @return string
 */
function prorank_version(): string {
    return PRORANK_VERSION;
}

/**
 * Get plugin directory path
 *
 * @param string $path Optional path to append.
 * @return string
 */
function prorank_dir( string $path = '' ): string {
    $dir = PRORANK_PLUGIN_DIR;
    
    if ( $path ) {
        $dir .= ltrim( $path, '/\\' );
    }
    
    return $dir;
}

/**
 * Get plugin URL
 *
 * @param string $path Optional path to append.
 * @return string
 */
function prorank_url( string $path = '' ): string {
    $url = PRORANK_PLUGIN_URL;
    
    if ( $path ) {
        $url .= ltrim( $path, '/\\' );
    }
    
    return $url;
}

/**
 * Get ProRank SEO uploads directory path.
 *
 * @param string $path Optional path to append.
 * @return string
 */
function prorank_upload_dir( string $path = '' ): string {
    $upload_dir = wp_upload_dir();
    $base = trailingslashit( $upload_dir['basedir'] ) . 'prorank-seo/';

    if ( $path ) {
        $base .= ltrim( $path, '/\\' );
    }

    return $base;
}

/**
 * Get ProRank SEO page cache directory path.
 *
 * @param string $path Optional path to append.
 * @return string
 */
function prorank_page_cache_dir( string $path = '' ): string {
    $base = trailingslashit( prorank_upload_dir( 'page-cache' ) );

    if ( $path ) {
        $base .= ltrim( $path, '/\\' );
    }

    return $base;
}

/**
 * Get ProRank SEO page cache URL.
 *
 * @param string $path Optional path to append.
 * @return string
 */
function prorank_page_cache_url( string $path = '' ): string {
    $upload_dir = wp_upload_dir();
    $base = trailingslashit( $upload_dir['baseurl'] ) . 'prorank-seo/page-cache/';

    if ( $path ) {
        $base .= ltrim( $path, '/\\' );
    }

    return $base;
}

/**
 * Get the home path for the site root.
 *
 * @return string
 */
function prorank_get_home_path(): string {
    if ( ! function_exists( 'get_home_path' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }

    if ( function_exists( 'get_home_path' ) ) {
        $home_path = get_home_path();
    } else {
        $upload_dir = wp_upload_dir();
        $home_path = $upload_dir['basedir'] ?? '';
    }
    return trailingslashit( wp_normalize_path( $home_path ) );
}

/**
 * Sanitize inline CSS for output.
 *
 * @param string $css
 * @return string
 */
function prorank_sanitize_inline_css(string $css): string {
    return trim(wp_strip_all_tags($css));
}

/**
 * Sanitize inline JavaScript output.
 *
 * @param string $js
 * @return string
 */
function prorank_sanitize_inline_js(string $js): string {
    return trim(wp_strip_all_tags($js));
}

/**
 * Sanitize full HTML output while allowing common markup.
 *
 * @param string $html
 * @return string
 */
function prorank_sanitize_full_html(string $html): string {
    $allowed = wp_kses_allowed_html( 'post' );

    $allowed['html'] = [ 'lang' => true, 'dir' => true, 'class' => true, 'id' => true ];
    $allowed['head'] = [ 'class' => true, 'id' => true ];
    $allowed['body'] = [ 'class' => true, 'id' => true ];
    $allowed['meta'] = [
        'charset' => true,
        'name' => true,
        'content' => true,
        'http-equiv' => true,
        'property' => true,
    ];
    $allowed['link'] = [
        'rel' => true,
        'href' => true,
        'type' => true,
        'media' => true,
        'sizes' => true,
        'as' => true,
        'crossorigin' => true,
        'referrerpolicy' => true,
    ];
    $allowed['script'] = [
        'type' => true,
        'src' => true,
        'async' => true,
        'defer' => true,
        'id' => true,
        'class' => true,
        'data-*' => true,
    ];
    $allowed['style'] = [
        'type' => true,
        'media' => true,
        'id' => true,
    ];
    $allowed['noscript'] = [];
    $allowed['iframe'] = [
        'src' => true,
        'width' => true,
        'height' => true,
        'frameborder' => true,
        'allow' => true,
        'allowfullscreen' => true,
        'loading' => true,
        'referrerpolicy' => true,
    ];
    $allowed['svg'] = [
        'xmlns' => true,
        'viewbox' => true,
        'fill' => true,
        'width' => true,
        'height' => true,
        'class' => true,
    ];
    $allowed['path'] = [
        'd' => true,
        'fill' => true,
        'stroke' => true,
        'stroke-width' => true,
        'class' => true,
    ];
    $allowed['g'] = [ 'fill' => true, 'stroke' => true, 'class' => true ];
    $allowed['defs'] = [];
    $allowed['use'] = [ 'href' => true, 'xlink:href' => true ];
    $allowed['symbol'] = [ 'id' => true, 'viewbox' => true ];
    $allowed['circle'] = [ 'cx' => true, 'cy' => true, 'r' => true, 'fill' => true ];
    $allowed['rect'] = [ 'x' => true, 'y' => true, 'width' => true, 'height' => true, 'fill' => true ];
    $allowed['line'] = [ 'x1' => true, 'y1' => true, 'x2' => true, 'y2' => true, 'stroke' => true ];
    $allowed['polyline'] = [ 'points' => true, 'stroke' => true, 'fill' => true ];
    $allowed['polygon'] = [ 'points' => true, 'stroke' => true, 'fill' => true ];

    return wp_kses( $html, $allowed );
}

/**
 * Sanitize CSS output for cached CSS responses.
 *
 * @param string $css
 * @return string
 */
function prorank_sanitize_css_output(string $css): string {
    $css = wp_strip_all_tags( $css );
    return preg_replace( '/[\x00-\x08\x0B\x0C\x0E-\x1F]/', '', $css );
}

/**
 * Read a local file using WP_Filesystem when possible.
 *
 * @param string $path
 * @return string
 */
function prorank_read_file(string $path): string {
    if ($path === '' || !is_readable($path)) {
        return '';
    }

    if ( wp_parse_url( $path, PHP_URL_SCHEME ) ) {
        return '';
    }

    global $wp_filesystem;
    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    WP_Filesystem();

    if ( $wp_filesystem ) {
        $content = $wp_filesystem->get_contents($path);
        return is_string($content) ? $content : '';
    }

    return '';
}

/**
 * Check if we're in development mode
 *
 * @return bool
 */
function prorank_is_dev(): bool {
    $settings = get_option( 'prorank_seo_advanced_settings', [] );
    $debug_enabled = is_array( $settings ) && ! empty( $settings['debug_mode'] );

    if ( $debug_enabled ) {
        return true;
    }

    return defined( 'WP_DEBUG' ) && WP_DEBUG;
}

/**
 * Safe JSON encode with error handling
 *
 * @param mixed $data Data to encode.
 * @param int   $options JSON encode options.
 * @param int   $depth Maximum depth.
 * @return string|false
 */
function prorank_json_encode( $data, int $options = 0, int $depth = 512 ) {
    $json = json_encode( $data, $options, $depth );
    
    if ( json_last_error() !== JSON_ERROR_NONE ) {
        if ( prorank_is_dev() ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log( 'ProRank SEO JSON encode error: ' . json_last_error_msg() );
            }
        }
        return false;
    }
    
    return $json;
}

/**
 * Safe JSON decode with error handling
 *
 * @param string $json JSON string to decode.
 * @param bool   $associative Return associative array.
 * @param int    $depth Maximum depth.
 * @param int    $options Decode options.
 * @return mixed|null
 */
function prorank_json_decode( string $json, bool $associative = true, int $depth = 512, int $options = 0 ) {
    $data = json_decode( $json, $associative, $depth, $options );
    
    if ( json_last_error() !== JSON_ERROR_NONE ) {
        if ( prorank_is_dev() ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log( 'ProRank SEO JSON decode error: ' . json_last_error_msg() );
            }
        }
        return null;
    }
    
    return $data;
}

/**
 * Get asset URL with version for cache busting
 *
 * @param string $asset Asset path relative to plugin directory.
 * @return string
 */
function prorank_asset_url( string $asset ): string {
    $url = prorank_url( $asset );
    $version = prorank_version();
    
    if ( prorank_is_dev() ) {
        $version .= '.' . time();
    }
    
    return add_query_arg( 'ver', $version, $url );
}

/**
 * Check if a module is enabled
 *
 * @param string $module Module slug.
 * @return bool
 */
function prorank_is_module_enabled( string $module ): bool {
    // Get the module manager instance if plugin is loaded
    if ( class_exists( '\ProRank\SEO\Plugin' ) ) {
        try {
            return prorank()->modules()->is_active( $module );
        } catch ( \Exception $e ) {
            // Fallback if module manager isn't ready
            return false;
        }
    }
    
    return false;
}

/**
 * Check if internal link click tracking is enabled.
 *
 * @return bool
 */
function prorank_is_click_tracking_enabled(): bool {
    $settings = get_option( 'prorank_seo_internal_linking', null );
    if ( is_array( $settings ) && array_key_exists( 'track_clicks', $settings ) ) {
        return (bool) $settings['track_clicks'];
    }

    $legacy_settings = get_option( 'prorank_internal_linking_settings', null );
    if ( is_array( $legacy_settings ) && array_key_exists( 'track_clicks', $legacy_settings ) ) {
        return (bool) $legacy_settings['track_clicks'];
    }

    $legacy = get_option( 'prorank_link_tracking_enabled', null );
    if ( null !== $legacy ) {
        return (bool) $legacy;
    }

    $legacy = get_option( 'prorank_seo_enable_click_tracking', null );
    if ( null !== $legacy ) {
        return (bool) $legacy;
    }

    return false;
}

/**
 * Get internal linking settings with legacy fallback.
 *
 * @return array
 */
function prorank_get_internal_linking_settings(): array {
    $defaults = [
        'enabled' => true,
        'auto_linking' => false,
        'max_suggestions' => 10,
        'max_links_per_post' => 5,
        'min_word_count' => 100,
        'link_post_types' => ['post', 'page'],
        'exclude_categories' => [],
        'ignore_words' => '',
        'ignored_words' => '',
        'open_in_new_tab' => false,
        'open_internal_new_tab' => false,
        'open_external_new_tab' => true,
        'add_nofollow' => false,
        'add_title_attribute' => true,
        'track_clicks' => false,
        'ai_suggestions' => false,
        'suggestion_algorithm' => 'balanced',
        'anchor_text_length' => 4,
        'skip_sentences_with_links' => true,
        'randomize_suggestions' => false,
        'max_links_per_paragraph' => 2,
    ];

    $settings = get_option('prorank_seo_internal_linking', []);
    if (!is_array($settings)) {
        $settings = [];
    }

    $legacy_settings = get_option('prorank_internal_linking_settings', []);
    if (!is_array($legacy_settings)) {
        $legacy_settings = [];
    }

    return array_merge($defaults, $legacy_settings, $settings);
}

/**
 * Check if RUM collection is enabled.
 *
 * @return bool
 */
function prorank_is_rum_enabled(): bool {
    $enabled = get_option( 'prorank_rum_enabled', null );
    if ( null !== $enabled ) {
        return (bool) $enabled;
    }

    $beta = get_option( 'prorank_performance_beta', null );
    if ( null !== $beta ) {
        return (bool) $beta;
    }

    return (bool) apply_filters( 'prorank_enable_rum', false );
}

/**
 * Get plugin option with type safety
 *
 * @param string $option Option name.
 * @param mixed  $default Default value.
 * @return mixed
 */
function prorank_get_option( string $option, $default = null ) {
    // Get the settings manager instance if plugin is loaded
    if ( class_exists( '\ProRank\SEO\Plugin' ) ) {
        try {
            return prorank()->settings()->get( $option, $default );
        } catch ( \Exception $e ) {
            // Fallback to WordPress option
            return get_option( 'prorank_' . $option, $default );
        }
    }
    
    return get_option( 'prorank_' . $option, $default );
}

/**
 * Update plugin option with type safety
 *
 * @param string $option Option name.
 * @param mixed  $value Option value.
 * @return bool
 */
function prorank_update_option( string $option, $value ): bool {
    // Get the settings manager instance if plugin is loaded
    if ( class_exists( '\ProRank\SEO\Plugin' ) ) {
        try {
            return prorank()->settings()->update( $option, $value );
        } catch ( \Exception $e ) {
            // Fallback to WordPress option
            return update_option( 'prorank_' . $option, $value );
        }
    }
    
    return update_option( 'prorank_' . $option, $value );
}

/**
 * Log a message if in development mode
 *
 * @param string $message Message to log.
 * @param string $level Log level (error, warning, info).
 */
function prorank_log( string $message, string $level = 'info' ): void {
    if ( ! prorank_is_dev() ) {
        return;
    }

    $level = strtolower( $level );
    if ( ! prorank_log_level_allows( $level ) ) {
        return;
    }

    $prefix = '[ProRank SEO] [' . strtoupper( $level ) . '] ';
    $line = $prefix . $message;

    do_action( 'prorank_seo_log', $line, $level );
}

/**
 * Get configured log level.
 *
 * @return string
 */
function prorank_get_log_level(): string {
    $settings = get_option( 'prorank_seo_advanced_settings', [] );
    $level = '';

    if ( is_array( $settings ) && ! empty( $settings['log_level'] ) ) {
        $level = $settings['log_level'];
    }

    if ( $level === '' ) {
        $level = get_option( 'prorank_seo_log_level', 'error' );
    }

    if ( ! in_array( $level, [ 'error', 'warning', 'info', 'debug' ], true ) ) {
        $level = 'error';
    }

    return $level;
}

/**
 * Check whether a log level should be emitted.
 *
 * @param string $level Log level.
 * @return bool
 */
function prorank_log_level_allows( string $level ): bool {
    $levels = [
        'error' => 0,
        'warning' => 1,
        'info' => 2,
        'debug' => 3,
    ];

    $level = strtolower( $level );
    if ( ! isset( $levels[ $level ] ) ) {
        $level = 'info';
    }

    $configured = prorank_get_log_level();

    return $levels[ $level ] <= $levels[ $configured ];
}

/**
 * Get a sanitized server variable.
 *
 * @param string $key Server key.
 * @return string
 */
function prorank_get_server_var( string $key ): string {
    if ( ! isset( $_SERVER[ $key ] ) ) {
        return '';
    }

    return sanitize_text_field( wp_unslash( $_SERVER[ $key ] ) );
}

/**
 * Get basic schema settings with defaults.
 *
 * @return array
 */
function prorank_get_basic_schema_settings(): array {
    $defaults = [
        'disable_all_schema' => false,
        'enable_website_schema' => true,
        'enable_organization_schema' => true,
        'enable_article_schema' => true,
        'enable_breadcrumb_schema' => true,
        'organization_name' => get_bloginfo( 'name' ),
        'organization_logo' => '',
        'organization_type' => 'Organization',
    ];

    $settings = get_option( 'prorank_basic_schema_settings', [] );
    if ( ! is_array( $settings ) ) {
        return $defaults;
    }

    return array_merge( $defaults, $settings );
}

/**
 * Check if a basic schema setting is enabled.
 *
 * @param string $key Setting key.
 * @return bool
 */
function prorank_is_basic_schema_enabled( string $key ): bool {
    $settings = prorank_get_basic_schema_settings();

    if ( ! empty( $settings['disable_all_schema'] ) ) {
        return false;
    }

    return ! empty( $settings[ $key ] );
}

/**
 * Check if schema output is globally enabled.
 *
 * @return bool
 */
function prorank_is_schema_output_enabled(): bool {
    $settings = prorank_get_basic_schema_settings();

    return empty( $settings['disable_all_schema'] );
}
