<?php
/**
 * Main Plugin Class
 *
 * @package ProRank\SEO
 */

declare(strict_types=1);

namespace ProRank\SEO;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Core\ServiceContainer;
use ProRank\SEO\Core\Config\DomainConfig;
use ProRank\SEO\Core\SettingsManager;
use ProRank\SEO\Core\NotificationManager;
use ProRank\SEO\Core\LicenseManager;
use ProRank\SEO\Core\ModuleManager;
use ProRank\SEO\Core\ModuleLoader;
// use ProRank\SEO\Core\AdminPage;
use ProRank\SEO\Core\AdminMenus;
// use ProRank\SEO\Admin\AdminNotices;
use ProRank\SEO\Core\PresetManager;
use ProRank\SEO\Core\SettingsSnapshot;
use ProRank\SEO\Core\Buffer;
// Import REST API Controllers
use ProRank\SEO\Core\RestApi\DashboardController;
use ProRank\SEO\Core\RestApi\ModulesController;
use ProRank\SEO\Core\RestApi\UserPreferencesController;
use ProRank\SEO\Core\RestApi\SettingsController;
use ProRank\SEO\Core\RestApi\AdvancedSettingsController;
use ProRank\SEO\Core\RestApi\DataController;
use ProRank\SEO\Core\RestApi\MetaController;
use ProRank\SEO\Core\RestApi\TitlesMetaBulkController;
use ProRank\SEO\Core\RestApi\OnboardingController;
use ProRank\SEO\Core\RestApi\SitemapsController;
use ProRank\SEO\Core\RestApi\ReadabilityController;
use ProRank\SEO\Core\RestApi\HeadlineAnalyzerController;
use ProRank\SEO\Core\RestApi\LinkingController;
use ProRank\SEO\Core\RestApi\RedirectsController;
use ProRank\SEO\Core\RestApi\ImportExportController;
use ProRank\SEO\Core\RestApi\EnhancedAuditController;
use ProRank\SEO\Core\RestApi\CommandPaletteController;
use ProRank\SEO\Core\RestApi\PresetsController;
use ProRank\SEO\Core\RestApi\CacheController;
// use ProRank\SEO\Core\RestApi\SiteAuditController; // Removed - using EnhancedAuditController instead
use ProRank\SEO\Core\RestApi\SettingsSnapshotController;
use ProRank\SEO\Core\RestApi\ImageOptimizationController;
use ProRank\SEO\Core\RestApi\ImagesController;
use ProRank\SEO\Core\RestApi\FontOptimizationController;
use ProRank\SEO\Core\RestApi\SystemController;
use ProRank\SEO\Core\RestApi\PerformanceSettingsController;
use ProRank\SEO\Core\RestApi\WizardStateController;
use ProRank\SEO\Core\Tasks\LocalCssCacheQueue;
use ProRank\SEO\Frontend\HeadOutput;
use ProRank\SEO\Frontend\SiteBasicsOutput;
use ProRank\SEO\Frontend\SpeculationRules;
use ProRank\SEO\Frontend\VirtualTxtFiles;
use ProRank\SEO\Admin\BulkEditIntegration;
use ProRank\SEO\Services\Cdn\CdnManager;
use ProRank\SEO\Frontend\ThirdPartyDefer;
use ProRank\SEO\Frontend\LazyMedia;
use ProRank\SEO\Frontend\RumBeacon;

/**
 * Main Plugin Class
 */
class Plugin {
    
    /**
     * Plugin version
     *
     * @var string
     */
    const VERSION = '1.0.0';
    
    /**
     * Plugin instance
     *
     * @var Plugin|null
     */
    private static ?Plugin $instance = null;
    
    /**
     * Service container
     *
     * @var ServiceContainer
     */
    private ServiceContainer $services;
    
    /**
     * Admin notices handler
     *
     * @var AdminNotices
     */
    // private AdminNotices $admin_notices;
    
    /**
     * Get plugin instance
     *
     * @return Plugin
     */
    public static function get_instance(): Plugin {
        static $creating = false;
        
        if ( null === self::$instance ) {
            if ($creating) {
                // Prevent recursive instantiation
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log('[ProRank SEO] ERROR: Recursive Plugin instantiation detected!');
                }
                throw new \RuntimeException('Recursive Plugin instantiation detected');
            }
            
            $creating = true;
            self::$instance = new self();
            $creating = false;
        }
        
        return self::$instance;
    }

    /**
     * Handle content changes to trigger cache/CDN purge
     */
    public function handle_content_change( $post_id, $post, $update ): void {
        if ( wp_is_post_autosave( $post_id ) || wp_is_post_revision( $post_id ) ) {
            return;
        }
        $this->purge_urls(
            array_merge(
                $this->collect_urls_for_post( $post_id ),
                $this->collect_global_urls()
            )
        );
    }

    /**
     * Handle post deletion
     */
    public function handle_content_delete( $post_id ): void {
        $post_id = (int) $post_id;
        if ( $post_id <= 0 ) {
            return;
        }
        $this->purge_urls(
            array_merge(
                $this->collect_urls_for_post( $post_id ),
                $this->collect_global_urls()
            )
        );
    }

    /**
     * Handle status transitions (e.g., draft -> publish)
     */
    public function handle_post_status_transition( $new_status, $old_status, $post ): void {
        if ( $new_status === $old_status ) {
            return;
        }
        $this->purge_urls(
            array_merge(
                $this->collect_urls_for_post( $post->ID ),
                $this->collect_global_urls()
            )
        );
    }

    /**
     * Handle menu updates
     */
    public function handle_menu_change( $nav_menu_id ): void {
        $this->purge_urls( $this->collect_global_urls() );
    }

    /**
     * Collect URLs related to a post for purging
     */
    private function collect_urls_for_post( int $post_id ): array {
        $urls = [];
        $permalink = get_permalink( $post_id );
        if ( $permalink ) {
            $urls[] = $permalink;
        }
        return array_filter( array_unique( $urls ) );
    }

    /**
     * Collect sitewide URLs that should be purged on structural changes
     */
    private function collect_global_urls(): array {
        $urls = [];
        if ( function_exists( 'home_url' ) ) {
            $urls[] = home_url( '/' );
            $urls[] = home_url( '/feed/' );
            $urls[] = home_url( '/sitemap.xml' );
            $urls[] = home_url( '/sitemap_index.xml' );
            $urls[] = home_url( '/robots.txt' );
        }
        $feed_link = function_exists( 'get_feed_link' ) ? get_feed_link() : '';
        if ( $feed_link ) {
            $urls[] = $feed_link;
        }
        return array_filter( array_unique( $urls ) );
    }

    /**
     * Purge URLs from CDN if configured
     */
    private function purge_urls( array $urls ): void {
        if ( empty( $urls ) ) {
            return;
        }
        $urls = array_unique( array_filter( $urls ) );
        $this->purge_litespeed_cache( $urls );
        if ( ! class_exists( CdnManager::class ) ) {
            return;
        }
        try {
            $cdn = new CdnManager();
            $result = $cdn->purge_urls( $urls );
            if ( is_wp_error( $result ) ) {
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log( 'ProRank CDN purge failed: ' . $result->get_error_message() );
                }
            } else {
                do_action( 'prorank_cdn_cache_cleared', get_option( 'prorank_cdn_provider', 'none' ), $urls );
                do_action( 'prorank_audit_log_event', 'cdn_purge', [ 'urls' => $urls, 'provider' => get_option( 'prorank_cdn_provider', 'none' ) ] );
            }
        } catch ( \Throwable $e ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log( 'ProRank CDN purge exception: ' . $e->getMessage() );
            }
        }
    }

    /**
     * Best-effort LiteSpeed Cache purge when LSCache is active.
     *
     * @param array $urls URLs to purge.
     */
    private function purge_litespeed_cache( array $urls ): void {
        if ( empty( $urls ) ) {
            return;
        }
        if ( ! defined( 'LSCWP_V' ) && ! class_exists( 'LiteSpeed_Cache_API' ) ) {
            return;
        }

        if ( class_exists( 'LiteSpeed_Cache_API' ) ) {
            if ( method_exists( 'LiteSpeed_Cache_API', 'purge_url' ) ) {
                foreach ( $urls as $url ) {
                    \LiteSpeed_Cache_API::purge_url( $url );
                }
                return;
            }
            if ( method_exists( 'LiteSpeed_Cache_API', 'purge' ) ) {
                foreach ( $urls as $url ) {
                    \LiteSpeed_Cache_API::purge( $url );
                }
                return;
            }
            if ( method_exists( 'LiteSpeed_Cache_API', 'purge_all' ) ) {
                \LiteSpeed_Cache_API::purge_all();
                return;
            }
        }

        // phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- LiteSpeed Cache hooks.
        if ( has_action( 'litespeed_purge_all' ) ) {
            do_action( 'litespeed_purge_all' );
            return;
        }
        if ( has_action( 'litespeed_purge_url' ) ) {
            foreach ( $urls as $url ) {
                do_action( 'litespeed_purge_url', $url );
            }
            return;
        }
        if ( has_action( 'litespeed_purge' ) ) {
            foreach ( $urls as $url ) {
                do_action( 'litespeed_purge', $url );
            }
        }
        // phpcs:enable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Initialize domain configuration first
        DomainConfig::init();
        
        $this->services = new ServiceContainer();
        // $this->admin_notices = new AdminNotices();
        
        $this->init_services();
        $this->init_hooks();
        
    }
    
    /**
     * Initialize services
     */
    private function init_services(): void {
        // Initialize core services in dependency order
        $this->services->register('settings', fn() => new SettingsManager());
        $this->services->register('notifications', fn() => new NotificationManager());
        $this->services->register('modules', fn() => new ModuleManager());
        $this->services->register('menus', fn() => new \ProRank\SEO\Core\MenuRegistry());
        $this->services->register('presets', function() {
            return new PresetManager($this->settings());
        });
        $this->services->register('snapshots', function() {
            return new SettingsSnapshot($this->settings());
        });

        // Module loading will be done after plugin is fully initialized
    }
    
    /**
     * Initialize Action Scheduler
     */
    private function init_action_scheduler(): void {
        // Load Action Scheduler if not already loaded
        $action_scheduler_path = PRORANK_PLUGIN_DIR . 'vendor/woocommerce/action-scheduler/action-scheduler.php';
        if ( file_exists( $action_scheduler_path ) && ! class_exists( 'ActionScheduler' ) ) {
            require_once $action_scheduler_path;
        }
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks(): void {
        // Initialize Action Scheduler
        $this->init_action_scheduler();
        
        // Load modules very early
        add_action( 'plugins_loaded', [ $this, 'load_modules' ], 5 );
        // Ensure background image optimization actions always have a callback
        add_action( 'plugins_loaded', [ $this, 'register_image_optimization_actions' ], 20 );
        
        // Core initialization
        add_action( 'init', [ $this, 'init' ] );
        add_action( 'admin_init', [ $this, 'admin_init' ] );
        
        // Admin menu - using MenuHandler
        $menu_handler = new \ProRank\SEO\Admin\MenuHandler($this);
        add_action( 'admin_menu', [ $menu_handler, 'init' ] );
        
        // Initialize Meta Box
        $meta_box = new \ProRank\SEO\Admin\MetaBox($this);
        $meta_box->init();
        
        // Initialize Editor Sidebar
        $editor_sidebar = new \ProRank\SEO\Admin\EditorSidebar($this);
        $editor_sidebar->init();
        
        // Initialize Admin Bar Menu
        $admin_bar_menu = new \ProRank\SEO\Admin\AdminBarMenu();
        $admin_bar_menu->init();
        
        // Initialize Menu Styles
        $menu_styles = new \ProRank\SEO\Admin\MenuStyles();
        $menu_styles->init();

        // Admin scripts (enqueue main React app + import map)
        new \ProRank\SEO\Admin\AdminScripts($this);
        
        // Admin assets
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_assets' ] );

        // Audit report HTML download (admin-ajax)
        add_action( 'wp_ajax_prorank_audit_report', [ new \ProRank\SEO\Core\RestApi\EnhancedAuditController($this->services), 'download_audit_report_html' ] );
        
        // Media library AI integration is not available in the free plugin

        // Frontend assets (if needed)
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_frontend_assets' ] );

        // REST API initialization
        add_action( 'rest_api_init', [ $this, 'init_rest_api' ] );

        // Initialize Frontend Head Output
        add_action( 'init', [ $this, 'init_frontend_head_output' ] );
        add_action( 'init', [ $this, 'init_virtual_txt_files' ] );
        add_action( 'init', [ $this, 'init_speculation_rules' ] );
        add_action( 'init', [ $this, 'init_third_party_defer' ] );
        add_action( 'init', [ $this, 'init_lazy_media' ] );
        add_action( 'init', [ $this, 'init_rum_beacon' ] );
        add_action( 'init', [ $this, 'ensure_rum_guard_schedule' ] );
        add_action( 'init', [ $this, 'init_local_css_cache_queue' ] );
        
        // Cache/CDN purges on content changes
        add_action( 'save_post', [ $this, 'handle_content_change' ], 10, 3 );
        add_action( 'deleted_post', [ $this, 'handle_content_delete' ], 10, 1 );
        add_action( 'transition_post_status', [ $this, 'handle_post_status_transition' ], 10, 3 );
        add_action( 'wp_update_nav_menu', [ $this, 'handle_menu_change' ], 10, 1 );

        // Background performance jobs
        add_action( 'prorank_generate_critical_css', [ $this, 'run_critical_css_cron' ], 10, 1 );
        add_action( 'prorank_scan_unused_css', [ $this, 'run_unused_css_cron' ], 10, 1 );
        add_action( 'prorank_rum_guard', [ $this, 'run_rum_guard' ], 10, 0 );
    }
    
    /**
     * Load modules early
     */
    public function load_modules(): void {
        static $modules_loaded = false;
        if (!$modules_loaded) {
            $module_loader = new ModuleLoader($this->modules());
            $module_loader->load();
            $modules_loaded = true;
            
            // Initialize modules immediately
            $this->modules()->init_active_modules();
        }
    }

    /**
     * Register image optimization background actions even when modules are not initialized.
     *
     * @return void
     */
    public function register_image_optimization_actions(): void {
        if ( ! has_action( 'prorank_bulk_optimization_continue' ) ) {
            add_action( 'prorank_bulk_optimization_continue', [ $this, 'handle_bulk_optimization_continue' ], 10, 2 );
        }

        if ( ! has_action( 'prorank_bulk_optimization_batch' ) ) {
            add_action( 'prorank_bulk_optimization_batch', [ $this, 'handle_bulk_optimization_batch' ], 10, 2 );
        }

        if ( ! has_action( 'prorank_optimize_image' ) ) {
            add_action( 'prorank_optimize_image', [ $this, 'handle_optimize_image_action' ], 10, 1 );
        }
    }

    /**
     * Handle bulk optimization continuation when Action Scheduler fires.
     *
     * @param string $job_id Job ID.
     * @param int    $next_offset Next offset.
     * @return void
     */
    public function handle_bulk_optimization_continue( $job_id = '', $next_offset = 0 ): void {
        if ( ! class_exists( ImageOptimizationController::class ) ) {
            return;
        }

        $controller = new ImageOptimizationController();
        $controller->handle_bulk_optimization_continue( (string) $job_id, (int) $next_offset );
    }

    /**
     * Handle bulk optimization batch when WP-Cron fallback fires.
     *
     * @param string $job_id Job ID.
     * @param int    $offset Offset for the next batch.
     * @return void
     */
    public function handle_bulk_optimization_batch( $job_id = '', $offset = 0 ): void {
        if ( ! class_exists( ImageOptimizationController::class ) ) {
            return;
        }

        $controller = new ImageOptimizationController();
        $controller->handle_bulk_optimization_continue( (string) $job_id, (int) $offset );
    }

    /**
     * Handle single image optimization when Action Scheduler fires.
     *
     * @param mixed $attachment_id Attachment ID or args array.
     * @return void
     */
    public function handle_optimize_image_action( $attachment_id = 0 ): void {
        if ( is_array( $attachment_id ) ) {
            $attachment_id = $attachment_id['attachment_id'] ?? 0;
        }

        $attachment_id = (int) $attachment_id;
        if ( $attachment_id <= 0 ) {
            return;
        }

        $module = $this->modules()->get_module( 'image_optimization' );
        if ( is_object( $module ) && method_exists( $module, 'process_single_image_optimization' ) ) {
            $module->process_single_image_optimization( $attachment_id );
            return;
        }

        if ( class_exists( \ProRank\SEO\Modules\Performance\ImageOptimizationModule::class ) ) {
            $fallback = new \ProRank\SEO\Modules\Performance\ImageOptimizationModule();
            $fallback->process_single_image_optimization( $attachment_id );
        }
    }
    
    /**
     * General initialization
     */
    public function init(): void {
        // Apply saved advanced settings early (security, features, etc.)
        AdvancedSettingsController::apply_saved_settings();

        // Ensure modules are loaded (in case plugins_loaded didn't fire)
        $this->load_modules();

        // Load Elementor integration if Elementor is active
        $this->load_elementor_integration();

        // Check for competitor plugins
        $this->check_competitors();

        // Audit system now handled by external prorank.io service
        // No local tables needed

        // Initialize centralized output buffering for front-end optimizations
        Buffer::init();

    }

    /**
     * Initialize local CSS cache queueing.
     *
     * @return void
     */
    public function init_local_css_cache_queue(): void {
        if ( is_admin() ) {
            return;
        }

        static $initialized = false;
        if ( $initialized ) {
            return;
        }
        $initialized = true;

        if ( class_exists( LocalCssCacheQueue::class ) ) {
            $queue = new LocalCssCacheQueue();
            $queue->init();
        }
    }
    
    /**
     * Admin initialization
     */
    public function admin_init(): void {
        // Register settings
        // $this->settings()->register_settings();
        
        // Initialize media enhancements
        new Admin\MediaEnhancements($this);

        // Initialize media library optimization features
        new Admin\MediaLibraryOptimization($this);

        // Initialize bulk edit integration
        new BulkEditIntegration();

        // Initialize user profile E-E-A-T fields
        $user_profile_fields = new Admin\UserProfileFields();
        $user_profile_fields->init();

        // Initialize dashboard widget
        $dashboard_widget = new Admin\DashboardWidget();
        $dashboard_widget->init();
        
        // Check for plugin updates
        $this->check_updates();
        
        // Initialize admin notices
        // $this->admin_notices->init();
        
        // Initialize onboarding
        $this->init_onboarding();
        
        // Initialize audit settings page
        if (class_exists('\ProRank\SEO\Admin\Pages\AuditSettings')) {
            $audit_settings = new \ProRank\SEO\Admin\Pages\AuditSettings();
            $audit_settings->init();
        }
        
        // Local SEO components are not loaded in the free plugin
    }
    
    /**
     * Initialize admin menu
     * COMMENTED OUT: Using MenuHandler instead to avoid duplicate menu registration
     * All render methods below are preserved and used by MenuHandler
     */
    /*
    public function init_admin_menu(): void {
        // Direct WordPress menu registration (bypass MenuRegistry for now)
        add_menu_page(
            __('ProRank SEO', 'prorank-seo'),
            __('ProRank SEO', 'prorank-seo'),
            'manage_options',
            'prorank-seo',
            [$this, 'render_admin_page'],
            $this->get_menu_icon(),
            30
        );
        
        // Register dashboard as first submenu (replaces parent)
        add_submenu_page(
            'prorank-seo',
            __('Dashboard', 'prorank-seo'),
            __('Dashboard', 'prorank-seo'),
            'manage_options',
            'prorank-seo',
            [$this, 'render_admin_page']
        );
        
        // Add standard submenus based on module groups from ia-modules.json
        $module_groups = [
            [
                'slug' => 'on-page-seo',
                'title' => __('On-Page SEO', 'prorank-seo'),
                'capability' => 'manage_options',
            ],
            [
                'slug' => 'technical-seo',
                'title' => __('Technical SEO', 'prorank-seo'),
                'capability' => 'manage_options',
            ],
            [
                'slug' => 'performance',
                'title' => __('Performance', 'prorank-seo'),
                'capability' => 'manage_options',
            ],
            [
                'slug' => 'tools',
                'title' => __('Tools', 'prorank-seo'),
                'capability' => 'manage_options',
            ],
        ];
        
        foreach ($module_groups as $group) {
            $menu_title = $group['title'];
            
            // Add badge if specified
            if (!empty($group['badge'])) {
                $badge_class = $group['badge'] === 'new' ? 'prorank-menu-badge-new' : 'prorank-menu-badge';
                $menu_title .= sprintf(' <span class="%s">%s</span>', $badge_class, strtoupper($group['badge']));
            }
            
            add_submenu_page(
                'prorank-seo',
                $group['title'],
                $menu_title,
                $group['capability'],
                'prorank-seo&tab=' . $group['slug'],
                [$this, 'render_admin_page']
            );
        }
        
        // Add settings page at the end
        add_submenu_page(
            'prorank-seo',
            __('Settings', 'prorank-seo'),
            __('Settings', 'prorank-seo'),
            'manage_options',
            'prorank-seo-settings',
            [$this, 'render_settings_page']
        );
        
    }
    */
    
    /**
     * Get menu icon
     *
     * Returns the ProRank favicon as a data URI for WordPress admin menu.
     *
     * @return string
     */
    private function get_menu_icon(): string {
        // ProRank favicon - orange P on dark background
        $svg = '<svg version="1.1" viewBox="0 0 1909 2048" width="2106" height="2259" xmlns="http://www.w3.org/2000/svg"><path transform="translate(959,1)" d="m0 0h10l6 3 5 8 12 32 8 19 9 24 7 20 16 40 10 26 3 10 13 34 4 9 12 31 7 20 10 25 9 23 7 18 6 18 8 21 7 16 11 29 3 10 14 36 5 12 11 28 4 13 12 36 3 16 1 13v13l-2 16-2 4-2 1-14 1-424 11h-14l-6 4-6 7-2 6-1 30-1 149v195l-3 8-7 6-35 12-28 11-27 10-37 13-16 6-12 5-30 11-11-2-11-9-16-13-11-8-9-8-11-9-5-4-5-5-14-12-10-8-12-9-13-11-11-9-5-4-5-5-13-11-14-11-9-7-13-11-14-11-8-8-14-11-9-8-6-5-5-3-13-11-11-9-10-9-13-11-11-9-8-6-10-8-13-11-11-9-12-11-10-8-11-9-11-8-15-13-10-8-12-11-22-18-11-8-15-13-10-8-12-11-11-9-4-5-1-2v-12l4-6 8-4 27-1 36-1 11-1 24-1 47-1 42-2 47-1 35-2 47-1 26-1 9-1 73-2 42-2h19l44-2 19-1 47-1 25-1 10-1 35-1h12l12-1 10-5 4-7 7-20 9-24 5-11 10-27 6-18 11-29 4-8 13-35 7-20 11-28 10-26 4-10 4-13 12-31 8-20 8-21 6-18 10-26 4-9 11-29 5-15 13-34 5-12 11-29 7-20 9-24 4-9 12-31 4-13 4-8z" fill="#0B121F"/><path transform="translate(728,646)" d="m0 0h324l24 2 24 4 20 5 20 7 16 7 19 10 15 10 14 11 13 12 11 12 10 13 10 15 11 21 7 18 7 25 4 25 1 11v62l-3 24-4 19-9 27-10 21-11 18-10 13-11 13-14 14-11 9-14 10-15 9-17 9-23 9-26 7-23 4-22 2h-185v275h-142z" fill="#FE6900"/><path transform="translate(1156,637)" d="m0 0h26l567 19 143 5 9 3 5 5 3 9-3 8-8 8-11 9-8 6-11 9-13 11-9 7-5 5-11 10-11 9-8 6-11 9-13 11-10 8-5 5-10 9-30 24-13 11-11 9-7 7-11 9-13 11-13 10-26 22-11 10-11 9-12 10-8 6-13 11-11 9-4 2-2 4-13 11-11 9-8 6-11 9-13 11-11 9-12 11-10 8-13 10-14 11-14 12-10 9-16 13-14 11-8 6-11 9-16 13-10 9-14 11-11 9-5 3-16 13-14 11-10 9-14 11-24 18-25 20-13 11-16 12-11 8-13 10-17 13-5 5-16 12-14 11-7 4-15 12-15 11-5 5-19 14-11 8-10 6-7 1-4-2h-12l-12 1-25 1-50 1-16 1h-17l-1-4v-31l1-24 1-59v-142l-3-1v-3h3l1 3 22 1 9 1 29 1h15l60 3h29l28-3 23-4 29-9 25-12 19-10 12-7 15-11 12-10 16-16 9-11 11-13 10-18 6-13 4-13 8-28 3-13 4-30 1-25v-39l-3-29-7-28-7-19-4-10-8-15-8-12-7-8-8-10-11-12-7-7v-2l-4-2-13-11-15-10-24-13-9-5-5-6-1-3v-11l3-6z" fill="#0B121F"/><path transform="translate(1369,1159)" d="m0 0 9 1 6 4 4 6 4 15 10 40 5 22 6 24 5 17 6 25 3 14 8 33 3 9 8 32 4 20 7 27 5 18 6 25 4 18 7 27 9 35 4 18 8 33 4 13 7 29 6 26 5 20 5 18 6 24 6 27 7 25 9 36v10l-3 6-6 5-9 2-8-3-16-8-19-10-18-11-29-16-12-6-21-12-21-13-24-14-12-6-22-13-15-10-27-16-16-9-21-13-13-9-24-15-14-8-23-15-16-11-17-11-11-7-9-5-20-13-10-7-23-15-14-9-9-5-17-11-12-8-10-7-25-16-14-8-18-12-5-6-3-9v-8l4-9 4-5 13-10 8-7 17-13 9-7 5-3 26-20 13-11 17-13 11-8 16-12 13-10 7-5 5-5 14-11 13-10 13-9 14-11 13-10 9-8 12-9 13-10 9-6 14-11 13-10 9-8 28-21 11-8 18-14 5-4h2l2-4 12-9 18-14 6-3z" fill="#0B121F"/><path transform="translate(714,1135)" d="m0 0h3l1 41v73l1 138-1 24h142l55 1h71l21 1-5 5-11 8-11 9-14 10-8 6-7 4-13 10-15 11h-2l-1 3-19 14-14 10-17 12-18 13-5 5-18 13-11 8-9 6-19 14-9 7-13 10-14 10-9 6-17 12-16 12-10 8-17 12-15 10-28 20h-2l-1 3-19 13-14 10-7 4-17 12-14 10-15 11-18 12-7 4-24 16-15 11-20 13-15 9-17 11-12 8-10 7-18 11-8 5-8 4-11 7-5-1-4-7-1-5v-13l7-30 4-13 7-29 3-15 7-28 4-14 6-25 3-15 7-28 4-15 6-24 3-15 7-28 4-15 6-24 3-15 7-28 4-15 6-24 5-23 5-20 4-15 6-24 3-15 7-28 4-15 6-24 3-15 7-28 5-18 2-5 25-10 48-17 10-3 37-13 15-6 37-13z" fill="#0B121F"/><path transform="translate(915,789)" d="m0 0h44l53 1 22 2 12 3 19 9 10 8 12 12 8 14 5 14 2 14v28l-3 19-7 16-7 12-4 2-2 4-10 9-13 8-11 5-12 3-8 1-31 1-36 1h-25l-29-1-12-1-1-23v-45l1-115z" fill="#0B121F"/><path transform="translate(870,776)" d="m0 0h170l22 4 15 5 15 8 13 10 11 11 10 16 6 15 4 20v41l-4 19-6 15-7 12-9 10-5 5-8 7-15 9-19 7-20 4-19 1h-154zm45 13-23 1-1 115v45l1 23 12 1 29 1h25l47-1 20-1 16-3 13-5 12-7 11-9 4-4v-2l4-2 8-14 6-14 3-19v-28l-2-14-5-14-8-14-15-15-11-7-18-8-16-3-24-1-44-1z" fill="#fff"/></svg>';
        return 'data:image/svg+xml;base64,' . base64_encode($svg);
    }
    
    /**
     * Render admin page
     */
    public function render_admin_page(): void {
        echo '<div class="wrap"><div id="prorank-admin-root" class="prorank-admin-container"></div></div>';
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page(): void {
        echo '<div class="wrap"><div id="prorank-admin-root" class="prorank-admin-container"></div></div>';
    }
    
    /**
     * Enqueue admin assets
     *
     * @param string $hook_suffix Current admin page hook suffix
     */
    public function enqueue_admin_assets( string $hook_suffix ): void {
        // Only load main admin assets on our admin pages
        if ( ! $this->is_prorank_admin_page( $hook_suffix ) ) {
            return;
        }
        
        // Get asset file for admin bundle (fallback if asset file doesn't exist - Vite builds)
        $asset_file_path = PRORANK_PLUGIN_DIR . 'build/admin.asset.php';
        if (file_exists($asset_file_path)) {
            $asset_file = include $asset_file_path;
            $dependencies = $asset_file['dependencies'] ?? [];
            $version = $asset_file['version'] ?? self::VERSION;
        } else {
            // Fallback for Vite builds - no asset.php file
            $dependencies = ['wp-element', 'wp-components', 'wp-api-fetch', 'wp-i18n'];
            $version = self::VERSION;
        }

        // Admin script loading is handled by AdminScripts.php which properly
        // sets up ES modules with import maps. This prevents conflicts.
        // PropTypes and Recharts are bundled by Vite, no separate vendor files needed.

        // Only add localized data if AdminScripts has enqueued the main script
        if (wp_script_is('prorank-admin-js', 'enqueued')) {
            wp_localize_script(
                'prorank-admin-js',
                'prorankSeoAdmin',
                $this->get_admin_localized_data()
            );
        }
        
        // Enqueue admin styles (Vite builds use index/main, webpack builds use admin.css).
        $main_css = PRORANK_PLUGIN_DIR . 'build/main.css';
        $index_css = PRORANK_PLUGIN_DIR . 'build/index.css';
        $admin_css = PRORANK_PLUGIN_DIR . 'build/admin.css';
        $app_css = PRORANK_PLUGIN_DIR . 'build/App.css';
        $enqueued_main = false;

        if (file_exists($main_css)) {
            wp_enqueue_style(
                'prorank-seo-admin-main',
                PRORANK_PLUGIN_URL . 'build/main.css',
                ['wp-components'],
                filemtime($main_css)
            );
            $enqueued_main = true;
        } elseif (file_exists($index_css)) {
            wp_enqueue_style(
                'prorank-seo-admin-main',
                PRORANK_PLUGIN_URL . 'build/index.css',
                ['wp-components'],
                filemtime($index_css)
            );
            $enqueued_main = true;
        } elseif (file_exists($admin_css)) {
            wp_enqueue_style(
                'prorank-seo-admin-main',
                PRORANK_PLUGIN_URL . 'build/admin.css',
                ['wp-components'],
                filemtime($admin_css)
            );
            $enqueued_main = true;
        } else {
            // Fallback for older builds.
            $legacy_css = PRORANK_PLUGIN_DIR . 'admin/assets/js/App.css';
            if (file_exists($legacy_css)) {
                wp_enqueue_style(
                    'prorank-seo-admin-main',
                    PRORANK_PLUGIN_URL . 'admin/assets/js/App.css',
                    ['wp-components'],
                    filemtime($legacy_css)
                );
                $enqueued_main = true;
            }
        }

        if ($enqueued_main && file_exists($app_css)) {
            wp_enqueue_style(
                'prorank-seo-admin-app',
                PRORANK_PLUGIN_URL . 'build/App.css',
                ['prorank-seo-admin-main'],
                filemtime($app_css)
            );
        }
        
        // Enqueue admin menu fix styles
        $menu_fix_css = PRORANK_PLUGIN_DIR . 'admin/assets/css/admin-menu-fix.css';
        if (file_exists($menu_fix_css)) {
            wp_enqueue_style(
                'prorank-admin-menu-fix',
                PRORANK_PLUGIN_URL . 'admin/assets/css/admin-menu-fix.css',
                ['prorank-seo-admin'],
                PRORANK_VERSION
            );
        }
    }
    
    /**
     * Get build information
     *
     * @return array
     */
    private function get_build_info(): array {
        $build_file = PRORANK_PLUGIN_DIR . 'build-info.json';
        $default_info = [
            'fingerprint' => 'dev-' . substr(md5(self::VERSION), 0, 8),
            'timestamp' => time(),
            'commit' => 'development',
        ];
        
        if (file_exists($build_file)) {
            $build_data = json_decode(prorank_read_file($build_file), true);
            if ($build_data) {
                return array_merge($default_info, $build_data);
            }
        }
        
        return $default_info;
    }
    
    /**
     * Get modules with availability status.
     *
     * Free plugin only exposes loaded modules without locked placeholders.
     *
     * @return array<string, array<string, mixed>>
     */
    private function get_filtered_modules(): array {
        $loaded_modules = $this->modules()->get_modules();
        $result = [];

        foreach ($loaded_modules as $slug => $module) {
            if (!is_object($module)) {
                continue;
            }

            $name = method_exists($module, 'get_name') ? $module->get_name() : ucfirst(str_replace('-', ' ', (string) $slug));
            $description = method_exists($module, 'get_description') ? $module->get_description() : '';
            $parent_slug = method_exists($module, 'get_parent_slug') ? $module->get_parent_slug() : null;
            $enabled = method_exists($module, 'is_enabled') ? (bool) $module->is_enabled() : true;

            $result[(string) $slug] = [
                'id' => (string) $slug,
                'name' => $name,
                'description' => $description,
                'tier' => 'free',
                'parent_slug' => $parent_slug,
                'enabled' => $enabled,
                'locked' => false,
                'available' => true,
                'loaded' => true,
            ];
        }

        return $result;
    }



    private function get_admin_localized_data(): array {
        $current_user = wp_get_current_user();
        $server_software = '';
        if ( isset( $_SERVER['SERVER_SOFTWARE'] ) ) {
            $server_software = sanitize_text_field( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ) );
        }
        $litespeed_server = $server_software !== '' && stripos( $server_software, 'litespeed' ) !== false;
        $litespeed_cache = defined( 'LSCWP_V' ) || class_exists( 'LiteSpeed_Cache_API' );
        
        // Get build information
        $build_info = $this->get_build_info();
        
        return [
            'apiUrl' => home_url( '/wp-json/prorank-seo/v1' ),
            'nonce' => wp_create_nonce( 'wp_rest' ),
            'adminUrl' => admin_url(),
            'siteUrl' => home_url(),
            'pluginUrl' => PRORANK_PLUGIN_URL,
            'version' => self::VERSION,
            'buildFingerprint' => $build_info['fingerprint'],
            'buildTimestamp' => $build_info['timestamp'],
            'buildCommit' => $build_info['commit'],
            'isMultisite' => is_multisite(),
            'currentUser' => [
                'id' => $current_user->ID,
                'name' => $current_user->display_name,
                'email' => $current_user->user_email,
                'avatar' => get_avatar_url( $current_user->ID ),
                'capabilities' => $current_user->allcaps,
            ],
            'settings' => $this->settings()->get_all(),
            'modules' => $this->get_filtered_modules(),
            'notifications' => $this->notifications()->get_notifications(),
            'competitors' => $this->get_competitor_plugins(),
            'urls' => DomainConfig::getAllUrls(),
            'server' => [
                'litespeed' => [
                    'server' => $litespeed_server,
                    'cache_plugin' => $litespeed_cache,
                    'supported' => $litespeed_server || $litespeed_cache,
                ],
            ],
        ];
    }
    
    /**
     * Check if current page is a ProRank admin page
     *
     * @param string $hook_suffix Hook suffix
     * @return bool
     */
    private function is_prorank_admin_page( string $hook_suffix ): bool {
        // Check if it's the top-level page or any submenu page
        if ( strpos( $hook_suffix, 'prorank-seo' ) !== false ) {
            return true;
        }
        
        // Also check the specific hook formats WordPress uses
        $prorank_pages = [
            'toplevel_page_prorank-seo',
            'prorank-seo_page_prorank-on-page-seo',
            'prorank-seo_page_prorank-performance',
            'prorank-seo_page_prorank-technical-seo',
            'prorank-seo_page_prorank-site-audit',
            'prorank-seo_page_prorank-support',
            'prorank-seo_page_prorank-seo-settings',
        ];
        
        if ( in_array( $hook_suffix, $prorank_pages, true ) ) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets(): void {
        // Frontend assets will be enqueued by individual modules as needed
    }
    
    /**
     * Check for plugin updates
     */
    private function check_updates(): void {
        // Update check logic here
    }
    
    /**
     * Initialize onboarding
     */
    private function init_onboarding(): void {
        // Check if onboarding is needed
        $onboarding_completed = get_option( 'prorank_onboarding_completed', false );
        
        if ( ! $onboarding_completed && current_user_can( 'manage_options' ) ) {
            // Show onboarding notice
            // $this->admin_notices->add(
            //     'onboarding',
            //     sprintf(
            //         __( 'Welcome to ProRank SEO! <a href="%s">Click here to complete the setup wizard</a>.', 'prorank-seo' ),
            //         admin_url( 'admin.php?page=prorank-seo&onboarding=1' )
            //     ),
            //     'info',
            //     false
            // );
        }
    }
    
    /**
     * Get service by name
     *
     * @param string $name Service name
     * @return mixed
     */
    public function get( string $name ) {
        return $this->services->get( $name );
    }
    
    /**
     * Get settings manager
     *
     * @return SettingsManager
     */
    public function settings(): SettingsManager {
        return $this->services->get('settings');
    }
    
    /**
     * Get notification manager
     *
     * @return NotificationManager
     */
    public function notifications(): NotificationManager {
        return $this->services->get('notifications');
    }

    /**
     * Get license manager
     *
     * @return LicenseManager
     */
    public function license(): LicenseManager {
        return LicenseManager::get_instance();
    }
    
    /**
     * Get module manager
     *
     * @return ModuleManager
     */
    public function modules(): ModuleManager {
        return $this->services->get('modules');
    }
    
    /**
     * Get menu registry
     *
     * @return \ProRank\SEO\Core\MenuRegistry
     */
    public function menus(): \ProRank\SEO\Core\MenuRegistry {
        return $this->services->get('menus');
    }
    
    /**
     * Check if plugin is initialized
     *
     * @return bool
     */
    public function is_initialized(): bool {
        return !empty($this->services);
    }

    /**
     * Load Elementor integration if Elementor is active
     */
    private function load_elementor_integration(): void {
        // Only load if Elementor is active
        if (!defined('ELEMENTOR_VERSION')) {
            return;
        }

        // Load the integration class
        $integration_file = PRORANK_PLUGIN_DIR . 'includes/Integrations/Elementor/ElementorIntegration.php';
        if (file_exists($integration_file)) {
            require_once $integration_file;
            \ProRank\SEO\Integrations\Elementor\ElementorIntegration::get_instance();
        }
    }

    private function check_competitors(): void {
        // Only check in admin
        if (!is_admin()) {
            return;
        }

        $detector = new \ProRank\SEO\Core\Compatibility\CompetitorDetector();
        $competitors = $detector->get_active_competitors_details();
        
        if (!empty($competitors)) {
            // Add a dismissible admin notice with deactivation links
            add_action('admin_notices', function() use ($competitors) {
                ?>
                <div class="notice notice-warning is-dismissible prorank-competitor-notice">
                    <p><strong><?php echo esc_html__('ProRank SEO Conflict Detection:', 'prorank-seo'); ?></strong></p>
                    <p><?php echo esc_html__('We\'ve detected other SEO plugins that may conflict with ProRank SEO. Running multiple SEO plugins can cause issues with your site\'s performance and search rankings.', 'prorank-seo'); ?></p>
                    
                    <div style="margin: 15px 0;">
                        <?php foreach ($competitors as $competitor): ?>
                            <div style="margin: 10px 0; padding: 10px; background: #fff; border-left: 4px solid #ff9800; display: flex; align-items: center; justify-content: space-between;">
                                <div>
                                    <strong><?php echo esc_html($competitor['name']); ?></strong>
                                    <span style="color: #666; margin-left: 10px;"><?php echo esc_html__('is currently active', 'prorank-seo'); ?></span>
                                </div>
                                <div>
                                    <a href="<?php echo esc_url($competitor['deactivate_url']); ?>" 
                                       class="button button-primary"
                                       onclick="return confirm('<?php echo esc_js(sprintf(
                                           /* translators: %s: placeholder value */
                                           __('Are you sure you want to deactivate %s?', 'prorank-seo'), $competitor['name'])); ?>');">
                                        <?php echo esc_html__('Deactivate', 'prorank-seo'); ?>
                                    </a>
                                    <a href="<?php echo esc_url(admin_url('plugins.php')); ?>" 
                                       class="button" 
                                       style="margin-left: 5px;">
                                        <?php echo esc_html__('Manage Plugins', 'prorank-seo'); ?>
                                    </a>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <p style="margin-top: 15px;">
                        <strong><?php echo esc_html__('Recommendation:', 'prorank-seo'); ?></strong> 
                        <?php echo esc_html__('We recommend deactivating other SEO plugins before using ProRank SEO to avoid conflicts and ensure optimal performance.', 'prorank-seo'); ?>
                    </p>
                    
                    <p>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=prorank-seo&tab=import-export')); ?>" class="button button-secondary">
                            <?php echo esc_html__('Import Settings from Other SEO Plugins', 'prorank-seo'); ?>
                        </a>
                    </p>
                </div>
                <?php
            });
        }
    }
    
    /**
     * Get competitor plugins
     *
     * @return array
     */
    private function get_competitor_plugins(): array {
        $detector = new \ProRank\SEO\Core\Compatibility\CompetitorDetector();
        return $detector->get_active_competitors();
    }
    
    /**
     * Get service container
     *
     * @return ServiceContainer
     */
    public function get_container(): ServiceContainer {
        return $this->services;
    }
    
    /**
     * Initialize frontend head output
     */
    public function init_frontend_head_output(): void {
        // Only initialize on frontend requests
        if (!is_admin() && !wp_doing_ajax() && !wp_doing_cron()) {
            try {
                $head_output = new HeadOutput($this);
                $head_output->init();
                new SiteBasicsOutput();
            } catch (\Exception $e) {
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log('[ProRank SEO] Failed to initialize HeadOutput: ' . $e->getMessage());
                }
            }
        }
    }

    /**
     * Initialize virtual root text files.
     */
    public function init_virtual_txt_files(): void {
        if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
            return;
        }

        try {
            $virtual = new VirtualTxtFiles();
            $virtual->init();
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Failed to initialize VirtualTxtFiles: ' . $e->getMessage());
            }
        }
    }

    /**
     * Initialize speculation rules (safe prefetch/prerender)
     */
    public function init_speculation_rules(): void {
        if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
            return;
        }
        try {
            $speculation = new SpeculationRules();
            $speculation->init();
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Failed to initialize SpeculationRules: ' . $e->getMessage());
            }
        }
    }

    /**
     * Initialize third-party defer for INP improvements
     */
    public function init_third_party_defer(): void {
        if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
            return;
        }
        try {
            $defer = new ThirdPartyDefer();
            $defer->init();
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Failed to initialize ThirdPartyDefer: ' . $e->getMessage());
            }
        }
    }

    /**
     * Initialize lazy media enhancer
     */
    public function init_lazy_media(): void {
        if (is_admin()) {
            return;
        }
        try {
            $lazy = new LazyMedia();
            $lazy->init();
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Failed to initialize LazyMedia: ' . $e->getMessage());
            }
        }
    }

    /**
     * Initialize RUM beacon (INP/CLS/LCP)
     */
    public function init_rum_beacon(): void {
        if (is_admin()) {
            return;
        }
        if ( function_exists( 'prorank_is_rum_enabled' ) && ! prorank_is_rum_enabled() ) {
            return;
        }
        try {
            $rum = new RumBeacon();
            $rum->init();
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Failed to initialize RumBeacon: ' . $e->getMessage());
            }
        }
    }

    /**
     * Cron: generate critical CSS (background)
     */
    public function run_critical_css_cron( $target = 'important' ): void {
        // Handle array args from wp_schedule_single_event
        if ( is_array( $target ) && isset( $target['target'] ) ) {
            $target = $target['target'];
        }
        try {
            // Avoid running the legacy target-based generator when a task UUID is passed
            // (CriticalCssGeneratorModule schedules task IDs as the first argument).
            if (is_string($target) && preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $target)) {
                return;
            }

            $generator = new \ProRank\SEO\Modules\Performance\CriticalCssGenerator();
            $generator->generate(is_string($target) ? $target : 'important');
        } catch (\Throwable $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Critical CSS cron failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Cron: scan unused CSS (background)
     */
    public function run_unused_css_cron( $target = 'all' ): void {
        try {
            $scanner = new \ProRank\SEO\Modules\Performance\UnusedCssScanner();
            $safelist = get_option('prorank_unused_css_safelist', '');
            $scanner->scan(is_string($target) ? $target : 'all', $safelist, true);
        } catch (\Throwable $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Unused CSS cron failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Ensure RUM guard is scheduled
     */
    public function ensure_rum_guard_schedule(): void {
        if ( ! wp_next_scheduled( 'prorank_rum_guard' ) ) {
            wp_schedule_event( time() + HOUR_IN_SECONDS, 'daily', 'prorank_rum_guard' );
        }
    }

    /**
     * RUM guard: evaluate INP/CLS/LCP samples and roll back critical CSS if degraded
     */
    public function run_rum_guard(): void {
        $buffer = get_option( 'prorank_rum_buffer', [] );
        if ( ! is_array( $buffer ) || empty( $buffer ) ) {
            return;
        }
        $sum_inp = 0; $sum_cls = 0; $sum_lcp = 0; $count = 0;
        foreach ( $buffer as $sample ) {
            $sum_inp += isset( $sample['inp'] ) ? (int) $sample['inp'] : 0;
            $sum_cls += isset( $sample['cls'] ) ? (float) $sample['cls'] : 0;
            $sum_lcp += isset( $sample['lcp'] ) ? (int) $sample['lcp'] : 0;
            $count++;
        }
        if ( $count === 0 ) {
            return;
        }
        $avg_inp = $sum_inp / $count;
        $avg_cls = $sum_cls / $count;
        $avg_lcp = $sum_lcp / $count;

        $inp_bad = $avg_inp > 400;
        $cls_bad = $avg_cls > 0.25;
        $lcp_bad = $avg_lcp > 3500;

        if ( $inp_bad || $cls_bad || $lcp_bad ) {
            try {
                $generator = new \ProRank\SEO\Modules\Performance\CriticalCssGenerator();
                $rolled = $generator->rollback();
                if ( $rolled ) {
                    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                        prorank_log( '[ProRank SEO] RUM guard rolled back critical CSS due to degraded vitals (INP:' . $avg_inp . ' CLS:' . $avg_cls . ' LCP:' . $avg_lcp . ')' );
                    }
                }
            } catch ( \Throwable $e ) {
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log( '[ProRank SEO] RUM guard rollback failed: ' . $e->getMessage() );
                }
            }
        }
        // Clear buffer after evaluation
        update_option( 'prorank_rum_buffer', [] );
    }
    
    /**
     * Initialize REST API
     */
    public function init_rest_api(): void {
        // Initialize REST API controllers with correct dependencies
        $controllers = [];
        
        if (class_exists('ProRank\SEO\Core\RestApi\DashboardController')) {
            $controllers[] = new DashboardController($this->modules());
        }
        
        if (class_exists('ProRank\SEO\Core\RestApi\ModulesController')) {
            $modules_controller = new ModulesController();
            $modules_controller->set_module_manager($this->modules());
            $controllers[] = $modules_controller;
        }

        if (class_exists('ProRank\SEO\Core\RestApi\WizardStateController')) {
            $controllers[] = new WizardStateController();
        }
        
        if (class_exists('ProRank\SEO\Core\RestApi\SettingsController')) {
            $controllers[] = new SettingsController($this->settings());
        }
        
        if (class_exists('ProRank\SEO\Core\RestApi\CacheController')) {
            $controllers[] = new CacheController();
        }
        
        if (class_exists('ProRank\SEO\Core\RestApi\AdvancedSettingsController')) {
            $controllers[] = new AdvancedSettingsController();
        }
        
        if (class_exists('ProRank\SEO\Core\RestApi\DataController')) {
            $controllers[] = new DataController();
        }
        
        if (class_exists('ProRank\SEO\Core\RestApi\SystemController')) {
            $controllers[] = new SystemController();
        }
        
        if (class_exists('ProRank\SEO\Core\RestApi\EnhancedAuditController')) {
            try {
                $controller = new EnhancedAuditController();
                $controllers[] = $controller;
            } catch (\Throwable $e) {
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log('[ProRank SEO] EnhancedAuditController instantiation failed: ' . $e->getMessage());
                }
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    prorank_log($e->getTraceAsString());
                }
            }
        }

        // SiteAuditController removed - only EnhancedAuditController handles audit routes
        // if (class_exists('ProRank\SEO\Core\RestApi\SiteAuditController')) {
        //     $controllers[] = new SiteAuditController();
        // }

        if (class_exists('ProRank\SEO\Core\RestApi\AuditSettingsController')) {
            $controllers[] = new \ProRank\SEO\Core\RestApi\AuditSettingsController();
        }

        // AuditExportController is disabled in free to avoid external service dependencies.

        // SEO Feature controllers
        if (class_exists('ProRank\SEO\Core\RestApi\MetaController')) {
            $controllers[] = new MetaController();
        }

        if (class_exists('ProRank\SEO\Core\RestApi\SchemaPreviewController')) {
            $controllers[] = new \ProRank\SEO\Core\RestApi\SchemaPreviewController();
        }

        if (class_exists('ProRank\SEO\Core\RestApi\TitlesMetaBulkController')) {
            $controllers[] = new TitlesMetaBulkController();
        }
        
        if (class_exists('ProRank\SEO\Core\RestApi\OnboardingController')) {
            $controllers[] = new OnboardingController();
        }
        
        // Sitemap controllers
        if (class_exists('ProRank\SEO\Core\RestApi\SitemapsController')) {
            $controllers[] = new SitemapsController();
        }
        
        // Content controllers
        if (class_exists('ProRank\SEO\Core\RestApi\ReadabilityController')) {
            $controllers[] = new ReadabilityController();
        }

        if (class_exists('ProRank\SEO\Core\RestApi\HeadlineAnalyzerController')) {
            $controllers[] = new HeadlineAnalyzerController();
        }

        // Some controllers are shipped in the Free plugin so UI components can call
        // their routes without getting 404s. Access control is enforced inside the
        // controllers (Premium plugin + license tier).
        if (class_exists('ProRank\SEO\Core\RestApi\LinkingController')) {
            $controllers[] = new LinkingController();
        }
        
        // Technical SEO controllers
        if (class_exists('ProRank\SEO\Core\RestApi\RedirectsController')) {
            $controllers[] = new RedirectsController();
        }
        
        // Import/Export controller
        if (class_exists('ProRank\SEO\Core\RestApi\ImportExportController')) {
            $controllers[] = new ImportExportController($this->modules());
        }
        
        // Command Palette controller
        if (class_exists('ProRank\SEO\Core\RestApi\CommandPaletteController')) {
            $controllers[] = new CommandPaletteController($this->services->get('menus'));
        }
        
        // User Preferences controller
        if (class_exists('ProRank\SEO\Core\RestApi\UserPreferencesController')) {
            $controllers[] = new UserPreferencesController();
        }
        
        // Presets controller
        if (class_exists('ProRank\SEO\Core\RestApi\PresetsController')) {
            $controllers[] = new PresetsController($this->services->get('presets'));
        }
        
        // Settings Snapshot controller
        if (class_exists('ProRank\SEO\Core\RestApi\SettingsSnapshotController')) {
            $controllers[] = new SettingsSnapshotController($this->services->get('snapshots'));
        }

        if (class_exists('ProRank\SEO\Core\RestApi\FeedbackController')) {
            $controllers[] = new \ProRank\SEO\Core\RestApi\FeedbackController();
        }

        // Image controller
        if (class_exists('ProRank\SEO\Core\RestApi\ImageOptimizationController')) {
            $controllers[] = new ImageOptimizationController();
        }
        
        if (class_exists('ProRank\SEO\Core\RestApi\ImagesController')) {
            $controllers[] = new ImagesController();
        }
        
        // Font Optimization controller
        if (class_exists('ProRank\SEO\Core\RestApi\FontOptimizationController')) {
            $controllers[] = new FontOptimizationController();
        }

        // Performance Settings controller (Critical CSS, Unused CSS, etc.)
        if (class_exists('ProRank\SEO\Core\RestApi\PerformanceSettingsController')) {
            $controllers[] = new PerformanceSettingsController();
        }

        // Browser Cache endpoint
        if (class_exists('ProRank\SEO\Api\Endpoints\BrowserCacheEndpoint')) {
            $controllers[] = new \ProRank\SEO\Api\Endpoints\BrowserCacheEndpoint();
        }
        
        // Register all routes
        foreach ( $controllers as $controller ) {
            if (method_exists($controller, 'register_routes')) {
                $controller->register_routes();
            }
        }
    }
    
}
