<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries throughout
/**
 * Advanced Site Audit Module
 *
 * Comprehensive site audit engine with crawler and multi-point analysis
 *
 * @package ProRank\SEO\Modules\Tools
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\Tools;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Modules\BaseModule;
use ProRank\SEO\Core\Audits\SiteAuditEngine;
use WP_Error;

/**
 * SiteAuditModule class
 */
class SiteAuditModule extends BaseModule {
    
    /**
     * Module slug
     *
     * @var string
     */
    protected string $slug = 'site-audit';
    
    /**
     * Module name
     *
     * @var string
     */
    protected string $name = 'Comprehensive Site Audit';
    
    /**
     * Module description
     *
     * @var string
     */
    protected string $description = 'Complete site health check with crawler and multi-point analysis';
    
    /**
     * Required tier
     *
     * @var string
     */
    protected string $feature_tier = 'free';
    
    /**
     * Parent module
     *
     * @var string|null
     */
    protected ?string $parent_slug = null;
    
    /**
     * Site audit engine
     *
     * @var SiteAuditEngine|null
     */
    private ?SiteAuditEngine $engine = null;
    
    /**
     * Action Scheduler actions
     */
    private const ACTION_START_AUDIT = 'prorank_audit_start';
    private const ACTION_CRAWL_BATCH = 'prorank_audit_crawl';
    private const ACTION_CHECK_URL = 'prorank_audit_check_url';
    private const ACTION_COMPLETE_AUDIT = 'prorank_audit_complete';
    
    /**
     * Audit states
     */
    private const STATE_IDLE = 'idle';
    private const STATE_CRAWLING = 'crawling';
    private const STATE_CHECKING = 'checking';
    private const STATE_COMPLETED = 'completed';
    private const STATE_FAILED = 'failed';
    
    /**
     * Check types
     */
    private const CHECK_TYPES = [
        'meta_tags',
        'schema_validation',
        'broken_links',
        'accessibility',
        'https_status',
        'page_speed',
        'mobile_friendly',
        'content_quality',
        'duplicate_content',
        'core_web_vitals',
        'image_optimization',
        'structured_data',
        'redirect_chains',
        'robots_meta',
        'canonical_tags',
        'hreflang_tags',
        'open_graph',
        'twitter_cards',
        'heading_structure',
        'internal_links',
        'external_links',
        'xml_sitemap',
        'page_size',
        'compression',
        'caching_headers',
        'security_headers',
        'mixed_content',
        'viewport_meta',
        'lang_attribute',
        'readability',
        'keyword_optimization',
    ];
    
    /**
     * Check if module is standalone
     *
     * @return bool
     */
    public function is_standalone(): bool {
        return false;
    }
    
    /**
     * Initialize module hooks (required by ModuleInterface)
     *
     * @return void
     */
    public function init_hooks(): void {
        // Register REST routes
        add_action('rest_api_init', [$this, 'register_rest_routes']);

        // Custom cron schedules for audits
        add_filter('cron_schedules', [$this, 'register_cron_schedules']);
        
        // Register action scheduler hooks
        add_action(self::ACTION_CRAWL_BATCH, [$this, 'handle_crawl_batch']);
        add_action(self::ACTION_CHECK_URL, [$this, 'handle_check_url'], 10, 3);
        add_action(self::ACTION_COMPLETE_AUDIT, [$this, 'handle_complete_audit']);
        
        add_action('prorank_audit_scheduled_run', [$this, 'handle_scheduled_audit']);
        add_action('prorank_audit_scheduled_alert', [$this, 'maybe_send_alert_from_status']);

        // Reschedule audits when settings change
        add_action('prorank_audit_settings_updated', [$this, 'handle_audit_settings_update']);
        
        // Schedule recurring audit if enabled
        $this->maybe_schedule_recurring_audit();
    }

    /**
     * Register audit-specific cron schedules.
     */
    public function register_cron_schedules(array $schedules): array {
        if (!isset($schedules['weekly'])) {
            $schedules['weekly'] = [
                'interval' => WEEK_IN_SECONDS,
                'display' => __('Once Weekly', 'prorank-seo'),
            ];
        }

        if (!isset($schedules['biweekly'])) {
            $schedules['biweekly'] = [
                'interval' => 2 * WEEK_IN_SECONDS,
                'display' => __('Once Every Two Weeks', 'prorank-seo'),
            ];
        }

        if (!isset($schedules['monthly'])) {
            $schedules['monthly'] = [
                'interval' => MONTH_IN_SECONDS,
                'display' => __('Once Monthly', 'prorank-seo'),
            ];
        }

        return $schedules;
    }

    /**
     * Handle schedule refresh when audit settings change.
     */
    public function handle_audit_settings_update(): void {
        $this->maybe_schedule_recurring_audit();
    }
    
    /**
     * Initialize module
     *
     * @return void
     */
    public function init(): void {
        parent::init();
        
        // Initialize engine
        try {
            if (class_exists('\ProRank\SEO\Core\Audits\SiteAuditEngine')) {
                $this->engine = new SiteAuditEngine();
            }
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Failed to initialize SiteAuditEngine: ' . $e->getMessage());
            }
        }
        
        // Create database tables
        $this->create_tables();
    }
    
    /**
     * Register REST routes
     *
     * @return void
     */
    public function register_rest_routes(): void {
        // Use the lightweight local audit REST controller in the free plugin.
        if ( class_exists( '\ProRank\SEO\Core\RestApi\EnhancedAuditController' ) ) {
            return;
        }

        $namespace = 'prorank-seo/v1';
        
        // Test endpoint
        register_rest_route($namespace, '/audit/test', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_test'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);
        
        // Status endpoint
        register_rest_route($namespace, '/audit/status', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_get_status'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);
        
        // Start audit endpoint
        register_rest_route($namespace, '/audit/start', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_start_audit'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);
        
        // Stop audit endpoint
        register_rest_route($namespace, '/audit/stop', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_stop_audit'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);
        
        register_rest_route($namespace, '/audit/reset', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_reset_audit'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);
        
        // Issues endpoint
        register_rest_route($namespace, '/audit/issues', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_get_issues'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);
        
        // History endpoint
        register_rest_route($namespace, '/audit/history', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_get_history'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);
        
        // Settings endpoints
        register_rest_route($namespace, '/audit/settings', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_get_settings'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);
        
        register_rest_route($namespace, '/audit/settings', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_save_settings'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);
        
        // Auto-fix endpoint
        register_rest_route($namespace, '/audit/autofix', [
            'methods' => 'POST',
            'callback' => [$this, 'rest_autofix_issue'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);
        
        // Quick fixes endpoint
        register_rest_route($namespace, '/audit/quick-fixes', [
            'methods' => 'GET',
            'callback' => [$this, 'rest_get_quick_fixes'],
            'permission_callback' => [$this, 'rest_permission_check'],
        ]);
    }
    
    /**
     * REST permission check
     *
     * @return bool
     */
    public function rest_permission_check(): bool {
        return current_user_can('manage_options');
    }
    
    /**
     * Test endpoint
     *
     * @return array
     */
    public function rest_test(): array {
        return [
            'success' => true,
            'message' => 'Site Audit API is working',
            'version' => '3.0',
            'engine_loaded' => $this->engine !== null,
            'tables_exist' => $this->check_tables_exist(),
            'check_types' => self::CHECK_TYPES,
        ];
    }
    
    /**
     * Get audit status
     *
     * @return array|\WP_Error
     */
    public function rest_get_status() {
        try {
            // FIRST check for external audits (new system)
            global $wpdb;
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $external_audit_id = $wpdb->get_var(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                $wpdb->prepare(
                    "SELECT option_value FROM {$wpdb->options} WHERE option_name = %s LIMIT 1",
                    'prorank_external_audit_id'
                )
            );

            if ($external_audit_id && class_exists('\ProRank\SEO\Core\ExternalAuditClient')) {
                // External audit is running - delegate to ExternalAuditClient (premium only)
                $external_client = new \ProRank\SEO\Core\ExternalAuditClient();
                $external_status = $external_client->get_current_audit_status();

                if ($external_status) {
                    return $external_status;
                }
            }

            // Fallback to local audit system (old system)
            // Ensure tables exist
            $this->create_tables();

            $state = get_option('prorank_audit_state', self::STATE_IDLE);
            $current_audit_id = get_option('prorank_audit_current_id');

            if ($current_audit_id && in_array($state, [self::STATE_CRAWLING, self::STATE_CHECKING], true)) {
                $this->maybe_run_inline_processing($current_audit_id);
            }

            $response_state = $state;
            if ($state === self::STATE_FAILED) {
                $response_state = 'error';
            }

            $response = [
                'state'       => $response_state,
                'audit_id'    => $current_audit_id,
                'can_start'   => $state === self::STATE_IDLE,
                'progress'    => [
                    'percent'      => 0,
                    'checked_urls' => 0,
                    'total_urls'   => 0,
                ],
                'stats'       => [
                    'urls_found'    => 0,
                    'urls_checked'  => 0,
                    'issues_found'  => 0,
                    'warnings_found'=> 0,
                    'passed_checks' => 0,
                ],
                'categories'  => $this->get_empty_categories(),
            ];

            $resolved_audit_id = $current_audit_id;

            if (!$resolved_audit_id) {
                $table_audits = $wpdb->prefix . 'prorank_audits';
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                $resolved_audit_id = $wpdb->get_var(
                    "SELECT id FROM {$table_audits} ORDER BY end_time DESC, start_time DESC LIMIT 1"
                );
            }

            if ($resolved_audit_id) {
                global $wpdb;
                $table_audits = $wpdb->prefix . 'prorank_audits';
                $table_urls = $wpdb->prefix . 'prorank_audit_urls';
                $table_issues = $wpdb->prefix . 'prorank_audit_issues';
                
                // Get audit stats
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                $audit = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM {$table_audits} WHERE id = %s",
                    $resolved_audit_id
                ));
                
                if ($audit) {
                    $current_audit_id = $resolved_audit_id;
                    $state = $audit->status ?: $state;
                    $response_state = $state === self::STATE_FAILED ? 'error' : $state;
                    $response['state'] = $response_state;
                    $response['audit_id'] = $current_audit_id;
                    $response['can_start'] = !in_array($response_state, [self::STATE_CRAWLING, self::STATE_CHECKING], true);

                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                    $urls_found = (int) $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(*) FROM {$table_urls} WHERE audit_id = %s",
                        $current_audit_id
                    ));
                    
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                    $urls_checked = (int) $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(*) FROM {$table_urls} WHERE audit_id = %s AND status IN ('checked','completed')",
                        $current_audit_id
                    ));
                    
                    // Prevent negative remaining/over 100% when counts drift
                    if ($urls_checked > $urls_found) {
                        $urls_found = $urls_checked;
                    }
                    
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                    $issues_found = (int) $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity IN ('critical', 'high', 'medium', 'low')",
                        $current_audit_id
                    ));
                    
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                    $warnings_found = (int) $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity = 'warning'",
                        $current_audit_id
                    ));
                    
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                    $passed_checks = (int) $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity = 'passed'",
                        $current_audit_id
                    ));
                    
                    $checked = min($urls_checked, $urls_found);
                    $progress_percent = $urls_found > 0 ? min(100, round(($checked / max(1, $urls_found)) * 100)) : 0;
                    $response['progress'] = [
                        'percent'      => $progress_percent,
                        'checked_urls' => $urls_checked,
                        'total_urls'   => $urls_found,
                    ];
                    $response['stats']      = [
                        'urls_found'    => $urls_found,
                        'urls_checked'  => $urls_checked,
                        'issues_found'  => $issues_found,
                        'warnings_found'=> $warnings_found,
                        'passed_checks' => $passed_checks,
                    ];
                    $response['score']      = $audit->score ?? 0;
                    $response['start_time'] = $audit->start_time;
                    if (!empty($audit->end_time)) {
                        $response['completed_at'] = $audit->end_time;
                    }
                    $response['categories'] = $this->get_category_stats($current_audit_id);

                    update_option('prorank_audit_last_progress', [
                        'checked' => $urls_checked,
                        'time' => time(),
                    ], false);
                }
            }
            
            return $response;
            
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Error in rest_get_status: ' . $e->getMessage());
            }
            return new \WP_Error('audit_error', esc_html($e->getMessage()));
        }
    }

    /**
     * Start audit
     *
     * @return array|\WP_Error
     */
    public function rest_start_audit(\WP_REST_Request $request = null) {
        try {
            // Ensure tables exist
            $this->create_tables();
            
            $state = get_option('prorank_audit_state', self::STATE_IDLE);
            
            if ($state !== self::STATE_IDLE) {
                return new \WP_Error('audit_running', __('An audit is already running', 'prorank-seo'));
            }

            $payload = [];
            if ($request instanceof \WP_REST_Request) {
                $payload = $request->get_json_params();
            }

            if (!empty($payload['settings']) && is_array($payload['settings'])) {
                $settings = $payload['settings'];
                if (isset($settings['max_urls'])) {
                    update_option('prorank_audit_max_urls', max(10, (int) $settings['max_urls']));
                }
                if (isset($settings['max_depth'])) {
                    update_option('prorank_audit_max_depth', max(1, (int) $settings['max_depth']));
                }
                if (!empty($settings['check_types']) && is_array($settings['check_types'])) {
                    $valid_checks = array_flip(self::CHECK_TYPES);
                    $clean_checks = [];
                    foreach ($settings['check_types'] as $key => $value) {
                        if (isset($valid_checks[$key])) {
                            $clean_checks[$key] = (bool) $value;
                        }
                    }
                    update_option('prorank_audit_check_types', $clean_checks);
                }
            }
            
            // Create new audit
            global $wpdb;
            $table_audits = $wpdb->prefix . 'prorank_audits';

            // Get audit settings from saved options
            $audit_settings = [
                'max_urls' => (int) get_option('prorank_audit_max_urls', 500),
                'max_depth' => (int) get_option('prorank_audit_max_depth', 5),
            ];
            $check_types = get_option('prorank_audit_check_types', []);

            $audit_id = wp_generate_uuid4();
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->insert(
                $table_audits,
                [
                    'id' => $audit_id,
                    'status' => self::STATE_CRAWLING,
                    'start_time' => current_time('mysql'),
                    'score' => 0,
                    'stats' => json_encode([
                        'urls_found' => 0,
                        'urls_checked' => 0,
                        'issues_found' => 0,
                        'warnings_found' => 0,
                        'passed_checks' => 0,
                    ]),
                    'options' => json_encode([
                        'settings' => $audit_settings,
                        'check_types' => $check_types,
                    ]),
                ],
                ['%s', '%s', '%s', '%d', '%s', '%s']
            );
            
            // Update options
            update_option('prorank_audit_state', self::STATE_CRAWLING);
            update_option('prorank_audit_current_id', $audit_id);
            update_option('prorank_audit_start_time', time());
            update_option('prorank_audit_last_progress', [
                'checked' => 0,
                'time' => time(),
            ], false);

            $inline_processing = (defined('DISABLE_WP_CRON') && DISABLE_WP_CRON) || !function_exists('as_enqueue_async_action');
            update_option('prorank_audit_inline_processing', $inline_processing, false);
            
            // Add initial URLs to crawl
            $this->add_initial_urls($audit_id);
            
            // Schedule crawl action
            if (function_exists('as_enqueue_async_action')) {
                as_enqueue_async_action(self::ACTION_CRAWL_BATCH, [$audit_id]);
            } else {
                // Fallback to direct call
                $this->handle_crawl_batch($audit_id);
            }

            if (function_exists('spawn_cron')) {
                spawn_cron();
            }

            if ($inline_processing) {
                $this->process_inline_batch($audit_id, 5);
            }
            
            return [
                'success' => true,
                'audit_id' => $audit_id,
                'message' => __('Audit started successfully', 'prorank-seo'),
            ];
            
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Error in rest_start_audit: ' . $e->getMessage());
            }
            return new \WP_Error('start_error', esc_html($e->getMessage()));
        }
    }

    /**
     * Stop audit
     *
     * @return array|\WP_Error
     */
    public function rest_stop_audit() {
        try {
            // Use the engine if available
            if ($this->engine) {
                $result = $this->engine->stop_audit();
                if ($result) {
                    return [
                        'success' => true,
                        'message' => __('Audit stopped successfully', 'prorank-seo'),
                    ];
                }
            }
            
            // Fallback implementation with better cleanup
            global $wpdb;
            $table_audits = $wpdb->prefix . 'prorank_audits';
            
            // Find any running audits
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $running_audits = $wpdb->get_results(
                "SELECT id, audit_id FROM {$table_audits} 
                WHERE status IN ('crawling', 'checking', 'running') 
                ORDER BY start_time DESC"
            );
            
            if (empty($running_audits)) {
                // Check if there's a stuck audit based on options
                $current_id = get_option('prorank_audit_current_id');
                if ($current_id) {
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                    $wpdb->update(
                        $table_audits,
                        [
                            'status' => 'stopped',
                            'end_time' => current_time('mysql'),
                        ],
                        ['id' => $current_id],
                        ['%s', '%s'],
                        ['%s']
                    );
                }
            } else {
                // Stop all running audits
                foreach ($running_audits as $audit) {
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                    $wpdb->update(
                        $table_audits,
                        [
                            'status' => 'stopped',
                            'end_time' => current_time('mysql'),
                        ],
                        ['id' => $audit->id],
                        ['%s', '%s'],
                        ['%s']
                    );
                    
                    // Clean up scheduled events for this audit
                    $this->clear_audit_scheduled_events($audit->audit_id ?? $audit->id);
                }
            }
            
            // Clear all audit-related cron events
            $this->clear_all_audit_crons();
            
            // Reset state options
            update_option('prorank_audit_state', self::STATE_IDLE);
            delete_option('prorank_audit_current_id');
            delete_option('prorank_audit_start_time');
            
            // Clear any transients
            delete_transient('prorank_audit_status');
            delete_transient('prorank_audit_progress');
            
            return [
                'success' => true,
                'message' => __('Audit stopped and cleaned up successfully', 'prorank-seo'),
            ];
            
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Error in rest_stop_audit: ' . $e->getMessage());
            }
            return new \WP_Error('stop_error', esc_html($e->getMessage()));
        }
    }

    /**
     * Reset audit - force clear all audit data and states
     *
     * @return array|\WP_Error
     */
    public function rest_reset_audit() {
        try {
            // Use the engine if available
            if ($this->engine) {
                $this->engine->clear_audit_data();
            }
            
            global $wpdb;
            $table_audits = $wpdb->prefix . 'prorank_audits';
            $table_urls = $wpdb->prefix . 'prorank_audit_urls';
            $table_issues = $wpdb->prefix . 'prorank_audit_issues';
            
            // Mark all running audits as stopped
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query(
                "UPDATE {$table_audits} 
                SET status = 'stopped', end_time = NOW() 
                WHERE status IN ('crawling', 'checking', 'running')"
            );
            
            // Clear all audit-related cron events
            $this->clear_all_audit_crons();
            
            // Reset all options
            update_option('prorank_audit_state', self::STATE_IDLE);
            delete_option('prorank_audit_current_id');
            delete_option('prorank_audit_start_time');
            
            // Clear all transients
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_%prorank_audit%'");
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_%prorank_audit%'");
            
            // Clear Action Scheduler tasks if available
            if (function_exists('as_unschedule_all_actions')) {
                $actions = [
                    self::ACTION_CRAWL_BATCH,
                    self::ACTION_CHECK_URL,
                    self::ACTION_COMPLETE_AUDIT,
                    'prorank_schedule_url_checks',
                    'prorank_check_audit_url',
                    'prorank_run_site_crawler',
                ];
                
                foreach ($actions as $action) {
                    as_unschedule_all_actions($action);
                }
            }
            
            return [
                'success' => true,
                'message' => __('Audit system has been reset successfully', 'prorank-seo'),
            ];
            
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Error in rest_reset_audit: ' . $e->getMessage());
            }
            return new \WP_Error('reset_error', esc_html($e->getMessage()));
        }
    }

    /**
     * Get audit issues
     *
     * @param \WP_REST_Request $request
     * @return array|\WP_Error
     */
    public function rest_get_issues($request) {
        try {
            // Ensure tables exist
            $this->create_tables();
            
            $audit_id = $request->get_param('audit_id') ?? get_option('prorank_audit_current_id');
            $severity = $request->get_param('severity') ?? 'all';
            $category = $request->get_param('category') ?? 'all';
            
            if (!$audit_id) {
                return [];
            }
            
            global $wpdb;
            $table_issues = $wpdb->prefix . 'prorank_audit_issues';
            
            $where_conditions = ["audit_id = %s"];
            $where_values = [$audit_id];
            
            if ($severity !== 'all') {
                $where_conditions[] = "severity = %s";
                $where_values[] = $severity;
            } else {
                // Exclude passed checks by default
                $where_conditions[] = "severity != 'passed'";
            }
            
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
            $sql = "SELECT * FROM {$table_issues} WHERE " . implode(' AND ', $where_conditions) . " ORDER BY 
                    CASE severity 
                        WHEN 'critical' THEN 1 
                        WHEN 'high' THEN 2 
                        WHEN 'medium' THEN 3 
                        WHEN 'low' THEN 4 
                        WHEN 'warning' THEN 5 
                        ELSE 6 
                    END, url ASC";
            
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
            $issues = $wpdb->get_results($wpdb->prepare($sql, ...$where_values));
            
            // Format issues
            $formatted = [];
            foreach ($issues as $issue) {
                $issue_category = $this->get_issue_category($issue->type);
                
                if ($category !== 'all' && $issue_category !== $category) {
                    continue;
                }
                
                $formatted[] = [
                    'id' => $issue->id,
                    'url' => $issue->url,
                    'type' => $issue->type,
                    'message' => $issue->message,
                    'severity' => $issue->severity,
                    'category' => $issue_category,
                    'can_autofix' => $this->can_autofix($issue->type),
                    'data' => json_decode($issue->data, true) ?? [],
                ];
            }
            
            return $formatted;
            
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Error in rest_get_issues: ' . $e->getMessage());
            }
            return new \WP_Error('issues_error', esc_html($e->getMessage()));
        }
    }

    /**
     * Get audit history
     *
     * @return array|\WP_Error
     */
    public function rest_get_history() {
        try {
            // Ensure tables exist
            $this->create_tables();
            
            global $wpdb;
            $table_audits = $wpdb->prefix . 'prorank_audits';
            
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $history = $wpdb->get_results(
                "SELECT * FROM {$table_audits} 
                ORDER BY start_time DESC 
                LIMIT 20"
            );
            
            // Format history with deltas vs previous run
            $formatted = [];
            $previous = null;
            foreach ($history as $audit) {
                $stats = json_decode($audit->stats, true) ?? [];
                $entry = [
                    'id' => $audit->id,
                    'status' => $audit->status,
                    'score' => (int) $audit->score,
                    'start_time' => $audit->start_time,
                    'end_time' => $audit->end_time,
                    'stats' => $stats,
                    'duration' => $audit->end_time ? 
                        human_time_diff(strtotime($audit->start_time), strtotime($audit->end_time)) : 
                        'In progress',
                ];

                if ($previous) {
                    $entry['delta_score'] = (int) $audit->score - (int) ($previous->score ?? 0);
                    $prev_stats = json_decode($previous->stats, true) ?? [];
                    $curr_issues = (int)($stats['total_issue_instances'] ?? 0);
                    $prev_issues = (int)($prev_stats['total_issue_instances'] ?? 0);
                    $entry['delta_issues'] = $curr_issues - $prev_issues;
                }

                $formatted[] = $entry;
                $previous = $audit;
            }
            
            return $formatted;
            
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Error in rest_get_history: ' . $e->getMessage());
            }
            return new \WP_Error('history_error', esc_html($e->getMessage()));
        }
    }

    /**
     * Get settings
     *
     * @return array
     */
    public function rest_get_settings(): array {
        $saved_checks = get_option('prorank_audit_check_types', []);
        $default_checks = [
            'meta_tags' => true,
            'broken_links' => true,
            'image_optimization' => true,
            'page_speed' => false,
            'mobile_friendly' => false,
            'https_status' => true,
            'schema_validation' => true,
            'content_quality' => true,
        ];
        
        return [
            'max_urls' => (int) get_option('prorank_audit_max_urls', 500),
            'max_depth' => (int) get_option('prorank_audit_max_depth', 5),
            'check_types' => array_merge($default_checks, $saved_checks),
            'schedule_enabled' => (bool) get_option('prorank_audit_schedule_enabled', false),
            'schedule_frequency' => get_option('prorank_audit_schedule_frequency', 'weekly'),
            'available_checks' => self::CHECK_TYPES,
        ];
    }
    
    /**
     * Save settings
     *
     * @param \WP_REST_Request $request
     * @return array
     */
    public function rest_save_settings($request): array {
        $settings = $request->get_json_params();
        
        if (isset($settings['max_urls'])) {
            update_option('prorank_audit_max_urls', (int) $settings['max_urls']);
        }
        
        if (isset($settings['max_depth'])) {
            update_option('prorank_audit_max_depth', (int) $settings['max_depth']);
        }
        
        if (isset($settings['check_types'])) {
            update_option('prorank_audit_check_types', $settings['check_types']);
        }
        
        if (isset($settings['schedule_enabled'])) {
            update_option('prorank_audit_schedule_enabled', (bool) $settings['schedule_enabled']);
        }
        
        if (isset($settings['schedule_frequency'])) {
            update_option('prorank_audit_schedule_frequency', $settings['schedule_frequency']);
        }

        // Reschedule recurring audit based on new settings
        $this->maybe_schedule_recurring_audit();
        
        return [
            'success' => true,
            'message' => __('Settings saved successfully', 'prorank-seo'),
        ];
    }
    
    /**
     * Auto-fix issue
     *
     * @param \WP_REST_Request $request
     * @return array|\WP_Error
     */
    public function rest_autofix_issue($request) {
        try {
            $issue_id = $request->get_param('issue_id');
            $issue_type = $request->get_param('issue_type');
            
            if (!$issue_id || !$issue_type) {
                return new \WP_Error('missing_params', __('Missing required parameters', 'prorank-seo'));
            }
            
            if (!$this->can_autofix($issue_type)) {
                return new \WP_Error('not_fixable', __('This issue cannot be auto-fixed', 'prorank-seo'));
            }
            
            $result = $this->apply_autofix($issue_type, $issue_id);
            
            if ($result) {
                return [
                    'success' => true,
                    'message' => __('Issue fixed successfully', 'prorank-seo'),
                ];
            } else {
                return new \WP_Error('fix_failed', __('Failed to fix issue', 'prorank-seo'));
            }
            
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Error in rest_autofix_issue: ' . $e->getMessage());
            }
            return new \WP_Error('autofix_error', esc_html($e->getMessage()));
        }
    }

    /**
     * Get quick fixes
     *
     * @return array|\WP_Error
     */
    public function rest_get_quick_fixes() {
        try {
            $audit_id = get_option('prorank_audit_current_id');
            
            if (!$audit_id) {
                return [];
            }
            
            global $wpdb;
            $table_issues = $wpdb->prefix . 'prorank_audit_issues';
            
            // Get fixable issues
            $fixable_types = ['no_compression', 'no_caching'];
            $placeholders = array_fill(0, count($fixable_types), '%s');
            
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $sql = $wpdb->prepare(
                "SELECT type, COUNT(*) as count, severity 
                FROM {$table_issues} 
                WHERE audit_id = %s 
                AND type IN (" . implode(',', $placeholders) . ")
                GROUP BY type, severity
                ORDER BY 
                    CASE severity 
                        WHEN 'critical' THEN 1 
                        WHEN 'high' THEN 2 
                        WHEN 'medium' THEN 3 
                        WHEN 'low' THEN 4 
                        ELSE 5 
                    END",
                array_merge([$audit_id], $fixable_types)
            );
            
            $issues = $wpdb->get_results($sql);
            
            $fixes = [];
            foreach ($issues as $issue) {
                $fixes[] = [
                    'type' => $issue->type,
                    'count' => (int) $issue->count,
                    'severity' => $issue->severity,
                    'title' => $this->get_fix_title($issue->type),
                    'description' => $this->get_fix_description($issue->type),
                    'impact' => $this->get_fix_impact($issue->type),
                ];
            }
            
            return $fixes;
            
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Error in rest_get_quick_fixes: ' . $e->getMessage());
            }
            return new \WP_Error('quick_fixes_error', esc_html($e->getMessage()));
        }
    }

    /**
     * Update settings (alias for rest_save_settings)
     *
     * @param \WP_REST_Request $request
     * @return array
     */
    public function rest_update_settings($request): array {
        return $this->rest_save_settings($request);
    }
    
    /**
     * Auto-fix endpoint
     *
     * @param \WP_REST_Request $request
     * @return array|\WP_Error
     */
    public function rest_auto_fix($request) {
        return $this->rest_autofix_issue($request);
    }
    
    /**
     * Handle crawl batch
     *
     * @param string $audit_id
     * @return void
     */
    public function handle_crawl_batch(string $audit_id): void {
        try {
            if ($this->engine) {
                $this->engine->crawl_batch($audit_id);
            } else {
                // Fallback implementation
                $this->simple_crawl_batch($audit_id);
            }
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Error in handle_crawl_batch: ' . $e->getMessage());
            }
            update_option('prorank_audit_state', self::STATE_FAILED);
        }
    }
    
    /**
     * Handle check URL
     *
     * @param string $audit_id
     * @param int $url_id
     * @param string $url
     * @return void
     */
    public function handle_check_url(string $audit_id, int $url_id, string $url): void {
        try {
            if ($this->engine) {
                $this->engine->check_url($audit_id, $url_id, $url);
            } else {
                // Fallback implementation
                $this->simple_check_url($audit_id, $url_id, $url);
            }
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Error in handle_check_url: ' . $e->getMessage());
            }
        }
    }
    
    /**
     * Handle complete audit
     *
     * @param string $audit_id
     * @return void
     */
    public function handle_complete_audit(string $audit_id): void {
        try {
            if ($this->engine) {
                $this->engine->complete_audit($audit_id);
            } else {
                // Fallback implementation
                $this->simple_complete_audit($audit_id);
            }
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Error in handle_complete_audit: ' . $e->getMessage());
            }
        }

        // Send alert summary
        $this->maybe_send_alert($audit_id);
    }
    
    /**
     * Simple crawl batch implementation
     *
     * @param string $audit_id
     * @return void
     */
    private function simple_crawl_batch(string $audit_id): void {
        global $wpdb;
        $table_urls = $wpdb->prefix . 'prorank_audit_urls';
        
        // Get pending URLs - increased batch size for faster processing
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $pending_urls = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table_urls} 
            WHERE audit_id = %s AND status = 'pending' 
            ORDER BY depth ASC 
            LIMIT 25",
            $audit_id
        ));
        
        if (empty($pending_urls)) {
            // No more URLs to crawl, transition to checking
            $this->transition_to_checking($audit_id);
            return;
        }
        
        foreach ($pending_urls as $url_row) {
            // Mark as crawled
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->update(
                $table_urls,
                ['status' => 'crawled'],
                ['id' => $url_row->id],
                ['%s'],
                ['%d']
            );
        }
        
        // Schedule next batch
        if (function_exists('as_enqueue_async_action')) {
            as_enqueue_async_action(self::ACTION_CRAWL_BATCH, [$audit_id]);
        }
    }

    /**
     * Run inline audit processing when background jobs are not running.
     *
     * @param string $audit_id
     * @return void
     */
    private function maybe_run_inline_processing(string $audit_id): void {
        $inline_processing = (bool) get_option('prorank_audit_inline_processing', false);

        if (!$inline_processing) {
            $last_progress = get_option('prorank_audit_last_progress', []);
            $last_checked = isset($last_progress['checked']) ? (int) $last_progress['checked'] : 0;
            $last_time = isset($last_progress['time']) ? (int) $last_progress['time'] : 0;

            if ($last_time > 0 && (time() - $last_time) > 30 && $last_checked === 0) {
                $inline_processing = true;
                update_option('prorank_audit_inline_processing', true, false);
            }
        }

        if (!$inline_processing) {
            return;
        }

        $this->process_inline_batch($audit_id, 5);
    }

    /**
     * Process a small batch of URLs inline to show progress immediately.
     *
     * @param string $audit_id
     * @param int $limit
     * @return void
     */
    private function process_inline_batch(string $audit_id, int $limit = 5): void {
        global $wpdb;
        $table_urls = $wpdb->prefix . 'prorank_audit_urls';

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $urls = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table_urls}
            WHERE audit_id = %s AND status IN ('pending', 'crawled')
            ORDER BY depth ASC
            LIMIT %d",
            $audit_id,
            $limit
        ));

        if (empty($urls)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $remaining = (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$table_urls}
                WHERE audit_id = %s AND status IN ('pending', 'crawled')",
                $audit_id
            ));

            if ($remaining === 0) {
                $this->simple_complete_audit($audit_id);
            }

            return;
        }

        update_option('prorank_audit_state', self::STATE_CHECKING);

        foreach ($urls as $url_row) {
            if ($url_row->status === 'pending') {
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                $wpdb->update(
                    $table_urls,
                    ['status' => 'crawled'],
                    ['id' => $url_row->id],
                    ['%s'],
                    ['%d']
                );
            }

            $this->simple_check_url($audit_id, (int) $url_row->id, $url_row->url);
        }
    }
    
    /**
     * Simple check URL implementation
     *
     * @param string $audit_id
     * @param int $url_id
     * @param string $url
     * @return void
     */
    private function simple_check_url(string $audit_id, int $url_id, string $url): void {
        // Get enabled check types from settings
        $saved_check_types = get_option('prorank_audit_check_types', []);
        $default_checks = [
            'meta_tags' => true,
            'https_status' => true,
            'image_optimization' => true,
            'headings_structure' => true,
            'broken_links' => true,
            'internal_linking' => true,
            'external_links' => true,
            'schema_validation' => true,
            'page_speed' => true,
            'accessibility' => true,
            'open_graph' => true,
            'twitter_cards' => true,
        ];
        $checks = array_merge($default_checks, $saved_check_types);

        // Get page content with optimized settings
        $response = wp_remote_get($url, [
            'timeout' => 3, // Reduced timeout for faster processing
            'redirection' => 2, // Limit redirects
            'user-agent' => 'ProRank SEO Site Audit/3.0',
            'sslverify' => true, // Skip SSL verification for speed
            'compress' => true, // Enable compression
        ]);

        if (is_wp_error($response)) {
            $this->add_issue($audit_id, $url, 'unreachable',
                'URL is unreachable: ' . $response->get_error_message(), 'high');
            $this->mark_url_checked($url_id);
            return;
        }

        $body = wp_remote_retrieve_body($response);
        $headers = wp_remote_retrieve_headers($response);

        // Run checks conditionally based on settings
        if (!empty($checks['meta_tags']) || !empty($checks['canonical_tags'])) {
            $this->check_meta_tags($audit_id, $url, $body);
        }
        if (!empty($checks['https_status']) || !empty($checks['ssl_certificate'])) {
            $this->check_https($audit_id, $url);
        }
        if (!empty($checks['image_optimization'])) {
            $this->check_images($audit_id, $url, $body);
        }
        if (!empty($checks['headings_structure'])) {
            $this->check_headings($audit_id, $url, $body);
        }
        if (!empty($checks['broken_links']) || !empty($checks['internal_linking']) || !empty($checks['external_links'])) {
            $this->check_links($audit_id, $url, $body);
        }
        if (!empty($checks['schema_validation'])) {
            $this->check_schema($audit_id, $url, $body);
        }
        if (!empty($checks['page_speed']) || !empty($checks['core_web_vitals'])) {
            $this->check_performance($audit_id, $url, $headers);
        }
        if (!empty($checks['accessibility'])) {
            $this->check_accessibility($audit_id, $url, $body);
        }
        if (!empty($checks['open_graph']) || !empty($checks['twitter_cards'])) {
            $this->check_social_tags($audit_id, $url, $body);
        }

        // Mark URL as checked
        $this->mark_url_checked($url_id);
    }
    
    /**
     * Simple complete audit implementation
     *
     * @param string $audit_id
     * @return void
     */
    private function simple_complete_audit(string $audit_id): void {
        global $wpdb;
        $table_audits = $wpdb->prefix . 'prorank_audits';
        $table_urls = $wpdb->prefix . 'prorank_audit_urls';
        $table_issues = $wpdb->prefix . 'prorank_audit_issues';
        
        // Check if all URLs are checked
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $pending = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_urls} 
            WHERE audit_id = %s AND status != 'completed'",
            $audit_id
        ));
        
        if ($pending > 0) {
            // Still have pending URLs, check again later
            if (function_exists('as_schedule_single_action')) {
                as_schedule_single_action(time() + 60, self::ACTION_COMPLETE_AUDIT, [$audit_id]);
            }
            return;
        }
        
        // Calculate comprehensive score
        $score = $this->calculate_audit_score($audit_id);
        
        // Get final stats
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $total_urls = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_urls} WHERE audit_id = %s",
            $audit_id
        ));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $total_issues = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity IN ('critical', 'high', 'medium', 'low')",
            $audit_id
        ));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $warnings = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity = 'warning'",
            $audit_id
        ));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $passed = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity = 'passed'",
            $audit_id
        ));
        
        // Update audit
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->update(
            $table_audits,
            [
                'status' => self::STATE_COMPLETED,
                'end_time' => current_time('mysql'),
                'score' => $score,
                'stats' => json_encode([
                    'urls_found' => $total_urls,
                    'urls_checked' => $total_urls,
                    'issues_found' => $total_issues,
                    'warnings_found' => $warnings,
                    'passed_checks' => $passed,
                ]),
            ],
            ['id' => $audit_id],
            ['%s', '%s', '%d', '%s'],
            ['%s']
        );
        
        // Update state
        update_option('prorank_audit_state', self::STATE_IDLE);
        delete_option('prorank_audit_current_id');
    }
    
    /**
     * Transition to checking phase
     *
     * @param string $audit_id
     * @return void
     */
    private function transition_to_checking(string $audit_id): void {
        update_option('prorank_audit_state', self::STATE_CHECKING);
        
        // Get all crawled URLs
        global $wpdb;
        $table_urls = $wpdb->prefix . 'prorank_audit_urls';
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $urls = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table_urls} 
            WHERE audit_id = %s AND status = 'crawled'",
            $audit_id
        ));
        
        // Schedule URL checks
        foreach ($urls as $url) {
            if (function_exists('as_enqueue_async_action')) {
                as_enqueue_async_action(self::ACTION_CHECK_URL, [$audit_id, $url->id, $url->url]);
            }
        }
        
        // Schedule completion check
        if (function_exists('as_schedule_single_action')) {
            as_schedule_single_action(time() + 300, self::ACTION_COMPLETE_AUDIT, [$audit_id]);
        }
    }
    
    /**
     * Add initial URLs to crawl
     *
     * @param string $audit_id
     * @return void
     */
    private function add_initial_urls(string $audit_id): void {
        global $wpdb;
        $table_urls = $wpdb->prefix . 'prorank_audit_urls';

        // Get settings - use saved values instead of hardcoded limits
        $max_urls = (int) get_option('prorank_audit_max_urls', 500);

        // Calculate proportional limits based on max_urls setting
        $sitemap_limit = min(intval($max_urls * 0.5), 500);   // 50% for sitemap, max 500
        $posts_limit = min(intval($max_urls * 0.25), 200);    // 25% for posts, max 200
        $pages_limit = min(intval($max_urls * 0.25), 200);    // 25% for pages, max 200

        $urls_to_add = [
            home_url('/') => 0,
        ];

        // Add sitemap URLs (use sitemap_limit instead of hardcoded 50)
        $sitemap_url = home_url('/sitemap.xml');
        $sitemap_response = wp_remote_get($sitemap_url);

        if (!is_wp_error($sitemap_response)) {
            $sitemap_content = wp_remote_retrieve_body($sitemap_response);
            if (!empty($sitemap_content)) {
                // Parse sitemap
                preg_match_all('/<loc>(.*?)<\/loc>/i', $sitemap_content, $matches);
                if (!empty($matches[1])) {
                    foreach (array_slice($matches[1], 0, $sitemap_limit) as $url) {
                        $urls_to_add[$url] = 1;
                    }
                }
            }
        }

        // Add recent posts (use posts_limit instead of hardcoded 20)
        $recent_posts = get_posts([
            'numberposts' => $posts_limit,
            'post_status' => 'publish',
        ]);

        foreach ($recent_posts as $post) {
            $urls_to_add[get_permalink($post)] = 1;
        }

        // Add recent pages (use pages_limit instead of hardcoded 20)
        $recent_pages = get_pages([
            'number' => $pages_limit,
            'post_status' => 'publish',
        ]);

        foreach ($recent_pages as $page) {
            $urls_to_add[get_permalink($page)] = 1;
        }

        // Enforce max_urls limit on total URLs
        if (count($urls_to_add) > $max_urls) {
            $urls_to_add = array_slice($urls_to_add, 0, $max_urls, true);
        }

        // Insert URLs
        foreach ($urls_to_add as $url => $depth) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->insert(
                $table_urls,
                [
                    'audit_id' => $audit_id,
                    'url' => $url,
                    'status' => 'pending',
                    'depth' => $depth,
                ],
                ['%s', '%s', '%s', '%d']
            );
        }
    }
    
    /**
     * Check meta tags
     *
     * @param string $audit_id
     * @param string $url
     * @param string $html
     * @return void
     */
    private function check_meta_tags(string $audit_id, string $url, string $html): void {
        // Check title tag
        if (!preg_match('/<title[^>]*>(.*?)<\/title>/i', $html, $matches)) {
            $this->add_issue($audit_id, $url, 'missing_title', 
                'Page is missing a title tag', 'critical');
        } else {
            $title = trim(wp_strip_all_tags($matches[1]));
            if (strlen($title) < 10) {
                $this->add_issue($audit_id, $url, 'short_title', 
                    'Title tag is too short (less than 10 characters)', 'medium');
            } elseif (strlen($title) > 60) {
                $this->add_issue($audit_id, $url, 'long_title', 
                    'Title tag is too long (more than 60 characters)', 'low');
            } else {
                $this->add_passed($audit_id, $url, 'title_tag', 
                    'Title tag is properly set');
            }
        }
        
        // Check meta description
        if (!preg_match('/<meta[^>]+name=[\'"]description[\'"][^>]+content=[\'"]([^\'"]*)[\'"]/i', $html, $matches)) {
            $this->add_issue($audit_id, $url, 'missing_meta_description', 
                'Page is missing a meta description', 'high');
        } else {
            $description = trim($matches[1]);
            if (strlen($description) < 50) {
                $this->add_issue($audit_id, $url, 'short_meta_description', 
                    'Meta description is too short (less than 50 characters)', 'medium');
            } elseif (strlen($description) > 160) {
                $this->add_issue($audit_id, $url, 'long_meta_description', 
                    'Meta description is too long (more than 160 characters)', 'low');
            } else {
                $this->add_passed($audit_id, $url, 'meta_description', 
                    'Meta description is properly set');
            }
        }
        
        // Check canonical tag
        if (preg_match('/<link[^>]+rel=[\'"]canonical[\'"][^>]+href=[\'"]([^\'"]*)[\'"]/i', $html, $matches)) {
            $this->add_passed($audit_id, $url, 'canonical_tag', 
                'Canonical tag is present');
        } else {
            $this->add_warning($audit_id, $url, 'missing_canonical', 
                'Page is missing a canonical tag');
        }
        
        // Check robots meta
        if (preg_match('/<meta[^>]+name=[\'"]robots[\'"][^>]+content=[\'"]([^\'"]*)[\'"]/i', $html, $matches)) {
            $robots = strtolower($matches[1]);
            if (strpos($robots, 'noindex') !== false) {
                $this->add_warning($audit_id, $url, 'noindex', 
                    'Page has noindex directive');
            }
        }
    }
    
    /**
     * Check HTTPS
     *
     * @param string $audit_id
     * @param string $url
     * @return void
     */
    private function check_https(string $audit_id, string $url): void {
        if (strpos($url, 'https://') !== 0) {
            $this->add_issue($audit_id, $url, 'no_https', 
                'Page is not served over HTTPS', 'critical');
        } else {
            $this->add_passed($audit_id, $url, 'https', 
                'Page is served over HTTPS');
        }
    }
    
    /**
     * Check images
     *
     * @param string $audit_id
     * @param string $url
     * @param string $html
     * @return void
     */
    private function check_images(string $audit_id, string $url, string $html): void {
        preg_match_all('/<img[^>]+>/i', $html, $images);
        
        $missing_alt_count = 0;
        $large_images = 0;
        
        foreach ($images[0] as $img) {
            // Check for alt text
            if (!preg_match('/alt=[\'"]([^\'"]*)[\'"]/', $img, $alt_match) || empty(trim($alt_match[1]))) {
                $missing_alt_count++;
            }
            
            // Check image size (simplified check)
            if (preg_match('/width=[\'"](\d+)[\'"]/', $img, $width_match) && 
                preg_match('/height=[\'"](\d+)[\'"]/', $img, $height_match)) {
                $width = (int) $width_match[1];
                $height = (int) $height_match[1];
                if ($width > 1920 || $height > 1080) {
                    $large_images++;
                }
            }
        }
        
        if ($missing_alt_count > 0) {
            $this->add_issue($audit_id, $url, 'missing_alt_text', 
                "$missing_alt_count images missing alt text", 'medium');
        }
        
        if ($large_images > 0) {
            $this->add_issue($audit_id, $url, 'large_images', 
                "$large_images oversized images found", 'medium');
        }
        
        if ($missing_alt_count === 0 && count($images[0]) > 0) {
            $this->add_passed($audit_id, $url, 'image_alt_text', 
                'All images have alt text');
        }
    }
    
    /**
     * Check headings
     *
     * @param string $audit_id
     * @param string $url
     * @param string $html
     * @return void
     */
    private function check_headings(string $audit_id, string $url, string $html): void {
        // Check for H1
        preg_match_all('/<h1[^>]*>(.*?)<\/h1>/i', $html, $h1_matches);
        
        if (count($h1_matches[0]) === 0) {
            $this->add_issue($audit_id, $url, 'missing_h1', 
                'Page is missing an H1 tag', 'high');
        } elseif (count($h1_matches[0]) > 1) {
            $this->add_issue($audit_id, $url, 'multiple_h1', 
                'Page has multiple H1 tags', 'medium');
        } else {
            $this->add_passed($audit_id, $url, 'h1_tag', 
                'Page has exactly one H1 tag');
        }
        
        // Check heading hierarchy
        preg_match_all('/<h([1-6])[^>]*>/i', $html, $all_headings, PREG_OFFSET_CAPTURE);
        
        if (!empty($all_headings[1])) {
            $previous_level = 0;
            $hierarchy_broken = false;
            
            foreach ($all_headings[1] as $heading) {
                $level = (int) $heading[0];
                if ($level > $previous_level + 1 && $previous_level > 0) {
                    $hierarchy_broken = true;
                    break;
                }
                $previous_level = $level;
            }
            
            if ($hierarchy_broken) {
                $this->add_warning($audit_id, $url, 'heading_hierarchy', 
                    'Heading hierarchy is broken (skipped levels)');
            } else {
                $this->add_passed($audit_id, $url, 'heading_hierarchy', 
                    'Heading hierarchy is properly structured');
            }
        }
    }
    
    /**
     * Check links
     *
     * @param string $audit_id
     * @param string $url
     * @param string $html
     * @return void
     */
    private function check_links(string $audit_id, string $url, string $html): void {
        preg_match_all('/<a[^>]+href=[\'"]([^\'"\#]+)[\'"][^>]*>/i', $html, $links);
        
        $internal_links = 0;
        $external_links = 0;
        $broken_links = [];
        
        $home_url = home_url();
        
        foreach ($links[1] as $link) {
            if (strpos($link, 'http') === 0) {
                if (strpos($link, $home_url) === 0) {
                    $internal_links++;
                } else {
                    $external_links++;
                }
            } else {
                // Relative URL
                $internal_links++;
            }
        }
        
        if ($internal_links === 0) {
            $this->add_warning($audit_id, $url, 'no_internal_links', 
                'Page has no internal links');
        } else {
            $this->add_passed($audit_id, $url, 'internal_links', 
                "Page has $internal_links internal links");
        }
    }
    
    /**
     * Check schema markup
     *
     * @param string $audit_id
     * @param string $url
     * @param string $html
     * @return void
     */
    private function check_schema(string $audit_id, string $url, string $html): void {
        // Check for JSON-LD
        if (preg_match('/<script[^>]+type=[\'"]application\/ld\+json[\'"][^>]*>/i', $html)) {
            $this->add_passed($audit_id, $url, 'schema_markup', 
                'Page has JSON-LD schema markup');
        } else {
            $this->add_warning($audit_id, $url, 'no_schema', 
                'Page is missing schema markup');
        }
    }
    
    /**
     * Check performance indicators
     *
     * @param string $audit_id
     * @param string $url
     * @param mixed $headers
     * @return void
     */
    private function check_performance(string $audit_id, string $url, $headers): void {
        // Check compression
        $encoding = '';
        if (is_array($headers)) {
            $encoding = $headers['content-encoding'] ?? '';
        } elseif (is_object($headers)) {
            $encoding = $headers->getValues('content-encoding')[0] ?? '';
        }
        
        if (empty($encoding) || strpos($encoding, 'gzip') === false) {
            $this->add_issue($audit_id, $url, 'no_compression', 
                'Page is not compressed with GZIP', 'medium');
        } else {
            $this->add_passed($audit_id, $url, 'gzip_compression', 
                'Page is compressed with GZIP');
        }
        
        // Check caching headers
        $cache_control = '';
        if (is_array($headers)) {
            $cache_control = $headers['cache-control'] ?? '';
        } elseif (is_object($headers)) {
            $cache_control = $headers->getValues('cache-control')[0] ?? '';
        }
        
        if (empty($cache_control) || strpos($cache_control, 'no-cache') !== false) {
            $this->add_issue($audit_id, $url, 'no_caching', 
                'Page has no caching headers', 'low');
        } else {
            $this->add_passed($audit_id, $url, 'caching_headers', 
                'Page has proper caching headers');
        }
    }
    
    /**
     * Check accessibility
     *
     * @param string $audit_id
     * @param string $url
     * @param string $html
     * @return void
     */
    private function check_accessibility(string $audit_id, string $url, string $html): void {
        // Check lang attribute
        if (!preg_match('/<html[^>]+lang=[\'"]([^\'"]*)[\'"]/i', $html)) {
            $this->add_issue($audit_id, $url, 'missing_lang', 
                'HTML element is missing lang attribute', 'high');
        } else {
            $this->add_passed($audit_id, $url, 'lang_attribute', 
                'HTML has lang attribute');
        }
        
        // Check viewport meta
        if (!preg_match('/<meta[^>]+name=[\'"]viewport[\'"]/i', $html)) {
            $this->add_issue($audit_id, $url, 'missing_viewport', 
                'Page is missing viewport meta tag', 'high');
        } else {
            $this->add_passed($audit_id, $url, 'viewport_meta', 
                'Page has viewport meta tag');
        }
    }
    
    /**
     * Check social tags
     *
     * @param string $audit_id
     * @param string $url
     * @param string $html
     * @return void
     */
    private function check_social_tags(string $audit_id, string $url, string $html): void {
        // Check Open Graph tags
        $og_tags = ['og:title', 'og:description', 'og:image', 'og:url'];
        $found_og = 0;
        
        foreach ($og_tags as $tag) {
            if (preg_match('/<meta[^>]+property=[\'"]' . $tag . '[\'"]/i', $html)) {
                $found_og++;
            }
        }
        
        if ($found_og === 0) {
            $this->add_warning($audit_id, $url, 'no_open_graph', 
                'Page is missing Open Graph tags');
        } elseif ($found_og < count($og_tags)) {
            $this->add_warning($audit_id, $url, 'incomplete_open_graph', 
                'Page has incomplete Open Graph tags');
        } else {
            $this->add_passed($audit_id, $url, 'open_graph', 
                'Page has complete Open Graph tags');
        }
        
        // Check Twitter Card
        if (preg_match('/<meta[^>]+name=[\'"]twitter:card[\'"]/i', $html)) {
            $this->add_passed($audit_id, $url, 'twitter_card', 
                'Page has Twitter Card meta tags');
        } else {
            $this->add_warning($audit_id, $url, 'no_twitter_card', 
                'Page is missing Twitter Card meta tags');
        }
    }
    
    /**
     * Add an issue
     *
     * @param string $audit_id
     * @param string $url
     * @param string $type
     * @param string $message
     * @param string $severity
     * @return void
     */
    private function add_issue(string $audit_id, string $url, string $type, string $message, string $severity): void {
        global $wpdb;
        $table_issues = $wpdb->prefix . 'prorank_audit_issues';
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->insert(
            $table_issues,
            [
                'audit_id' => $audit_id,
                'url' => $url,
                'type' => $type,
                'message' => $message,
                'severity' => $severity,
                'data' => json_encode([]),
            ],
            ['%s', '%s', '%s', '%s', '%s', '%s']
        );
    }
    
    /**
     * Add a warning
     *
     * @param string $audit_id
     * @param string $url
     * @param string $type
     * @param string $message
     * @return void
     */
    private function add_warning(string $audit_id, string $url, string $type, string $message): void {
        $this->add_issue($audit_id, $url, $type, $message, 'warning');
    }
    
    /**
     * Add a passed check
     *
     * @param string $audit_id
     * @param string $url
     * @param string $type
     * @param string $message
     * @return void
     */
    private function add_passed(string $audit_id, string $url, string $type, string $message): void {
        $this->add_issue($audit_id, $url, $type, $message, 'passed');
    }
    
    /**
     * Mark URL as checked
     *
     * @param int $url_id
     * @return void
     */
    private function mark_url_checked(int $url_id): void {
        global $wpdb;
        $table_urls = $wpdb->prefix . 'prorank_audit_urls';
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->update(
            $table_urls,
            ['status' => 'completed'],
            ['id' => $url_id],
            ['%s'],
            ['%d']
        );
    }
    
    /**
     * Calculate audit score
     *
     * @param string $audit_id
     * @return int
     */
    private function calculate_audit_score(string $audit_id): int {
        global $wpdb;
        $table_issues = $wpdb->prefix . 'prorank_audit_issues';
        
        // Count issues by severity
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $critical = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity = 'critical'",
            $audit_id
        ));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $high = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity = 'high'",
            $audit_id
        ));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $medium = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity = 'medium'",
            $audit_id
        ));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $low = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity = 'low'",
            $audit_id
        ));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $warnings = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity = 'warning'",
            $audit_id
        ));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $passed = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_issues} WHERE audit_id = %s AND severity = 'passed'",
            $audit_id
        ));
        
        // Calculate score
        $total_checks = $critical + $high + $medium + $low + $warnings + $passed;
        if ($total_checks === 0) {
            return 100;
        }
        
        $passed_ratio = $passed / $total_checks;
        $score = $passed_ratio * 100;
        
        // Deduct points for issues
        $score -= ($critical * 10);
        $score -= ($high * 5);
        $score -= ($medium * 3);
        $score -= ($low * 1);
        $score -= ($warnings * 0.5);
        
        return max(0, min(100, round($score)));
    }
    
    /**
     * Get issue category
     *
     * @param string $type
     * @return string
     */
    private function get_issue_category(string $type): string {
        $categories = [
            // Technical SEO
            'missing_title' => 'technical-seo',
            'short_title' => 'technical-seo',
            'long_title' => 'technical-seo',
            'missing_meta_description' => 'technical-seo',
            'short_meta_description' => 'technical-seo',
            'long_meta_description' => 'technical-seo',
            'missing_canonical' => 'technical-seo',
            'noindex' => 'technical-seo',
            'broken_link' => 'technical-seo',
            'redirect_chain' => 'technical-seo',
            'missing_sitemap' => 'technical-seo',
            
            // On-Page SEO
            'missing_h1' => 'on-page-seo',
            'multiple_h1' => 'on-page-seo',
            'heading_hierarchy' => 'on-page-seo',
            'missing_alt_text' => 'on-page-seo',
            'no_internal_links' => 'on-page-seo',
            'keyword_stuffing' => 'on-page-seo',
            
            // Performance
            'slow_page' => 'performance',
            'large_images' => 'performance',
            'no_compression' => 'performance',
            'no_caching' => 'performance',
            'large_page_size' => 'performance',
            'render_blocking' => 'performance',
            
            // Security
            'no_https' => 'security',
            'mixed_content' => 'security',
            'missing_security_headers' => 'security',
            
            // Content Quality
            'thin_content' => 'content-quality',
            'duplicate_content' => 'content-quality',
            'poor_readability' => 'content-quality',
            'no_schema' => 'content-quality',
            
            // Social & Rich Results
            'no_open_graph' => 'social-rich-results',
            'incomplete_open_graph' => 'social-rich-results',
            'no_twitter_card' => 'social-rich-results',
            
            // Accessibility
            'missing_lang' => 'accessibility',
            'missing_viewport' => 'accessibility',
        ];
        
        return $categories[$type] ?? 'other';
    }
    
    /**
     * Check if issue can be auto-fixed
     *
     * @param string $type
     * @return bool
     */
    private function can_autofix(string $type): bool {
        $autofixable = [
            'no_compression',
            'no_caching',
        ];

        if (!in_array($type, $autofixable, true)) {
            return false;
        }

        $settings = get_option('prorank_audit_settings', []);
        $enabled = array_key_exists('enable_autofix', $settings)
            ? (bool) $settings['enable_autofix']
            : (bool) get_option('prorank_audit_autofix_enabled', false);

        if (!$enabled) {
            return false;
        }

        return true;
    }
    
    /**
     * Apply auto-fix
     *
     * @param string $type
     * @param int $issue_id
     * @return bool
     */
    private function apply_autofix(string $type, int $issue_id): bool {
        switch ($type) {
            case 'no_compression':
                return $this->enable_gzip_compression();
                
            case 'no_caching':
                return $this->enable_browser_caching();
                
            default:
                return false;
        }
    }
    
    /**
     * Enable GZIP compression
     *
     * @return bool
     */
    private function enable_gzip_compression(): bool {
        $htaccess_path = prorank_get_home_path() . '.htaccess';

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        if (!$wp_filesystem || !$wp_filesystem->is_writable($htaccess_path)) {
            return false;
        }
        
        $gzip_rules = "\n# BEGIN ProRank SEO GZIP Compression\n";
        $gzip_rules .= "<IfModule mod_deflate.c>\n";
        $gzip_rules .= "  AddOutputFilterByType DEFLATE text/plain\n";
        $gzip_rules .= "  AddOutputFilterByType DEFLATE text/html\n";
        $gzip_rules .= "  AddOutputFilterByType DEFLATE text/xml\n";
        $gzip_rules .= "  AddOutputFilterByType DEFLATE text/css\n";
        $gzip_rules .= "  AddOutputFilterByType DEFLATE application/xml\n";
        $gzip_rules .= "  AddOutputFilterByType DEFLATE application/xhtml+xml\n";
        $gzip_rules .= "  AddOutputFilterByType DEFLATE application/rss+xml\n";
        $gzip_rules .= "  AddOutputFilterByType DEFLATE application/javascript\n";
        $gzip_rules .= "  AddOutputFilterByType DEFLATE application/x-javascript\n";
        $gzip_rules .= "</IfModule>\n";
        $gzip_rules .= "# END ProRank SEO GZIP Compression\n";
        
        $htaccess_content = $wp_filesystem ? $wp_filesystem->get_contents($htaccess_path) : '';
        if (!is_string($htaccess_content)) {
            $htaccess_content = '';
        }
        
        if (strpos($htaccess_content, '# BEGIN ProRank SEO GZIP Compression') === false) {
            $htaccess_content = $gzip_rules . $htaccess_content;
            return (bool) ($wp_filesystem && $wp_filesystem->put_contents($htaccess_path, $htaccess_content, FS_CHMOD_FILE));
        }
        
        return true;
    }
    
    /**
     * Enable browser caching
     *
     * @return bool
     */
    private function enable_browser_caching(): bool {
        $htaccess_path = prorank_get_home_path() . '.htaccess';

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        if (!$wp_filesystem || !$wp_filesystem->is_writable($htaccess_path)) {
            return false;
        }
        
        $cache_rules = "\n# BEGIN ProRank SEO Browser Caching\n";
        $cache_rules .= "<IfModule mod_expires.c>\n";
        $cache_rules .= "  ExpiresActive On\n";
        $cache_rules .= "  ExpiresByType image/jpg \"access plus 1 year\"\n";
        $cache_rules .= "  ExpiresByType image/jpeg \"access plus 1 year\"\n";
        $cache_rules .= "  ExpiresByType image/gif \"access plus 1 year\"\n";
        $cache_rules .= "  ExpiresByType image/png \"access plus 1 year\"\n";
        $cache_rules .= "  ExpiresByType text/css \"access plus 1 month\"\n";
        $cache_rules .= "  ExpiresByType application/javascript \"access plus 1 month\"\n";
        $cache_rules .= "</IfModule>\n";
        $cache_rules .= "# END ProRank SEO Browser Caching\n";
        
        $htaccess_content = $wp_filesystem ? $wp_filesystem->get_contents($htaccess_path) : '';
        if (!is_string($htaccess_content)) {
            $htaccess_content = '';
        }
        
        if (strpos($htaccess_content, '# BEGIN ProRank SEO Browser Caching') === false) {
            $htaccess_content = $cache_rules . $htaccess_content;
            return (bool) ($wp_filesystem && $wp_filesystem->put_contents($htaccess_path, $htaccess_content, FS_CHMOD_FILE));
        }
        
        return true;
    }
    
    /**
     * Get fix title
     *
     * @param string $type
     * @return string
     */
    private function get_fix_title(string $type): string {
        $titles = [
            'no_compression' => __('Enable GZIP Compression', 'prorank-seo'),
            'no_caching' => __('Enable Browser Caching', 'prorank-seo'),
        ];
        
        return $titles[$type] ?? $type;
    }
    
    /**
     * Get fix description
     *
     * @param string $type
     * @return string
     */
    private function get_fix_description(string $type): string {
        $descriptions = [
            'no_compression' => __('Compress your pages to reduce file size and improve loading speed', 'prorank-seo'),
            'no_caching' => __('Set browser caching headers to improve repeat visitor performance', 'prorank-seo'),
        ];
        
        return $descriptions[$type] ?? '';
    }
    
    /**
     * Get fix impact
     *
     * @param string $type
     * @return string
     */
    private function get_fix_impact(string $type): string {
        $impacts = [
            'no_compression' => __('Can reduce page size by 70-90%', 'prorank-seo'),
            'no_caching' => __('Improves repeat visitor load time by 50%+', 'prorank-seo'),
        ];
        
        return $impacts[$type] ?? '';
    }
    
    /**
     * Get empty categories
     *
     * @return array
     */
    private function get_empty_categories(): array {
        return [
            'technical-seo' => ['issues' => 0, 'warnings' => 0, 'passed' => 0],
            'on-page-seo' => ['issues' => 0, 'warnings' => 0, 'passed' => 0],
            'performance' => ['issues' => 0, 'warnings' => 0, 'passed' => 0],
            'security' => ['issues' => 0, 'warnings' => 0, 'passed' => 0],
            'content-quality' => ['issues' => 0, 'warnings' => 0, 'passed' => 0],
            'social-rich-results' => ['issues' => 0, 'warnings' => 0, 'passed' => 0],
            'accessibility' => ['issues' => 0, 'warnings' => 0, 'passed' => 0],
        ];
    }
    
    /**
     * Get category stats
     *
     * @param string $audit_id
     * @return array
     */
    private function get_category_stats(string $audit_id): array {
        global $wpdb;
        $table_issues = $wpdb->prefix . 'prorank_audit_issues';
        
        $categories = $this->get_empty_categories();
        
        // Get all issues
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $issues = $wpdb->get_results($wpdb->prepare(
            "SELECT type, severity, COUNT(*) as count 
            FROM {$table_issues} 
            WHERE audit_id = %s 
            GROUP BY type, severity",
            $audit_id
        ));
        
        // Check if we got results
        if (!$issues || is_wp_error($issues)) {
            return $categories;
        }
        
        // Categorize issues
        foreach ($issues as $issue) {
            $category = $this->get_issue_category($issue->type);
            
            if (isset($categories[$category])) {
                if (in_array($issue->severity, ['critical', 'high', 'medium', 'low'])) {
                    $categories[$category]['issues'] += (int) $issue->count;
                } elseif ($issue->severity === 'warning') {
                    $categories[$category]['warnings'] += (int) $issue->count;
                } elseif ($issue->severity === 'passed') {
                    $categories[$category]['passed'] += (int) $issue->count;
                }
            }
        }
        
        return $categories;
    }
    
    /**
     * Clear scheduled events for a specific audit
     *
     * @param string $audit_id
     * @return void
     */
    private function clear_audit_scheduled_events(string $audit_id): void {
        // Clear specific audit events
        $hooks = [
            'prorank_crawl_batch',
            'prorank_check_url', 
            'prorank_complete_audit',
            'prorank_schedule_url_checks',
            'prorank_check_audit_url',
            'prorank_run_site_crawler',
        ];
        
        foreach ($hooks as $hook) {
            $timestamp = wp_next_scheduled($hook, [$audit_id]);
            if ($timestamp) {
                wp_unschedule_event($timestamp, $hook, [$audit_id]);
            }
        }
        
        // Clear all URL check events for this audit
        $crons = _get_cron_array();
        foreach ($crons as $timestamp => $cron) {
            foreach ($cron as $hook => $args) {
                if (strpos($hook, 'prorank_') === 0 && strpos($hook, 'audit') !== false) {
                    foreach ($args as $key => $arg) {
                        if (isset($arg['args'][0]) && $arg['args'][0] === $audit_id) {
                            wp_unschedule_event($timestamp, $hook, $arg['args']);
                        }
                    }
                }
            }
        }
    }
    
    /**
     * Clear all audit-related cron events
     *
     * @return void
     */
        /**
     * Clear all audit-related cron events
     */
    private function clear_all_audit_crons(): void {
        $crons = _get_cron_array();
        $cleared = 0;
        foreach ($crons as $timestamp => $cron) {
            foreach ($cron as $hook => $args) {
                if (strpos($hook, 'prorank_') === 0 && (strpos($hook, 'audit') !== false || strpos($hook, 'crawl') !== false || strpos($hook, 'check') !== false)) {
                    foreach ($args as $arg) {
                        wp_unschedule_event($timestamp, $hook, $arg['args']);
                        $cleared++;
                    }
                }
            }
        }
        if ($cleared > 0) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Cleared ' . $cleared . ' audit-related cron events');
            }
        }
    }

    /**
     * Send alert email when audit completes
     */
    private function maybe_send_alert(?string $audit_id = null): void {
        try {
            $status = $this->rest_get_status();
            if (!is_array($status) || empty($status['state']) || $status['state'] !== 'completed') {
                return;
            }
            $score = $status['score'] ?? 0;
            $critical = $status['stats']['critical_issues'] ?? 0;
            $high = $status['stats']['high_issues'] ?? 0;

            $settings = get_option('prorank_audit_settings', []);
            $email_enabled = array_key_exists('email_notifications', $settings)
                ? (bool) $settings['email_notifications']
                : (bool) get_option('prorank_audit_email_notifications', true);

            $critical_only = array_key_exists('email_critical_only', $settings)
                ? (bool) $settings['email_critical_only']
                : (bool) get_option('prorank_audit_email_critical_only', true);

            if (!$email_enabled) {
                return;
            }

            if ($critical_only && $critical < 1) {
                return;
            }

            if ($score >= 90 && $critical == 0 && $high == 0) {
                return;
            }

            $emails = array_key_exists('notification_emails', $settings)
                ? (string) $settings['notification_emails']
                : (string) get_option('prorank_audit_notification_emails', '');

            $recipients = array_filter(array_map('trim', preg_split('/[,\s]+/', $emails)));
            $to = $recipients ? implode(',', $recipients) : (string) get_option('admin_email');
            if (empty($to)) {
                return;
            }
            $subject = sprintf(
                /* translators: %s: numeric value */
                __('Site Audit results: Score %d', 'prorank-seo'), $score);
            $body = "Site Audit completed

";
            $body .= 'Score: ' . $score . "
";
            $body .= 'Critical issues: ' . $critical . "
";
            $body .= 'High issues: ' . $high . "
";
            $body .= 'Audit ID: ' . ($status['audit_id'] ?? '') . "
";
            $body .= 'View audit in WP Admin.';
            wp_mail($to, $subject, $body);
        } catch (\Exception $e) {
            // ignore
        }
    }


    /**
     * Schedule recurring audit if enabled
     */
    private function maybe_schedule_recurring_audit(): void {
        $settings = get_option('prorank_audit_settings', []);
        $enabled = array_key_exists('schedule_enabled', $settings)
            ? (bool) $settings['schedule_enabled']
            : (bool) get_option('prorank_audit_schedule_enabled', false);

        $frequency = $settings['schedule_frequency'] ?? get_option('prorank_audit_schedule_frequency', 'weekly');
        $schedule_time = $settings['schedule_time'] ?? get_option('prorank_audit_schedule_time', '02:00');

        $recurrence = in_array($frequency, ['daily', 'weekly', 'biweekly', 'monthly'], true)
            ? $frequency
            : 'weekly';

        $hook = 'prorank_audit_scheduled_run';
        $existing = wp_next_scheduled($hook);

        if ($enabled) {
            $next_time = $this->get_next_audit_time((string) $schedule_time, $recurrence);
            $current_schedule = $existing ? wp_get_schedule($hook) : false;

            if ($existing && ($current_schedule !== $recurrence || abs($existing - $next_time) > MINUTE_IN_SECONDS)) {
                wp_unschedule_event($existing, $hook);
                $existing = false;
            }

            if (!$existing) {
                wp_schedule_event($next_time, $recurrence, $hook);
            }
            return;
        }

        if ($existing) {
            wp_unschedule_event($existing, $hook);
        }
    }

    /**
     * Calculate next scheduled audit time from HH:MM and recurrence.
     */
    private function get_next_audit_time(string $time, string $recurrence): int {
        $hour = 2;
        $minute = 0;

        if (preg_match('/^(\d{1,2}):(\d{2})$/', $time, $matches)) {
            $hour = max(0, min(23, (int) $matches[1]));
            $minute = max(0, min(59, (int) $matches[2]));
        }

        $timezone = wp_timezone();
        $now = new \DateTime('now', $timezone);
        $next = clone $now;
        $next->setTime($hour, $minute, 0);

        if ($next <= $now) {
            switch ($recurrence) {
                case 'daily':
                    $next->modify('+1 day');
                    break;
                case 'biweekly':
                    $next->modify('+2 weeks');
                    break;
                case 'monthly':
                    $next->modify('+1 month');
                    break;
                case 'weekly':
                default:
                    $next->modify('+1 week');
                    break;
            }
        }

        return $next->getTimestamp();
    }

    /**
     * Handle scheduled audit run
     */
    public function handle_scheduled_audit(): void {
        try {
            $this->rest_start_audit();
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Scheduled audit failed: ' . $e->getMessage());
            }
        }
    }

    /**
     * Send alert summary when triggered by cron
     */
    public function maybe_send_alert_from_status(): void {
        $status = $this->rest_get_status();
        if (!is_array($status) || empty($status['state']) || $status['state'] !== 'completed') {
            return;
        }
        $this->maybe_send_alert($status['audit_id'] ?? null);
    }

    /**
     * Create database tables
     *
     * @return void
     */
    private function create_tables(): void {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Drop malformed tables if they exist
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}IF");
        
        // Audits table
        $table_audits = $wpdb->prefix . 'prorank_audits';
        $sql_audits = "CREATE TABLE IF NOT EXISTS $table_audits (
            id varchar(36) NOT NULL,
            status varchar(20) NOT NULL DEFAULT 'pending',
            start_time datetime DEFAULT NULL,
            end_time datetime DEFAULT NULL,
            score int(3) DEFAULT 0,
            stats longtext,
            options longtext,
            PRIMARY KEY (id),
            KEY status (status),
            KEY start_time (start_time)
        ) $charset_collate";
        
        // URLs table
        $table_urls = $wpdb->prefix . 'prorank_audit_urls';
        $sql_urls = "CREATE TABLE IF NOT EXISTS $table_urls (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            audit_id varchar(36) NOT NULL,
            url varchar(2000) NOT NULL,
            status varchar(20) NOT NULL DEFAULT 'pending',
            depth int(11) DEFAULT 0,
            PRIMARY KEY (id),
            KEY audit_id (audit_id),
            KEY status (status)
        ) $charset_collate";
        
        // Issues table
        $table_issues = $wpdb->prefix . 'prorank_audit_issues';
        $sql_issues = "CREATE TABLE IF NOT EXISTS $table_issues (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            audit_id varchar(36) NOT NULL,
            url varchar(2000) NOT NULL,
            type varchar(50) NOT NULL,
            message text NOT NULL,
            severity varchar(20) NOT NULL DEFAULT 'medium',
            data longtext,
            PRIMARY KEY (id),
            KEY audit_id (audit_id),
            KEY type (type),
            KEY severity (severity)
        ) $charset_collate";
        
        // Use direct queries instead of dbDelta to avoid parsing issues
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->query($sql_audits);
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->query($sql_urls);
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->query($sql_issues);

        // Add options column to existing tables (for upgrades)
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $column_exists = $wpdb->get_var("SHOW COLUMNS FROM {$table_audits} LIKE 'options'");
        if (!$column_exists) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("ALTER TABLE {$table_audits} ADD COLUMN options longtext AFTER stats");
        }

        // Add depth column to audit_urls table (for upgrades)
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $depth_exists = $wpdb->get_var("SHOW COLUMNS FROM {$table_urls} LIKE 'depth'");
        if (!$depth_exists) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("ALTER TABLE {$table_urls} ADD COLUMN depth int(11) DEFAULT 0 AFTER status");
        }
    }
    
    /**
     * Check if tables exist
     *
     * @return bool
     */
    private function check_tables_exist(): bool {
        global $wpdb;
        
        $tables = [
            $wpdb->prefix . 'prorank_audits',
            $wpdb->prefix . 'prorank_audit_urls',
            $wpdb->prefix . 'prorank_audit_issues',
        ];
        
        foreach ($tables as $table) {
            if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table)) !== $table) {
                return false;
            }
        }
        
        return true;
    }
}
