<?php
/**
 * Sitemaps Module
 *
 * Handles XML sitemap generation and management
 *
 * @package ProRank\SEO\Modules\Indexing
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\TechnicalSEO;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Modules\BaseModule;
use ProRank\SEO\Core\ModuleManager;

/**
 * Sitemaps module class
 */
class SitemapsModule extends BaseModule {
    
    /**
     * Module slug
     *
     * @var string
     */
    protected string $slug = 'sitemaps';
    
    /**
     * Module name
     *
     * @var string
     */
    protected string $name = 'XML Sitemaps';
    
    /**
     * Module description
     *
     * @var string
     */
    protected string $description = 'Automatic XML sitemap generation';
    
    /**
     * Required tier
     *
     * @var string
     */
    protected string $feature_tier = 'free';
    
    /**
     * Parent module
     *
     * @var string|null
     */
    protected ?string $parent_slug = 'technical-seo';
    
    /**
     * Sitemap providers
     *
     * @var array
     */
    private array $providers = [];
    
    /**
     * Settings manager instance
     *
     * @var \ProRank\SEO\Core\SettingsManager|null
     */
    private ?\ProRank\SEO\Core\SettingsManager $settings_manager = null;
    
    
    /**
     * Initialize the module
     *
     * @return void
     */
    public function init(): void {
        parent::init();
        
        // Get manager instances from plugin
        try {
            $plugin = $this->plugin();
            $this->settings_manager = $plugin->settings();
        } catch (\Exception $e) {
            // Managers not available, skip initialization
            return;
        }
        
        // Register default providers
        $this->register_default_providers();
        
        // Allow other modules to register providers
        do_action('prorank_seo_register_sitemap_providers', $this);
    }
    
    /**
     * Initialize module hooks
     *
     * @return void
     */
    public function init_hooks(): void {
        // Module is already active if init_hooks is being called
        
        // Register query vars early
        add_filter('query_vars', [$this, 'add_query_vars'], 10);
        
        // Register rewrite rules
        add_action('init', [$this, 'add_rewrite_rules'], 10);
        
        // Handle sitemap requests
        add_action('template_redirect', [$this, 'handle_sitemap_request'], 1);
        
        // Alternative handler for sitemap requests
        add_action('parse_request', [$this, 'parse_sitemap_request'], 1);
        
        // Flush rewrite rules on activation/deactivation
        add_action('prorank_seo_module_activated_' . $this->slug, [$this, 'flush_rewrite_rules']);
        add_action('prorank_seo_module_deactivated_' . $this->slug, [$this, 'flush_rewrite_rules']);
        
        // Add robots.txt rules
        add_filter('robots_txt', [$this, 'add_robots_txt_rules'], 10, 2);
        
        // Ping search engines when content is updated
        add_action('transition_post_status', [$this, 'maybe_ping_search_engines'], 10, 3);
        add_action('created_term', [$this, 'ping_on_term_change'], 10, 3);
        add_action('edited_term', [$this, 'ping_on_term_change'], 10, 3);
        add_action('delete_term', [$this, 'ping_on_term_change'], 10, 3);

        // 2025 Enhancement: Invalidate sitemap cache when content changes
        add_action('save_post', [$this, 'invalidate_sitemap_cache_on_save'], 10, 2);
        add_action('delete_post', [$this, 'invalidate_sitemap_cache_on_delete'], 10);
        add_action('created_term', [$this, 'invalidate_sitemap_cache_on_term_change'], 10, 3);
        add_action('edited_term', [$this, 'invalidate_sitemap_cache_on_term_change'], 10, 3);
        add_action('delete_term', [$this, 'invalidate_sitemap_cache_on_term_change'], 10, 3);
    }
    
    /**
     * Register default sitemap providers
     *
     * @return void
     */
    private function register_default_providers(): void {
        // Register post provider
        $this->register_provider(new \ProRank\SEO\Modules\Indexing\Sitemaps\PostSitemapProvider());
        
        // Register page provider
        $this->register_provider(new \ProRank\SEO\Modules\Indexing\Sitemaps\PageSitemapProvider());
        
        // Register category provider
        $this->register_provider(new \ProRank\SEO\Modules\Indexing\Sitemaps\CategorySitemapProvider());
        
        // Register tag provider
        $this->register_provider(new \ProRank\SEO\Modules\Indexing\Sitemaps\TagSitemapProvider());
        
    }
    
    /**
     * Register a sitemap provider
     *
     * @param \ProRank\SEO\Modules\Indexing\Sitemaps\BaseSitemapProvider $provider Provider instance
     * @return void
     */
    public function register_provider(\ProRank\SEO\Modules\Indexing\Sitemaps\BaseSitemapProvider $provider): void {
        // Configure provider with settings if available
        if ($this->settings_manager) {
            // Inject settings manager
            if (method_exists($provider, 'set_settings_manager')) {
                $provider->set_settings_manager($this->settings_manager);
            }
            
            // Configure max entries
            $settings = $this->settings_manager->get('sitemaps', []);
            $max_entries = isset($settings['posts_per_sitemap']) ? (int) $settings['posts_per_sitemap'] : 1000;
            if (method_exists($provider, 'set_max_entries')) {
                $provider->set_max_entries($max_entries);
            }
        }
        
        $this->providers[$provider->get_slug()] = $provider;
    }
    
    /**
     * Get registered providers
     *
     * @return array
     */
    public function get_providers(): array {
        return apply_filters('prorank_seo_sitemap_providers', $this->providers);
    }
    
    /**
     * Add rewrite rules for sitemaps
     *
     * @return void
     */
    public function add_rewrite_rules(): void {
        // Sitemap index
        add_rewrite_rule(
            'sitemap_index\.xml$',
            'index.php?prorank_sitemap=index',
            'top'
        );
        
        // Individual sitemaps
        add_rewrite_rule(
            '([^/]+)-sitemap([0-9]+)?\.xml$',
            'index.php?prorank_sitemap=$matches[1]&prorank_sitemap_page=$matches[2]',
            'top'
        );
    }
    
    /**
     * Add query vars for sitemaps
     *
     * @param array $vars Query vars
     * @return array
     */
    public function add_query_vars(array $vars): array {
        $vars[] = 'prorank_sitemap';
        $vars[] = 'prorank_sitemap_page';
        return $vars;
    }
    
    /**
     * Parse sitemap requests directly from URL
     *
     * @param \WP $wp WordPress environment object
     * @return void
     */
    public function parse_sitemap_request(\WP $wp): void {
        $request_uri = \prorank_get_server_var( 'REQUEST_URI' );
        
        // Check if this is a sitemap request
        if (preg_match('/sitemap_index\.xml$/', $request_uri)) {
            // Set query vars manually
            $wp->query_vars['prorank_sitemap'] = 'index';
        } elseif (preg_match('/([^\/]+)-sitemap(\d+)?\.xml$/', $request_uri, $matches)) {
            // Set query vars manually
            $wp->query_vars['prorank_sitemap'] = $matches[1];
            if (!empty($matches[2])) {
                $wp->query_vars['prorank_sitemap_page'] = $matches[2];
            }
        }
    }
    
    /**
     * Handle sitemap requests
     *
     * @return void
     */
    public function handle_sitemap_request(): void {
        global $wp_query;
        
        $sitemap_type = get_query_var('prorank_sitemap');
        if (empty($sitemap_type)) {
            return;
        }
        
        // Module is already active if this method is being called
        
        // Check if sitemaps feature is enabled in settings
        $settings = $this->settings_manager ? $this->settings_manager->get('sitemaps', []) : [];
        // Default to enabled for free tier
        $enabled = isset($settings['enable_xml_sitemap']) ? $settings['enable_xml_sitemap'] : true;
        if (!$enabled) {
            $wp_query->set_404();
            status_header(404);
            return;
        }
        
        // Enable gzip compression if available
        $use_gzip = $this->maybe_enable_gzip_compression();

        // Set XML headers
        header('Content-Type: application/xml; charset=utf-8');
        header('X-Robots-Tag: noindex, follow');

        if ($use_gzip) {
            header('Content-Encoding: gzip');
        }

        // Handle sitemap index
        if ($sitemap_type === 'index') {
            $payload = $this->generate_sitemap_index();
            if ($use_gzip && function_exists('gzencode')) {
                $payload = gzencode($payload);
            }
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- XML output
            echo $payload;
            exit;
        }
        
        // Handle individual sitemaps
        $providers = $this->get_providers();
        if (isset($providers[$sitemap_type]) && is_object($providers[$sitemap_type])) {
            $provider = $providers[$sitemap_type];
            $page = (int) get_query_var('prorank_sitemap_page', 1);
            
            if (method_exists($provider, 'is_enabled') && $provider->is_enabled()) {
                // Validate page number
                if (method_exists($provider, 'get_total_pages')) {
                    $total_pages = $provider->get_total_pages();
                    if ($page > $total_pages) {
                        $wp_query->set_404();
                        status_header(404);
                        return;
                    }
                }
                
                // Generate sitemap with caching
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only refresh flag.
                $force_refresh = isset($_GET['refresh']) && sanitize_text_field( wp_unslash( $_GET['refresh'] ) ) === '1';

                if (method_exists($provider, 'generate_cached')) {
                    // Use cached version (2025 enhancement)
                    $payload = $provider->generate_cached($page, $force_refresh);
                    if ($use_gzip && function_exists('gzencode')) {
                        $payload = gzencode($payload);
                    }
                    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- XML output
                    echo $payload;
                    exit;
                } elseif (method_exists($provider, 'generate_xml')) {
                    // Fallback to direct generation
                    $payload = $provider->generate_xml($page);
                    if ($use_gzip && function_exists('gzencode')) {
                        $payload = gzencode($payload);
                    }
                    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- XML output
                    echo $payload;
                    exit;
                } elseif (method_exists($provider, 'generate')) {
                    // Backward compatibility
                    $payload = $provider->generate($page);
                    if ($use_gzip && function_exists('gzencode')) {
                        $payload = gzencode($payload);
                    }
                    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- XML output
                    echo $payload;
                    exit;
                }
            }
        }
        
        // If we get here, sitemap not found
        $wp_query->set_404();
        status_header(404);
    }
    
    /**
     * Generate sitemap index
     *
     * @return string XML content
     */
    private function generate_sitemap_index(): string {
        $xml = new \XMLWriter();
        $xml->openMemory();
        $xml->setIndent(true);
        
        // Start XML document
        $xml->startDocument('1.0', 'UTF-8');
        
        // Add promotional comment
        $xml->writeComment(' ');
        $xml->writeComment(' ===========================================================');
        $xml->writeComment('  🚀 Generated by ProRank SEO - The Ultimate WordPress SEO Plugin');
        $xml->writeComment('  🌐 Visit: https://prorank.io');
        $xml->writeComment(' ===========================================================');
        $xml->writeComment(' ');
        
        // Start sitemapindex
        $xml->startElement('sitemapindex');
        $xml->writeAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');
        
        // Get all sitemap entries from providers
        foreach ($this->get_providers() as $slug => $provider) {
            // Skip if provider is not an object
            if (!is_object($provider)) {
                continue;
            }
            
            // Check if provider is enabled
            if (!method_exists($provider, 'is_enabled') || !$provider->is_enabled()) {
                continue;
            }
            
            // Get index entries
            if (method_exists($provider, 'get_index_entries')) {
                $entries = $provider->get_index_entries();
                
                foreach ($entries as $entry) {
                    $xml->startElement('sitemap');
                    $xml->writeElement('loc', $entry['loc']);
                    
                    if (!empty($entry['lastmod'])) {
                        $xml->writeElement('lastmod', $entry['lastmod']);
                    }
                    
                    $xml->endElement(); // sitemap
                }
            }
        }
        
        $xml->endElement(); // sitemapindex
        $xml->endDocument();
        
        return $xml->outputMemory();
    }
    
    /**
     * Add sitemap URL to robots.txt
     *
     * @param string $output Robots.txt output
     * @param bool   $public Whether the site is public
     * @return string Modified output
     */
    public function add_robots_txt_rules(string $output, bool $public): string {
        if (!$public) {
            return $output;
        }
        
        // Check if module is active through the plugin instance
        try {
            $plugin = $this->plugin();
            $module_manager = $plugin->modules();
            if (!$module_manager->is_active($this->slug)) {
                return $output;
            }
        } catch (\Exception $e) {
            // If we can't check, assume active since this method was called
            // This prevents errors during plugin conflicts
        }
        
        $sitemap_url = home_url('sitemap_index.xml');
        $output .= "\nSitemap: " . $sitemap_url . "\n";
        
        return $output;
    }
    
    /**
     * Flush rewrite rules
     *
     * @return void
     */
    public function flush_rewrite_rules(): void {
        flush_rewrite_rules();
    }
    
    /**
     * Maybe ping search engines on post status change
     *
     * @param string  $new_status New post status
     * @param string  $old_status Old post status
     * @param \WP_Post $post      Post object
     * @return void
     */
    public function maybe_ping_search_engines(string $new_status, string $old_status, \WP_Post $post): void {
        // Only ping if transitioning to published
        if ($new_status !== 'publish' || $old_status === 'publish') {
            return;
        }
        
        // Check if this post type is included in sitemaps
        $settings = $this->settings_manager ? $this->settings_manager->get_settings('sitemaps') : [];
        $enabled_post_types = $settings['include_post_types'] ?? ['post', 'page'];
        
        if (!in_array($post->post_type, $enabled_post_types, true)) {
            return;
        }
        
        $this->ping_search_engines();
    }
    
    /**
     * Ping search engines on term changes
     *
     * @param int    $term_id  Term ID
     * @param int    $tt_id    Term taxonomy ID
     * @param string $taxonomy Taxonomy slug
     * @return void
     */
    public function ping_on_term_change(int $term_id, int $tt_id, string $taxonomy): void {
        // Check if this taxonomy is included in sitemaps
        $settings = $this->settings_manager ? $this->settings_manager->get_settings('sitemaps') : [];
        $enabled_taxonomies = $settings['include_taxonomies'] ?? ['category', 'post_tag'];
        
        if (!in_array($taxonomy, $enabled_taxonomies, true)) {
            return;
        }
        
        $this->ping_search_engines();
    }
    
    /**
     * Ping search engines about sitemap update
     *
     * @return void
     */
    private function ping_search_engines(): void {
        // Check if pinging is enabled
        $settings = $this->settings_manager ? $this->settings_manager->get_settings('sitemaps') : [];
        if (empty($settings['ping_search_engines'])) {
            return;
        }
        
        // Don't ping more than once per hour
        $last_ping = get_transient('prorank_seo_sitemap_last_ping');
        if ($last_ping !== false) {
            return;
        }
        
        $sitemap_url = home_url('sitemap_index.xml');
        
        // Ping Google
        wp_remote_get(
            'https://www.google.com/ping?sitemap=' . urlencode($sitemap_url),
            [
                'timeout' => 10,
                'blocking' => false,
                'user-agent' => 'ProRank SEO/' . PRORANK_SEO_VERSION,
            ]
        );
        
        // Ping Bing
        wp_remote_get(
            'https://www.bing.com/ping?sitemap=' . urlencode($sitemap_url),
            [
                'timeout' => 10,
                'blocking' => false,
                'user-agent' => 'ProRank SEO/' . PRORANK_SEO_VERSION,
            ]
        );
        
        // Set transient to prevent over-pinging
        set_transient('prorank_seo_sitemap_last_ping', time(), HOUR_IN_SECONDS);
        
        // Log the ping
        if (defined('WP_DEBUG') && WP_DEBUG) {
            prorank_log('[ProRank SEO] Pinged search engines with sitemap: ' . $sitemap_url);
        }
    }
    
    /**
     * Maybe enable gzip compression for sitemap output
     *
     * Enables gzip compression if available and not already enabled.
     * 2025 Enhancement: Reduces sitemap transfer size by ~70-90%.
     *
     * @return bool True if gzip was enabled, false otherwise
     */
    private function maybe_enable_gzip_compression(): bool {
        // Check if gzip is already enabled
        if (ini_get('zlib.output_compression')) {
            return false;
        }

        // Check if zlib extension is loaded
        if (!extension_loaded('zlib')) {
            return false;
        }

        // Check if client accepts gzip
        $accept_encoding = \prorank_get_server_var( 'HTTP_ACCEPT_ENCODING' );
        if (stripos($accept_encoding, 'gzip') === false) {
            return false;
        }

        // Check if gzip compression is enabled in settings
        $settings = $this->settings_manager ? $this->settings_manager->get_settings('sitemaps') : [];
        $gzip_enabled = $settings['enable_gzip'] ?? true; // Default: enabled

        if (!$gzip_enabled) {
            return false;
        }

        if (!function_exists('gzencode')) {
            return false;
        }

        return true;
    }

    /**
     * Invalidate sitemap cache on content changes
     *
     * Called when posts are saved, deleted, or status changes.
     *
     * @param int      $post_id Post ID
     * @param \WP_Post $post    Post object
     * @return void
     */
    public function invalidate_sitemap_cache_on_save(int $post_id, \WP_Post $post): void {
        // Skip autosaves and revisions
        if (wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) {
            return;
        }

        // Skip if post type is not included in sitemaps
        $settings = $this->settings_manager ? $this->settings_manager->get_settings('sitemaps') : [];
        $enabled_post_types = $settings['include_post_types'] ?? ['post', 'page'];

        if (!in_array($post->post_type, $enabled_post_types, true)) {
            return;
        }

        // Invalidate all sitemap caches
        \ProRank\SEO\Modules\Indexing\Sitemaps\BaseSitemapProvider::invalidate_all_caches();

        // Log cache invalidation
        if (defined('WP_DEBUG') && WP_DEBUG) {
            prorank_log(sprintf(
                '[ProRank SEO] Sitemap cache invalidated due to post save: %s (ID: %d)',
                $post->post_title,
                $post_id
            ));
        }
    }

    /**
     * Invalidate sitemap cache on post deletion
     *
     * @param int $post_id Post ID
     * @return void
     */
    public function invalidate_sitemap_cache_on_delete(int $post_id): void {
        // Get post before it's deleted
        $post = get_post($post_id);
        if (!$post) {
            return;
        }

        // Skip if post type is not included in sitemaps
        $settings = $this->settings_manager ? $this->settings_manager->get_settings('sitemaps') : [];
        $enabled_post_types = $settings['include_post_types'] ?? ['post', 'page'];

        if (!in_array($post->post_type, $enabled_post_types, true)) {
            return;
        }

        // Invalidate all sitemap caches
        \ProRank\SEO\Modules\Indexing\Sitemaps\BaseSitemapProvider::invalidate_all_caches();
    }

    /**
     * Invalidate sitemap cache on term changes
     *
     * @param int    $term_id  Term ID
     * @param int    $tt_id    Term taxonomy ID
     * @param string $taxonomy Taxonomy slug
     * @return void
     */
    public function invalidate_sitemap_cache_on_term_change(int $term_id, int $tt_id, string $taxonomy): void {
        // Skip if taxonomy is not included in sitemaps
        $settings = $this->settings_manager ? $this->settings_manager->get_settings('sitemaps') : [];
        $enabled_taxonomies = $settings['include_taxonomies'] ?? ['category', 'post_tag'];

        if (!in_array($taxonomy, $enabled_taxonomies, true)) {
            return;
        }

        // Invalidate all sitemap caches
        \ProRank\SEO\Modules\Indexing\Sitemaps\BaseSitemapProvider::invalidate_all_caches();
    }

    /**
     * Get module configuration
     *
     * @return array
     */
    public function get_config(): array {
        return [
            'name' => __('XML Sitemaps', 'prorank-seo'),
            'description' => __('Automatic XML sitemap generation', 'prorank-seo'),
            'icon' => 'dashicons-editor-ul',
            'tier' => 'free',
            'parent' => 'technical-seo',
            'settings_group' => 'sitemaps',
            'features' => [
                'base' => [
                    'tier' => 'free',
                    'items' => [
                        __('Post sitemaps', 'prorank-seo'),
                        __('Page sitemaps', 'prorank-seo'),
                        __('Category sitemaps', 'prorank-seo'),
                        __('Tag sitemaps', 'prorank-seo'),
                        __('Image sitemaps', 'prorank-seo'),
                    ],
                ],
            ],
        ];
    }
}
