<?php
/**
 * Resource Hints Module
 *
 * Adds DNS prefetch, preconnect, and preload hints for improved performance
 *
 * @package ProRank\SEO\Modules\Performance
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\Performance;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Modules\BaseModule;

/**
 * ResourceHintsModule class
 */
class ResourceHintsModule extends BaseModule {

    /**
     * Required tier for this module
     */
    protected string $feature_tier = 'free';
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->slug = 'resource_hints';
        $this->name = 'Resource Hints';
        $this->description = 'Add DNS prefetch, preconnect, and preload hints for faster loading';
$this->parent_slug = 'performance';
    }

    /**
     * Get setting from unified cache settings
     * Overrides BaseModule to read from prorank_cache_settings
     *
     * @param string $key Setting key
     * @param mixed $default Default value
     * @return mixed
     */
    protected function get_setting(string $key, $default = null) {
        static $settings = null;
        if ($settings === null) {
            $settings = get_option('prorank_cache_settings', []);
        }
        return $settings[$key] ?? $default;
    }

    /**
     * Initialize module hooks
     *
     * @return void
     */
    public function init_hooks(): void {
        // Check if module should run
        if (!$this->should_run()) {
            return;
        }
        
        // DNS Prefetch
        if ($this->get_setting('dns_prefetch_enabled', false)) {
            add_filter('wp_resource_hints', [$this, 'add_dns_prefetch_hints'], 10, 2);
        }
        
        // Font Preloading
        if ($this->get_setting('font_preload_enabled', false)) {
            add_action('wp_head', [$this, 'add_font_preload_links'], 2);
        }
        
        // Link Preloading
        if ($this->get_setting('link_preload_enabled', false)) {
            add_action('wp_enqueue_scripts', [$this, 'enqueue_link_preload_script']);
        }
    }
    
    /**
     * Check if module should run
     *
     * @return bool
     */
    private function should_run(): bool {
        // Check if module is enabled
        if (!$this->is_enabled()) {
            return false;
        }
        
        // Don't run in admin
        if (is_admin()) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Add DNS prefetch hints
     *
     * @param array  $hints Array of resource hints
     * @param string $relation_type Type of resource hint
     * @return array Modified hints array
     */
    public function add_dns_prefetch_hints(array $hints, string $relation_type): array {
        if ($relation_type !== 'dns-prefetch') {
            return $hints;
        }
        
        $hosts = $this->get_setting('dns_prefetch_hosts', '');
        if (empty($hosts)) {
            return $hints;
        }
        
        // Parse hosts from textarea
        $hosts_array = array_filter(array_map('trim', explode("\n", $hosts)));
        
        foreach ($hosts_array as $host) {
            // Validate host
            if ($this->is_valid_host($host)) {
                // Ensure protocol is removed
                $host = str_replace(['http://', 'https://'], '', $host);
                $host = rtrim($host, '/');
                
                if (!in_array($host, $hints, true)) {
                    $hints[] = $host;
                }
            }
        }
        
        return $hints;
    }
    
    /**
     * Add font preload links
     *
     * @return void
     */
    public function add_font_preload_links(): void {
        $font_urls = $this->get_setting('font_preload_urls', '');
        if (empty($font_urls)) {
            return;
        }
        
        // Parse URLs from textarea
        $urls_array = array_filter(array_map('trim', explode("\n", $font_urls)));
        
        foreach ($urls_array as $url) {
            if ($this->is_valid_font_url($url)) {
                // Build link tag with optional crossorigin attribute
                if ($this->is_external_url($url)) {
                    printf(
                        '<link rel="preload" href="%s" as="font" type="font/%s" crossorigin="anonymous">%s',
                        esc_url($url),
                        esc_attr($this->get_font_type($url)),
                        "\n"
                    );
                } else {
                    printf(
                        '<link rel="preload" href="%s" as="font" type="font/%s">%s',
                        esc_url($url),
                        esc_attr($this->get_font_type($url)),
                        "\n"
                    );
                }
            }
        }
    }
    
    /**
     * Enqueue link preload script
     *
     * @return void
     */
    public function enqueue_link_preload_script(): void {
        $delay = $this->get_setting('link_preload_delay', 50);
        
        $script = sprintf(
            '
            (function() {
                var delay = %d;
                var preloaded = new Set();
                var prefetcher = document.createElement("link");
                
                function preloadLink(url) {
                    if (preloaded.has(url)) return;
                    
                    preloaded.add(url);
                    var link = document.createElement("link");
                    link.rel = "prefetch";
                    link.href = url;
                    document.head.appendChild(link);
                }
                
                function handleHover(e) {
                    var link = e.target.closest("a");
                    if (!link) return;
                    
                    var url = link.href;
                    if (!url || url.indexOf("#") === 0) return;
                    
                    // Skip external links
                    if (link.hostname !== window.location.hostname) return;
                    
                    // Skip non-HTML links
                    if (link.pathname.match(/\.(jpg|jpeg|png|gif|svg|webp|ico|pdf|zip|rar|gz|7z|exe|dmg|mp3|mp4|avi|mov)$/i)) return;
                    
                    // Skip admin links
                    if (link.pathname.indexOf("/wp-admin/") === 0) return;
                    
                    clearTimeout(link.preloadTimeout);
                    link.preloadTimeout = setTimeout(function() {
                        preloadLink(url);
                    }, delay);
                }
                
                function handleLeave(e) {
                    var link = e.target.closest("a");
                    if (!link) return;
                    
                    clearTimeout(link.preloadTimeout);
                }
                
                // Attach event listeners
                document.addEventListener("mouseover", handleHover, { passive: true });
                document.addEventListener("mouseout", handleLeave, { passive: true });
                
                // Mobile: preload on touchstart
                document.addEventListener("touchstart", function(e) {
                    var link = e.target.closest("a");
                    if (!link) return;
                    
                    var url = link.href;
                    if (url && url.indexOf("#") !== 0 && link.hostname === window.location.hostname) {
                        preloadLink(url);
                    }
                }, { passive: true });
            })();
            ',
            intval($delay)
        );
        
        wp_add_inline_script('jquery', $script, 'after');
    }
    
    /**
     * Validate host
     *
     * @param string $host Host to validate
     * @return bool
     */
    private function is_valid_host(string $host): bool {
        $host = str_replace(['http://', 'https://'], '', $host);
        $host = rtrim($host, '/');
        
        // Basic validation
        return preg_match('/^[a-z0-9.-]+\.[a-z]{2,}$/i', $host) === 1;
    }
    
    /**
     * Validate font URL
     *
     * @param string $url URL to validate
     * @return bool
     */
    private function is_valid_font_url(string $url): bool {
        // Check if it's a valid URL or relative path
        if (!filter_var($url, FILTER_VALIDATE_URL) && !preg_match('/^\/[^\/]/', $url)) {
            return false;
        }
        
        // Check if it has a font extension
        return preg_match('/\.(woff2?|ttf|otf|eot)$/i', $url) === 1;
    }
    
    /**
     * Check if URL is external
     *
     * @param string $url URL to check
     * @return bool
     */
    private function is_external_url(string $url): bool {
        $site_host = wp_parse_url(home_url(), PHP_URL_HOST);
        $url_host = wp_parse_url($url, PHP_URL_HOST);
        
        return $url_host && $url_host !== $site_host;
    }
    
    /**
     * Get font type from URL
     *
     * @param string $url Font URL
     * @return string Font type
     */
    private function get_font_type(string $url): string {
        if (preg_match('/\.woff2$/i', $url)) {
            return 'woff2';
        } elseif (preg_match('/\.woff$/i', $url)) {
            return 'woff';
        } elseif (preg_match('/\.ttf$/i', $url)) {
            return 'truetype';
        } elseif (preg_match('/\.otf$/i', $url)) {
            return 'opentype';
        } elseif (preg_match('/\.eot$/i', $url)) {
            return 'embedded-opentype';
        }
        
        return 'woff2'; // Default
    }
}