<?php
// phpcs:disable WordPress.DB.SlowDBQuery
/**
 * Image Optimization Module
 *
 * Generates WebP and AVIF versions of images for better performance
 *
 * @package ProRank\SEO\Modules\Performance
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\Performance;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Modules\BaseModule;
use ProRank\SEO\Core\RestApi\ImageOptimizationController;
use ProRank\SEO\Core\ImageOptimization\OptimizationEngine;

/**
 * ImageOptimizationModule class
 */
class ImageOptimizationModule extends BaseModule {
    
    /**
     * Supported image formats
     *
     * @var array
     */
    private const SUPPORTED_FORMATS = ['jpg', 'jpeg', 'png'];
    
    /**
     * WebP quality default
     *
     * @var int
     */
    private const DEFAULT_WEBP_QUALITY = 75;
    
    /**
     * AVIF quality default
     *
     * @var int
     */
    private const DEFAULT_AVIF_QUALITY = 65;

    /**
     * JPEG XL quality default
     *
     * @var int
     */
    private const DEFAULT_JXL_QUALITY = 75;

    /**
     * Compression types
     *
     * @var array
     */
    private const COMPRESSION_TYPES = [
        'aggressive' => [
            'jpeg_quality' => 70,
            'webp_quality' => 65,
            'avif_quality' => 55,
            'jxl_quality' => 60,
        ],
        'lossy' => [
            'jpeg_quality' => 80,
            'webp_quality' => 75,
            'avif_quality' => 65,
            'jxl_quality' => 75,
        ],
        'glossy' => [
            'jpeg_quality' => 85,
            'webp_quality' => 82,
            'avif_quality' => 75,
            'jxl_quality' => 82,
        ],
        'lossless' => [
            'jpeg_quality' => 100,
            'webp_quality' => 100,
            'avif_quality' => 100,
            'jxl_quality' => 100,
        ],
    ];
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->slug = 'image_optimization';
        $this->name = 'Advanced Image Optimization';
        $this->description = 'Next-gen image optimization with WebP, AVIF, PDF support and AI-powered features';
        $this->feature_tier = 'free';
        $this->parent_slug = 'performance';
    }

    /**
     * Override settings lookup to prefer the dedicated option store.
     *
     * REST controllers persist settings into the `prorank_image_optimization_settings`
     * option. To keep compatibility with the SettingsManager, check that option first
     * and fall back to the base implementation when not present.
     *
     * @param string $key     Setting key.
     * @param mixed  $default Default value.
     * @return mixed
     */
    protected function get_setting(string $key, $default = null) {
        $option_settings = get_option('prorank_image_optimization_settings', []);

        if (is_array($option_settings) && array_key_exists($key, $option_settings)) {
            return $option_settings[$key];
        }

        return parent::get_setting($key, $default);
    }
    
    /**
     * Initialize module hooks
     *
     * @return void
     */
    public function init_hooks(): void {
        // Always register bulk/background optimization handlers (triggered by REST API)
        // These must be outside should_run() check so bulk optimization works
        add_action('prorank_bulk_optimize_images', [$this, 'process_bulk_optimization']);
        add_action('prorank_optimize_image', [$this, 'process_single_image_optimization']);
        add_action('prorank_bulk_optimization_continue', [$this, 'handle_bulk_optimization_continue'], 10, 2);

        // Check if module should run for other hooks
        if (!$this->should_run()) {
            return;
        }

        $skip_frontend = $this->is_request_excluded();
        
        // Hook into image upload for auto-optimization
        if ($this->get_setting('optimize_on_upload', true)) {
            add_filter('wp_generate_attachment_metadata', [$this, 'generate_optimized_images'], 10, 2);
        }
        
        // Hook to delete optimized images when original is deleted
        add_action('delete_attachment', [$this, 'delete_optimized_images']);
        
        // Hook for PNG to JPEG conversion
        if ($this->get_setting('png_to_jpeg', false)) {
            add_filter('wp_generate_attachment_metadata', [$this, 'convert_png_to_jpeg'], 9, 2);
        }
        
        // Hook for CMYK to RGB conversion
        if ($this->get_setting('cmyk_to_rgb', true)) {
            add_filter('wp_generate_attachment_metadata', [$this, 'convert_cmyk_to_rgb'], 8, 2);
        }
        
        // Hook for EXIF data management
        if ($this->get_setting('remove_exif', true)) {
            add_filter('wp_generate_attachment_metadata', [$this, 'manage_exif_data'], 7, 2);
        }
        
        // Hook for PDF optimization
        if ($this->get_setting('optimize_pdf', false)) {
            add_filter('wp_generate_attachment_metadata', [$this, 'optimize_pdf'], 10, 2);
        }
        
        // Hook for retina image generation
        if ($this->get_setting('generate_retina', false)) {
            add_filter('wp_generate_attachment_metadata', [$this, 'generate_retina_images'], 11, 2);
        }
        
        // Hook for AI training protection
        if ($this->get_setting('no_ai_training', false)) {
            add_action('add_attachment', [$this, 'add_ai_training_protection']);
            add_filter('wp_get_attachment_image_attributes', [$this, 'add_noai_attributes'], 10, 3);
        }
        
        // Hook for processing failed optimization queue
        add_action('prorank_retry_failed_optimizations', [$this, 'process_failed_optimization_queue']);
        
        // Get delivery method
        $delivery_method = $this->get_setting('delivery_method', 'picture_hooks');

        // 'picture' should use output buffering to catch ALL images (including Elementor)
        // 'picture_hooks' only works for images in the_content, which Elementor bypasses
        if ($delivery_method === 'picture') {
            $delivery_method = 'picture_global';
        }

        if ($delivery_method === 'picture_hooks' && !$skip_frontend) {
            // Filter to convert img to picture element (only for hooks mode)
            add_filter('wp_get_attachment_image', [$this, 'convert_to_picture_element'], 10, 5);

            // Filter for post content images
            add_filter('the_content', [$this, 'convert_content_images'], 99);

            // Filter for excerpt
            add_filter('the_excerpt', [$this, 'convert_content_images'], 99);

            // Filter for widget text
            add_filter('widget_text', [$this, 'convert_content_images'], 99);

            // Filter for custom HTML widget
            add_filter('widget_custom_html', [$this, 'convert_content_images'], 99);
        } elseif ($delivery_method === 'picture_global' && !$skip_frontend) {
            // Process entire output buffer - this catches ALL images including Elementor
            // Don't add wp_get_attachment_image filter to avoid double-wrapping
            add_filter('prorank_output_buffer', [$this, 'process_output_buffer'], 25);
        } elseif ($delivery_method === 'htaccess') {
            // Generate .htaccess rules
            add_action('init', [$this, 'setup_htaccess_rules']);
        } elseif ($delivery_method === 'accept_header' && !$skip_frontend) {
            // Server-side Accept header detection - replaces URLs directly
            // Works for CSS backgrounds and all images, not just img tags
            add_filter('prorank_output_buffer', [$this, 'process_accept_header_buffer'], 25);
        }

        // Client Hints support (response headers for DPR/Width/Viewport)
        if ($this->get_setting('client_hints_enabled', false)) {
            add_action('send_headers', [$this, 'send_client_hints_headers']);
        }
        
        // AJAX handler for bulk conversion
        add_action('wp_ajax_prorank_convert_images', [$this, 'handle_bulk_conversion']);

        // Note: prorank_optimize_image action is registered above should_run() check
        // to ensure Action Scheduler actions work even when module conditionals fail

        // Add custom image sizes for optimized formats
        add_action('init', [$this, 'register_image_sizes']);
        
        // Add rewrite rules for serving optimized images
        add_action('init', [$this, 'add_rewrite_rules']);
        
        // Schedule background optimization cron if enabled
        if ($this->get_setting('background_mode', false)) {
            add_action('init', [$this, 'schedule_background_optimization']);
            add_action('prorank_background_image_optimization', [$this, 'run_background_optimization']);
        }
        
        // Custom media folders admin menu
        if ($this->get_setting('custom_media_folders', false)) {
            add_action('admin_menu', [$this, 'add_custom_media_menu']);
        }
        
        // CDN integration
        if (!$skip_frontend && $this->get_setting('cdn_enabled', false)) {
            add_filter('wp_get_attachment_url', [$this, 'rewrite_cdn_url'], 10, 2);
            add_filter('wp_calculate_image_srcset', [$this, 'rewrite_srcset_cdn'], 10, 5);
            add_filter('the_content', [$this, 'rewrite_content_cdn_urls'], 100);
        }
    }
    
    /**
     * Check if module should run
     *
     * @return bool
     */
    private function should_run(): bool {
        // Check if module is enabled
        if (!$this->is_enabled()) {
            return false;
        }
        
        // Check if either WebP or AVIF is enabled
        return $this->get_setting('webp_enabled', false) ||
               $this->get_setting('avif_enabled', false) ||
               $this->get_setting('jxl_enabled', false);
    }

    /**
     * Check if current request path is excluded from front-end delivery.
     */
    private function is_request_excluded(): bool {
        if (is_admin()) {
            return false;
        }

        $raw = (string) $this->get_setting('exclude_paths', '');
        if ($raw === '') {
            return false;
        }

        $request_uri = \prorank_get_server_var('REQUEST_URI');
        if ($request_uri === '') {
            return false;
        }

        $path = (string) wp_parse_url($request_uri, PHP_URL_PATH);
        if ($path === '') {
            $path = $request_uri;
        }
        $path = '/' . ltrim($path, '/');

        foreach (preg_split('/\r\n|\r|\n/', $raw) as $pattern) {
            $pattern = trim((string) $pattern);
            if ($pattern === '') {
                continue;
            }
            if ($pattern[0] !== '/') {
                $pattern = '/' . $pattern;
            }

            if (strpos($pattern, '*') !== false) {
                $regex = '#^' . str_replace('\\*', '.*', preg_quote($pattern, '#')) . '$#';
                if (preg_match($regex, $path)) {
                    return true;
                }
                continue;
            }

            if (str_starts_with($path, $pattern)) {
                return true;
            }
        }

        return false;
    }
    
    /**
     * Generate optimized images when attachment is created
     *
     * @param array $metadata      Attachment metadata
     * @param int   $attachment_id Attachment ID
     * @return array Modified metadata
     */
    public function generate_optimized_images(array $metadata, int $attachment_id): array {
        // Check if it's an image
        if (!wp_attachment_is_image($attachment_id)) {
            return $metadata;
        }

        // Get file info
        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return $metadata;
        }

        // Check if format is supported
        $extension = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        if (!in_array($extension, self::SUPPORTED_FORMATS)) {
            return $metadata;
        }

        // Check if file should be excluded
        if ($this->should_exclude_file($file_path, $attachment_id)) {
            return $metadata;
        }

        // Store original file size before any optimization
        $original_size = filesize($file_path);
        update_post_meta($attachment_id, '_prorank_original_size', $original_size);

        // Track format generation status
        $webp_generated = false;
        $avif_generated = false;
        $total_savings = 0;

        // Backup original if enabled
        if ($this->get_setting('backup_originals', true)) {
            $this->backup_original_image($file_path, $attachment_id);
        }

        // Optimize original + sizes with advanced engine (jpegli/pngquant/oxipng)
        $this->optimize_original_and_sizes($file_path, $attachment_id, $metadata);

        // Generate WebP if enabled
        if ($this->get_setting('webp_enabled', false)) {
            $webp_success = $this->generate_webp($file_path, $attachment_id);
            if ($webp_success) {
                $webp_generated = true;
                // Calculate savings from WebP
                $webp_path = $this->get_optimized_path($file_path, 'webp');
                if (file_exists($webp_path)) {
                    $webp_size = filesize($webp_path);
                    if ($webp_size < $original_size) {
                        $total_savings = max($total_savings, $original_size - $webp_size);
                    }
                }
            }

            // Also generate WebP for all sizes
            if (!empty($metadata['sizes'])) {
                $upload_dir = wp_upload_dir();
                $base_dir = dirname($file_path);

                foreach ($metadata['sizes'] as $size => $size_data) {
                    // Check if this size is excluded
                    if ($this->should_exclude_file($file_path, $attachment_id, $size)) {
                        continue;
                    }

                    $size_path = $base_dir . '/' . $size_data['file'];
                    if (file_exists($size_path)) {
                        if ($this->generate_webp($size_path, $attachment_id, $size)) {
                            $webp_generated = true;
                        }
                    }
                }
            }
        }

        // Generate AVIF if enabled
        if ($this->get_setting('avif_enabled', false) && $this->is_avif_supported()) {
            $avif_success = $this->generate_avif($file_path, $attachment_id);
            if ($avif_success) {
                $avif_generated = true;
                // Calculate savings from AVIF (usually better than WebP)
                $avif_path = $this->get_optimized_path($file_path, 'avif');
                if (file_exists($avif_path)) {
                    $avif_size = filesize($avif_path);
                    if ($avif_size < $original_size) {
                        $total_savings = max($total_savings, $original_size - $avif_size);
                    }
                }
            }

            // Also generate AVIF for all sizes
            if (!empty($metadata['sizes'])) {
                $upload_dir = wp_upload_dir();
                $base_dir = dirname($file_path);

                foreach ($metadata['sizes'] as $size => $size_data) {
                    // Check if this size is excluded
                    if ($this->should_exclude_file($file_path, $attachment_id, $size)) {
                        continue;
                    }

                    $size_path = $base_dir . '/' . $size_data['file'];
                    if (file_exists($size_path)) {
                        if ($this->generate_avif($size_path, $attachment_id, $size)) {
                            $avif_generated = true;
                        }
                    }
                }
            }
        }

        // Generate JPEG XL if enabled
        $jxl_generated = false;
        if ($this->get_setting('jxl_enabled', false) && $this->can_generate_jxl()) {
            $jxl_success = $this->generate_jxl($file_path, $attachment_id);
            if ($jxl_success) {
                $jxl_generated = true;
                // Calculate savings from JXL (best compression)
                $jxl_path = $this->get_optimized_path($file_path, 'jxl');
                if (file_exists($jxl_path)) {
                    $jxl_size = filesize($jxl_path);
                    if ($jxl_size < $original_size) {
                        $total_savings = max($total_savings, $original_size - $jxl_size);
                    }
                }
            }

            // Also generate JXL for all sizes
            if (!empty($metadata['sizes'])) {
                $base_dir = dirname($file_path);

                foreach ($metadata['sizes'] as $size => $size_data) {
                    // Check if this size is excluded
                    if ($this->should_exclude_file($file_path, $attachment_id, $size)) {
                        continue;
                    }

                    $size_path = $base_dir . '/' . $size_data['file'];
                    if (file_exists($size_path)) {
                        if ($this->generate_jxl($size_path, $attachment_id, $size)) {
                            $jxl_generated = true;
                        }
                    }
                }
            }
        }

        // Store optimization status and per-attachment stats
        update_post_meta($attachment_id, '_prorank_optimized', '1');
        update_post_meta($attachment_id, '_prorank_optimization_date', current_time('mysql'));
        update_post_meta($attachment_id, '_prorank_bytes_saved', $total_savings);
        update_post_meta($attachment_id, '_prorank_webp_generated', $webp_generated ? '1' : '');
        update_post_meta($attachment_id, '_prorank_avif_generated', $avif_generated ? '1' : '');
        update_post_meta($attachment_id, '_prorank_jxl_generated', $jxl_generated ? '1' : '');

        return $metadata;
    }

    /**
     * Optimize original image and all generated sizes using the advanced engine.
     *
     * @param string $file_path     Original file path.
     * @param int    $attachment_id Attachment ID.
     * @param array  $metadata      Attachment metadata.
     * @return void
     */
    private function optimize_original_and_sizes(string $file_path, int $attachment_id, array $metadata): void {
        $engine = new OptimizationEngine();
        $options = [
            'compression_type' => $this->get_setting('compression_type', 'glossy'),
            'smart_compression' => $this->get_setting('smart_compression', true),
            'jpegli_enabled' => $this->get_setting('jpegli_enabled', false),
        ];

        $engine->optimize_file_path($file_path, $options);

        if (!empty($metadata['sizes'])) {
            $base_dir = dirname($file_path);
            foreach ($metadata['sizes'] as $size => $size_data) {
                if ($this->should_exclude_file($file_path, $attachment_id, $size)) {
                    continue;
                }
                $size_path = $base_dir . '/' . $size_data['file'];
                if (file_exists($size_path)) {
                    $engine->optimize_file_path($size_path, $options);
                }
            }
        }
    }

    /**
     * Send Client Hints headers to enable DPR/Width/Viewport hints.
     *
     * @return void
     */
    public function send_client_hints_headers(): void {
        if (is_admin() || headers_sent()) {
            return;
        }

        $hints = [];
        if ($this->get_setting('client_hints_dpr', true)) {
            $hints[] = 'DPR';
        }
        if ($this->get_setting('client_hints_width', true)) {
            $hints[] = 'Width';
        }
        if ($this->get_setting('client_hints_viewport', true)) {
            $hints[] = 'Viewport-Width';
        }
        if ($this->get_setting('client_hints_save_data', false)) {
            $hints[] = 'Save-Data';
        }

        if (!$hints) {
            return;
        }

        header('Accept-CH: ' . implode(', ', $hints));
        header('Accept-CH-Lifetime: 86400');
        header('Vary: ' . implode(', ', array_merge(['Accept'], $hints)), false);
    }
    
    /**
     * Generate WebP version of image
     *
     * @param string $source_path   Source image path
     * @param int    $attachment_id Attachment ID
     * @param string $size          Image size name
     * @return bool Success status
     */
    private function generate_webp(string $source_path, int $attachment_id, string $size = ''): bool {
        // Check if GD or Imagick supports WebP
        if (!$this->can_generate_webp()) {
            return false;
        }
        
        $webp_path = $this->get_optimized_path($source_path, 'webp');
        
        // Skip if already exists
        if (file_exists($webp_path)) {
            return true;
        }
        
        // Create directory if needed
        $dir = dirname($webp_path);
        if (!is_dir($dir)) {
            wp_mkdir_p($dir);
        }
        
        // Get quality setting based on compression type
        $compression_type = $this->get_setting('compression_type', 'lossy');
        $quality = self::COMPRESSION_TYPES[$compression_type]['webp_quality'] ?? self::DEFAULT_WEBP_QUALITY;
        
        // Override with custom quality if SmartCompression is disabled
        if (!$this->get_setting('smart_compression', true)) {
            $quality = $this->get_setting('webp_quality', self::DEFAULT_WEBP_QUALITY);
        }
        
        // Try to generate WebP
        $success = false;
        
        // Try Imagick first
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick($source_path);
                $imagick->setImageFormat('webp');
                $imagick->setImageCompressionQuality($quality);
                $success = $imagick->writeImage($webp_path);
                $imagick->destroy();
            } catch (\Exception $e) {
                // Fall back to GD
            }
        }
        
        // Try GD if Imagick failed
        if (!$success && function_exists('imagewebp')) {
            $image = $this->load_image_gd($source_path);
            if ($image) {
                $success = imagewebp($image, $webp_path, $quality);
                imagedestroy($image);
            }
        }
        
        if ($success) {
            // Store metadata
            $meta_key = '_prorank_webp_path' . ($size ? '_' . $size : '');
            update_post_meta($attachment_id, $meta_key, $webp_path);
            
            // Update statistics
            $this->update_optimization_stats($source_path, $webp_path, 'webp');
        }
        
        return $success;
    }
    
    /**
     * Check server WebP support
     *
     * @return array Support information
     */
    public function check_webp_support(): array {
        $support = [
            'gd' => false,
            'imagick' => false,
            'details' => [],
        ];
        
        // Check GD
        if (function_exists('gd_info')) {
            $gd_info = gd_info();
            $support['gd'] = !empty($gd_info['WebP Support']);
            $support['details']['gd_version'] = $gd_info['GD Version'] ?? 'Unknown';
        }
        
        // Check Imagick
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick();
                $formats = $imagick->queryFormats('WEBP');
                $support['imagick'] = !empty($formats);
                $support['details']['imagick_version'] = \Imagick::getVersion()['versionString'] ?? 'Unknown';
            } catch (\Exception $e) {
                $support['details']['imagick_error'] = esc_html($e->getMessage());
            }
        }
        
        return $support;
    }
    
    /**
     * Generate AVIF version of image
     *
     * @param string $source_path   Source image path
     * @param int    $attachment_id Attachment ID
     * @param string $size          Image size name
     * @return bool Success status
     */
    private function generate_avif(string $source_path, int $attachment_id, string $size = ''): bool {
        // Check if system supports AVIF
        if (!$this->can_generate_avif()) {
            return false;
        }
        
        $avif_path = $this->get_optimized_path($source_path, 'avif');
        
        // Skip if already exists
        if (file_exists($avif_path)) {
            return true;
        }
        
        // Create directory if needed
        $dir = dirname($avif_path);
        if (!is_dir($dir)) {
            wp_mkdir_p($dir);
        }
        
        // Get quality setting
        $quality = $this->get_setting('avif_quality', self::DEFAULT_AVIF_QUALITY);
        
        // Try to generate AVIF
        $success = false;
        
        // Try Imagick
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick($source_path);
                // Check if Imagick supports AVIF
                $formats = $imagick->queryFormats('AVIF');
                if (!empty($formats)) {
                    $imagick->setImageFormat('avif');
                    $imagick->setImageCompressionQuality($quality);
                    $success = $imagick->writeImage($avif_path);
                }
                $imagick->destroy();
            } catch (\Exception $e) {
                // AVIF not supported
            }
        }
        
        // Try GD if available (PHP 8.1+)
        if (!$success && function_exists('imageavif')) {
            $image = $this->load_image_gd($source_path);
            if ($image) {
                $success = imageavif($image, $avif_path, $quality);
                imagedestroy($image);
            }
        }
        
        if ($success) {
            // Store metadata
            $meta_key = '_prorank_avif_path' . ($size ? '_' . $size : '');
            update_post_meta($attachment_id, $meta_key, $avif_path);

            // Update statistics
            $this->update_optimization_stats($source_path, $avif_path, 'avif');
        }

        return $success;
    }

    /**
     * Generate JPEG XL version of an image
     *
     * @param string $source_path   Source image path
     * @param int    $attachment_id Attachment ID
     * @param string $size          Image size name
     * @return bool Success status
     */
    private function generate_jxl(string $source_path, int $attachment_id, string $size = ''): bool {
        $jxl_path = $this->get_optimized_path($source_path, 'jxl');

        // Skip if already exists
        if (file_exists($jxl_path)) {
            return true;
        }

        // Create directory if needed
        $dir = dirname($jxl_path);
        if (!is_dir($dir)) {
            wp_mkdir_p($dir);
        }

        // Get quality setting
        $compression_type = $this->get_setting('compression_type', 'lossy');
        $quality = self::COMPRESSION_TYPES[$compression_type]['jxl_quality'] ?? self::DEFAULT_JXL_QUALITY;

        $success = false;

        // Local processing only
        // Try Imagick (requires libjxl support)
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick($source_path);
                $formats = $imagick->queryFormats('JXL');
                if (!empty($formats)) {
                    $imagick->setImageFormat('jxl');
                    $imagick->setImageCompressionQuality($quality);
                    $success = $imagick->writeImage($jxl_path);
                }
                $imagick->destroy();
            } catch (\Exception $e) {
                // JXL not supported
            }
        }

        // Try cjxl CLI tool
        if (!$success && $this->has_jxl_cli()) {
            $distance = $this->quality_to_jxl_distance($quality);
            $cmd = sprintf(
                'cjxl -d %s "%s" "%s" 2>/dev/null',
                escapeshellarg($distance),
                escapeshellarg($source_path),
                escapeshellarg($jxl_path)
            );
            exec($cmd, $output, $return_var);
            $success = ($return_var === 0 && file_exists($jxl_path));
        }

        if ($success) {
            $meta_key = '_prorank_jxl_path' . ($size ? '_' . $size : '');
            update_post_meta($attachment_id, $meta_key, $jxl_path);
            $this->update_optimization_stats($source_path, $jxl_path, 'jxl');
        }

        return $success;
    }

    /**
     * Convert quality percentage to JXL distance value
     * JXL uses distance where 0 = lossless, higher = more compression
     *
     * @param int $quality Quality 0-100
     * @return string Distance value
     */
    private function quality_to_jxl_distance(int $quality): string {
        if ($quality >= 100) {
            return '0'; // Lossless
        }
        // Map 0-99 quality to 0.1-3.0 distance (lower quality = higher distance)
        $distance = 0.1 + ((100 - $quality) / 100) * 2.9;
        return number_format($distance, 2);
    }

    /**
     * Check if cjxl CLI tool is available
     *
     * @return bool
     */
    private function has_jxl_cli(): bool {
        static $has_cli = null;
        if ($has_cli === null) {
            exec('which cjxl 2>/dev/null', $output, $return_var);
            $has_cli = ($return_var === 0);
        }
        return $has_cli;
    }

    /**
     * Load image using GD
     *
     * @param string $path Image path
     * @return resource|false GD image resource
     */
    private function load_image_gd(string $path) {
        $info = wp_getimagesize($path);
        if (!$info) {
            return false;
        }
        
        switch ($info['mime']) {
            case 'image/jpeg':
                return imagecreatefromjpeg($path);
            case 'image/png':
                return imagecreatefrompng($path);
            default:
                return false;
        }
    }
    
    /**
     * Get optimized image path
     *
     * @param string $original_path Original image path
     * @param string $format        Target format (webp/avif)
     * @return string Optimized image path
     */
    public function get_optimized_path(string $original_path, string $format): string {
        $pathinfo = pathinfo($original_path);
        return $pathinfo['dirname'] . '/' . $pathinfo['filename'] . '.' . $format;
    }
    
    /**
     * Delete optimized images when original is deleted
     *
     * @param int $attachment_id Attachment ID
     * @return void
     */
    public function delete_optimized_images(int $attachment_id): void {
        // Get all optimized paths from metadata
        $meta_keys = $GLOBALS['wpdb']->get_col($GLOBALS['wpdb']->prepare(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
            "SELECT meta_key FROM {$GLOBALS['wpdb']->postmeta} 
             WHERE post_id = %d 
             AND meta_key LIKE '_prorank_%_path%'",
            $attachment_id
        ));
        
        foreach ($meta_keys as $meta_key) {
            $path = get_post_meta($attachment_id, $meta_key, true);
            if ($path && file_exists($path)) {
                @wp_delete_file($path);
            }
        }
    }
    
    /**
     * Add optimized formats to srcset
     *
     * @param array  $sources       Array of image sources
     * @param array  $size_array    Requested size
     * @param string $image_src     Image source URL
     * @param array  $image_meta    Image metadata
     * @param int    $attachment_id Attachment ID
     * @return array Modified sources
     */
    public function add_optimized_to_srcset(array $sources, array $size_array, string $image_src, array $image_meta, int $attachment_id): array {
        // Not used anymore - we're using picture element instead
        return $sources;
    }
    
    /**
     * Check if WebP generation is supported
     *
     * @return bool
     */
    private function can_generate_webp(): bool {
        // Check GD
        if (function_exists('imagewebp')) {
            return true;
        }
        
        // Check Imagick
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            $imagick = new \Imagick();
            $formats = $imagick->queryFormats('WEBP');
            return !empty($formats);
        }
        
        return false;
    }
    
    /**
     * Check if AVIF generation is supported
     *
     * @return bool
     */
    private function can_generate_avif(): bool {
        // Check GD (PHP 8.1+)
        if (function_exists('imageavif')) {
            return true;
        }
        
        // Check Imagick
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            $imagick = new \Imagick();
            $formats = $imagick->queryFormats('AVIF');
            return !empty($formats);
        }
        
        return false;
    }
    
    /**
     * Check if AVIF is supported by server
     *
     * @return bool
     */
    private function is_avif_supported(): bool {
        return $this->can_generate_avif();
    }

    /**
     * Check if JPEG XL generation is supported
     *
     * @return bool
     */
    private function can_generate_jxl(): bool {
        // Check Imagick for JXL support
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick();
                $formats = $imagick->queryFormats('JXL');
                if (!empty($formats)) {
                    return true;
                }
            } catch (\Exception $e) {
                // Continue to check CLI
            }
        }

        // Check for cjxl CLI tool
        return $this->has_jxl_cli();
    }
    
    /**
     * Update optimization statistics
     *
     * @param string $original_path Original file path
     * @param string $optimized_path Optimized file path
     * @param string $format Format type
     * @return void
     */
    private function update_optimization_stats(string $original_path, string $optimized_path, string $format): void {
        $original_size = filesize($original_path);
        $optimized_size = filesize($optimized_path);
        
        if ($original_size && $optimized_size) {
            $savings = $original_size - $optimized_size;
            $percentage = round(($savings / $original_size) * 100, 2);
            
            // Update global stats
            $stats = get_option('prorank_image_optimization_stats', [
                'total_images' => 0,
                'total_savings' => 0,
                'webp_count' => 0,
                'avif_count' => 0,
            ]);
            
            $stats['total_images']++;
            $stats['total_savings'] += $savings;
            $stats[$format . '_count']++;
            
            update_option('prorank_image_optimization_stats', $stats, false);
        }
    }
    
    /**
     * Handle bulk conversion AJAX request
     *
     * @return void
     */
    public function handle_bulk_conversion(): void {
        // Check nonce
        if (!check_ajax_referer('prorank_image_optimization', 'nonce', false)) {
            wp_send_json_error(__('Invalid nonce', 'prorank-seo'));
        }
        
        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'prorank-seo'));
        }
        
        // Get batch of images
        $args = [
            'post_type' => 'attachment',
            'post_mime_type' => 'image',
            'post_status' => 'inherit',
            'posts_per_page' => 5,
            'meta_query' => [
                [
                    'key' => '_prorank_optimized',
                    'compare' => 'NOT EXISTS',
                ],
            ],
        ];
        
        $attachments = get_posts($args);
        
        if (empty($attachments)) {
            wp_send_json_success([
                'complete' => true,
                'message' => __('All images have been optimized', 'prorank-seo'),
            ]);
        }
        
        $processed = 0;
        foreach ($attachments as $attachment) {
            $metadata = wp_get_attachment_metadata($attachment->ID);
            if ($metadata) {
                $this->generate_optimized_images($metadata, $attachment->ID);
                $processed++;
            }
        }
        
        // Get remaining count
        $remaining_args = $args;
        $remaining_args['posts_per_page'] = -1;
        $remaining_args['fields'] = 'ids';
        $remaining = count(get_posts($remaining_args));
        
        wp_send_json_success([
            'complete' => false,
            'processed' => $processed,
            'remaining' => $remaining,
            'message' => sprintf(
                /* translators: %d: number of images processed */
                __('Processed %d images', 'prorank-seo'),
                $processed
            ),
        ]);
    }
    
    /**
     * Register custom image sizes
     *
     * @return void
     */
    public function register_image_sizes(): void {
        // This is a placeholder for future custom sizes if needed
    }
    
    /**
     * Get optimization statistics
     *
     * @return array Statistics
     */
    public function get_stats(): array {
        $stats = get_option('prorank_image_optimization_stats', [
            'total_images' => 0,
            'total_savings' => 0,
            'webp_count' => 0,
            'avif_count' => 0,
            'bytes_saved' => 0,
        ]);

        // Add capability info
        $stats['can_generate_webp'] = $this->can_generate_webp();
        $stats['can_generate_avif'] = $this->can_generate_avif();
        $stats['can_generate_jxl'] = $this->can_generate_jxl();
        
        // Count optimized images
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Safe table name
        $optimized_count = $GLOBALS['wpdb']->get_var(
            "SELECT COUNT(DISTINCT post_id) FROM {$GLOBALS['wpdb']->postmeta} WHERE meta_key = '_prorank_optimized'"
        );

        $stats['optimized'] = (int) $optimized_count;
        $stats['optimized_images'] = (int) $optimized_count;

        // Get total image count
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Safe table name
        $total_count = $GLOBALS['wpdb']->get_var(
            "SELECT COUNT(*) FROM {$GLOBALS['wpdb']->posts} WHERE post_type = 'attachment' AND post_mime_type LIKE 'image/%'"
        );
        
        $stats['total'] = (int) $total_count;
        $stats['total_images'] = (int) $total_count;

        // Update cached stats with current total_images to prevent stale data
        $cached_stats = get_option('prorank_image_optimization_stats', []);
        if (!isset($cached_stats['total_images']) || $cached_stats['total_images'] !== $total_count) {
            $cached_stats['total_images'] = (int) $total_count;
            update_option('prorank_image_optimization_stats', $cached_stats, false);
        }

        // Normalize saved bytes and percent for UI
        if (!isset($stats['bytes_saved'])) {
            $stats['bytes_saved'] = $stats['total_savings'] ?? 0;
        }
        if (!isset($stats['optimization_percentage'])) {
            $stats['optimization_percentage'] = $stats['total'] > 0
                ? round(($stats['optimized'] / $stats['total']) * 100, 2)
                : 0;
        }

        // Add backup folder size if backups are enabled
        if ($this->get_setting('backup_originals', true)) {
            $upload_dir = wp_upload_dir();
            $backup_dir = $upload_dir['basedir'] . '/prorank-backups';
            
            if (is_dir($backup_dir)) {
                $backup_size = $this->get_directory_size($backup_dir);
                $stats['backup_size'] = $backup_size;
                $stats['backup_size_formatted'] = size_format($backup_size);
            }
        }
        
        return $stats;
    }
    
    /**
     * Convert img tag to picture element with WebP/AVIF sources
     *
     * @param string $html          Original HTML
     * @param int    $attachment_id Attachment ID
     * @param string $size          Image size
     * @param bool   $icon          Whether it's an icon
     * @param array  $attr          Image attributes
     * @return string Modified HTML
     */
    public function convert_to_picture_element(string $html, int $attachment_id, string $size, bool $icon, array $attr): string {
        // Skip if delivery is disabled
        if (!$this->get_setting('enable_picture_element', true)) {
            return $html;
        }
        
        // Skip icons
        if ($icon) {
            return $html;
        }
        
        // Skip if no attachment ID
        if (!$attachment_id) {
            return $html;
        }
        
        // Get image metadata
        $metadata = wp_get_attachment_metadata($attachment_id);
        if (!$metadata) {
            return $html;
        }
        
        // Parse the img tag
        $doc = new \DOMDocument();
        @$doc->loadHTML('<?xml encoding="UTF-8">' . $html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        $img = $doc->getElementsByTagName('img')->item(0);
        
        if (!$img) {
            return $html;
        }
        
        // Get original src
        $original_src = $img->getAttribute('src');
        $srcset = $img->getAttribute('srcset');
        $sizes = $img->getAttribute('sizes');
        
        // Check if optimized versions exist
        $has_webp = false;
        $has_avif = false;
        $webp_srcset = [];
        $avif_srcset = [];
        
        // Build srcsets for optimized formats
        if ($srcset) {
            $srcset_parts = explode(',', $srcset);
            foreach ($srcset_parts as $part) {
                $part = trim($part);
                if (preg_match('/(\S+)\s+(\d+w)/', $part, $matches)) {
                    $url = $matches[1];
                    $descriptor = $matches[2];

                    // Generate WebP URL (handles WordPress suffix like -2, -3)
                    $webp_url = $this->get_optimized_url($url, 'webp');
                    $actual_webp_url = $this->get_actual_optimized_url($webp_url);
                    if ($actual_webp_url) {
                        $webp_srcset[] = $actual_webp_url . ' ' . $descriptor;
                        $has_webp = true;
                    }

                    // Generate AVIF URL
                    $avif_url = $this->get_optimized_url($url, 'avif');
                    $actual_avif_url = $this->get_actual_optimized_url($avif_url);
                    if ($actual_avif_url) {
                        $avif_srcset[] = $actual_avif_url . ' ' . $descriptor;
                        $has_avif = true;
                    }
                }
            }
        }

        // Check main image - get actual URLs with WordPress suffixes
        $webp_src = $this->get_optimized_url($original_src, 'webp');
        $actual_webp_src = $this->get_actual_optimized_url($webp_src);
        if ($actual_webp_src) {
            $webp_src = $actual_webp_src;
            $has_webp = true;
        }

        $avif_src = $this->get_optimized_url($original_src, 'avif');
        $actual_avif_src = $this->get_actual_optimized_url($avif_src);
        if ($actual_avif_src) {
            $avif_src = $actual_avif_src;
            $has_avif = true;
        }
        
        // If no optimized versions exist, return original
        if (!$has_webp && !$has_avif) {
            return $html;
        }
        
        // Create picture element
        $picture = $doc->createElement('picture');
        
        // Add AVIF source if available
        if ($has_avif && $this->get_setting('avif_enabled', false)) {
            $avif_source = $doc->createElement('source');
            $avif_source->setAttribute('type', 'image/avif');
            $avif_source->setAttribute('srcset', $avif_src);
            
            if (!empty($avif_srcset)) {
                $avif_source->setAttribute('srcset', implode(', ', $avif_srcset));
            }
            
            if ($sizes) {
                $avif_source->setAttribute('sizes', $sizes);
            }
            
            $picture->appendChild($avif_source);
        }
        
        // Add WebP source if available
        if ($has_webp && $this->get_setting('webp_enabled', false)) {
            $webp_source = $doc->createElement('source');
            $webp_source->setAttribute('type', 'image/webp');
            $webp_source->setAttribute('srcset', $webp_src);
            
            if (!empty($webp_srcset)) {
                $webp_source->setAttribute('srcset', implode(', ', $webp_srcset));
            }
            
            if ($sizes) {
                $webp_source->setAttribute('sizes', $sizes);
            }
            
            $picture->appendChild($webp_source);
        }
        
        // Clone original img as fallback
        $fallback_img = $img->cloneNode(true);
        $picture->appendChild($fallback_img);
        
        // Replace img with picture
        $img->parentNode->replaceChild($picture, $img);
        
        // Get the modified HTML
        $body = $doc->getElementsByTagName('body')->item(0);
        if ($body) {
            $html = $doc->saveHTML($body);
            // Remove body tags
            $html = preg_replace('/<\/?body>/', '', $html);
        } else {
            $html = $doc->saveHTML();
        }
        
        return trim($html);
    }
    
    
    /**
     * Process output buffer to convert all images
     *
     * @param string $buffer Output buffer content
     * @return string Modified content
     */
    public function process_output_buffer(string $buffer): string {
        // Skip if not HTML
        if (!preg_match('/<html/i', $buffer)) {
            return $buffer;
        }

        // Skip admin pages
        if (is_admin()) {
            return $buffer;
        }

        // Process img tags, but skip those already inside picture elements
        // Use a more sophisticated approach: split by picture elements, process only outside
        return preg_replace_callback(
            '/(<picture[^>]*>.*?<\/picture>)|(<img[^>]+>)/is',
            function($matches) {
                // If it's a picture element, return unchanged
                if (!empty($matches[1])) {
                    return $matches[1];
                }
                // Otherwise process the img tag
                if (!empty($matches[2])) {
                    // Extract attributes from the img tag
                    if (preg_match('/<img([^>]+)>/i', $matches[2], $img_match)) {
                        return $this->process_img_tag($img_match);
                    }
                }
                return $matches[0];
            },
            $buffer
        );
    }


    /**
     * Process buffer using Accept header detection
     * Replaces image URLs with WebP/AVIF versions if browser supports them
     *
     * @param string $buffer Output buffer content
     * @return string Modified content
     */
    public function process_accept_header_buffer(string $buffer): string {
        // Skip if not HTML
        if (!preg_match('/<html/i', $buffer)) {
            return $buffer;
        }

        // Skip admin pages
        if (is_admin()) {
            return $buffer;
        }

        // Check browser support via Accept header
        $accepts_avif = $this->browser_accepts_avif();
        $accepts_webp = $this->browser_accepts_webp();

        // If browser doesn't support modern formats, return unchanged
        if (!$accepts_webp && !$accepts_avif) {
            return $buffer;
        }

        // Get upload directory info
        $upload_dir = wp_upload_dir();
        $upload_url = preg_quote($upload_dir['baseurl'], '/');

        // Match image URLs in the content (src, srcset, url(), etc.)
        $pattern = '/(' . $upload_url . '\/[^\s"\'\)]+\.(jpe?g|png))/i';

        return preg_replace_callback($pattern, function($matches) use ($accepts_avif, $accepts_webp) {
            $original_url = $matches[1];

            // Prefer AVIF if supported and enabled
            if ($accepts_avif && $this->get_setting('avif_enabled', false)) {
                $avif_url = $this->get_optimized_url($original_url, 'avif');
                $actual_avif = $this->get_actual_optimized_url($avif_url);
                if ($actual_avif) {
                    return $actual_avif;
                }
            }

            // Fall back to WebP
            if ($accepts_webp) {
                $webp_url = $this->get_optimized_url($original_url, 'webp');
                $actual_webp = $this->get_actual_optimized_url($webp_url);
                if ($actual_webp) {
                    return $actual_webp;
                }
            }

            // Return original if no optimized version exists
            return $original_url;
        }, $buffer);
    }

    /**
     * Check if browser accepts WebP via Accept header
     *
     * @return bool
     */
    private function browser_accepts_webp(): bool {
        $accept = \prorank_get_server_var( 'HTTP_ACCEPT' );
        return strpos($accept, 'image/webp') !== false;
    }

    /**
     * Check if browser accepts AVIF via Accept header
     *
     * @return bool
     */
    private function browser_accepts_avif(): bool {
        $accept = \prorank_get_server_var( 'HTTP_ACCEPT' );
        return strpos($accept, 'image/avif') !== false;
    }

    /**
     * Process individual img tag for picture element conversion
     *
     * @param array $matches Regex matches
     * @return string Modified HTML
     */
    private function process_img_tag(array $matches): string {
        $img_html = $matches[0];
        $attributes = $matches[1];
        
        // Extract src
        if (!preg_match('/src=["\']([^"\']*)["\']/i', $attributes, $src_matches)) {
            return $img_html;
        }
        
        $src = $src_matches[1];
        
        // Skip external images
        $upload_dir = wp_upload_dir();
        if (strpos($src, $upload_dir['baseurl']) === false) {
            return $img_html;
        }
        
        // Check if optimized versions exist (with WordPress suffix support)
        $has_webp = false;
        $has_avif = false;

        $webp_url = $this->get_optimized_url($src, 'webp');
        $actual_webp_url = $this->get_actual_optimized_url($webp_url);
        if ($actual_webp_url) {
            $webp_url = $actual_webp_url;
            $has_webp = true;
        }

        $avif_url = $this->get_optimized_url($src, 'avif');
        $actual_avif_url = $this->get_actual_optimized_url($avif_url);
        if ($actual_avif_url) {
            $avif_url = $actual_avif_url;
            $has_avif = true;
        }

        // If no optimized versions, return original
        if (!$has_webp && !$has_avif) {
            return $img_html;
        }

        // Build picture element
        $picture_html = '<picture>';

        // Add AVIF source
        if ($has_avif && $this->get_setting('avif_enabled', false)) {
            // Build srcset - prefer responsive srcset if available, otherwise single URL
            $avif_srcset = $avif_url;
            if (preg_match('/srcset=["\']([^"\']*)["\']/i', $attributes, $srcset_matches)) {
                $responsive_srcset = $this->convert_srcset_to_format($srcset_matches[1], 'avif');
                if ($responsive_srcset) {
                    $avif_srcset = $responsive_srcset;
                }
            }

            $picture_html .= '<source type="image/avif" srcset="' . esc_attr($avif_srcset) . '"';

            // Copy sizes if exists
            if (preg_match('/sizes=["\']([^"\']*)["\']/i', $attributes, $sizes_matches)) {
                $picture_html .= ' sizes="' . esc_attr($sizes_matches[1]) . '"';
            }

            $picture_html .= '>';
        }
        
        // Add WebP source
        if ($has_webp && $this->get_setting('webp_enabled', false)) {
            // Build srcset - prefer responsive srcset if available, otherwise single URL
            $webp_srcset = $webp_url;
            if (preg_match('/srcset=["\']([^"\']*)["\']/i', $attributes, $srcset_matches)) {
                $responsive_srcset = $this->convert_srcset_to_format($srcset_matches[1], 'webp');
                if ($responsive_srcset) {
                    $webp_srcset = $responsive_srcset;
                }
            }

            $picture_html .= '<source type="image/webp" srcset="' . esc_attr($webp_srcset) . '"';

            // Copy sizes if exists
            if (preg_match('/sizes=["\']([^"\']*)["\']/i', $attributes, $sizes_matches)) {
                $picture_html .= ' sizes="' . esc_attr($sizes_matches[1]) . '"';
            }

            $picture_html .= '>';
        }
        
        // Add original img as fallback
        $picture_html .= $img_html;
        $picture_html .= '</picture>';
        
        return $picture_html;
    }
    
    /**
     * Convert srcset to optimized format
     *
     * @param string $srcset Original srcset
     * @param string $format Target format (webp or avif)
     * @return string|null Converted srcset or null if no optimized versions
     */
    private function convert_srcset_to_format(string $srcset, string $format): ?string {
        $converted_parts = [];
        $parts = explode(',', $srcset);

        foreach ($parts as $part) {
            $part = trim($part);
            if (preg_match('/(\S+)\s+(\d+w|\d+x)/', $part, $matches)) {
                $url = $matches[1];
                $descriptor = $matches[2];

                $optimized_url = $this->get_optimized_url($url, $format);
                $actual_url = $this->get_actual_optimized_url($optimized_url);
                if ($actual_url) {
                    $converted_parts[] = $actual_url . ' ' . $descriptor;
                }
            }
        }

        return !empty($converted_parts) ? implode(', ', $converted_parts) : null;
    }
    
    /**
     * Setup .htaccess rules for serving optimized images
     *
     * @return void
     */
    public function setup_htaccess_rules(): void {
        // Check if server supports it
        if (!$this->server_supports_htaccess_delivery()) {
            return;
        }

        $htaccess_file = get_home_path() . '.htaccess';

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        if (!$wp_filesystem || !$wp_filesystem->is_writable($htaccess_file)) {
            return;
        }

        $rules = $this->generate_htaccess_rules();
        insert_with_markers($htaccess_file, 'ProRank Image Optimization', $rules);
    }
    
    /**
     * Generate .htaccess rules for image delivery
     *
     * @return array Array of rules
     */
    private function generate_htaccess_rules(): array {
        $rules = [];
        
        $rules[] = '<IfModule mod_rewrite.c>';
        $rules[] = 'RewriteEngine On';
        
        // AVIF rules
        if ($this->get_setting('avif_enabled', false)) {
            $rules[] = '';
            $rules[] = '# Serve AVIF images if available';
            $rules[] = 'RewriteCond %{HTTP_ACCEPT} image/avif';
            $rules[] = 'RewriteCond %{REQUEST_FILENAME} \.(jpe?g|png)$ [NC]';
            $rules[] = 'RewriteCond %{REQUEST_FILENAME}.avif -f';
            $rules[] = 'RewriteRule ^(.+)\.(jpe?g|png)$ $1.$2.avif [T=image/avif,L]';
        }
        
        // WebP rules
        if ($this->get_setting('webp_enabled', false)) {
            $rules[] = '';
            $rules[] = '# Serve WebP images if available';
            $rules[] = 'RewriteCond %{HTTP_ACCEPT} image/webp';
            $rules[] = 'RewriteCond %{REQUEST_FILENAME} \.(jpe?g|png)$ [NC]';
            $rules[] = 'RewriteCond %{REQUEST_FILENAME}.webp -f';
            $rules[] = 'RewriteRule ^(.+)\.(jpe?g|png)$ $1.$2.webp [T=image/webp,L]';
        }
        
        $rules[] = '</IfModule>';
        
        return $rules;
    }
    
    /**
     * Check if server supports .htaccess delivery
     *
     * @return bool
     */
    private function server_supports_htaccess_delivery(): bool {
        // Check if Apache
        $server_software = prorank_get_server_var( 'SERVER_SOFTWARE' );
        if ($server_software === '' || stripos($server_software, 'apache') === false) {
            return false;
        }
        
        // Check if mod_rewrite is loaded
        if (!function_exists('apache_get_modules')) {
            return true; // Assume it's available if we can't check
        }
        
        $modules = apache_get_modules();
        return in_array('mod_rewrite', $modules, true);
    }
    
    /**
     * Add rewrite rules for serving optimized images
     *
     * @return void
     */
    public function add_rewrite_rules(): void {
        // Add endpoint for checking optimized image availability
        add_rewrite_endpoint('prorank-image-check', EP_ROOT);
    }
    
    /**
     * Process single image optimization via Action Scheduler
     *
     * @param int $attachment_id Attachment ID to optimize
     * @return void
     */
    public function process_single_image_optimization(int $attachment_id): void {
        $metadata = wp_get_attachment_metadata($attachment_id);
        if (!$metadata) {
            return;
        }

        $progress = get_transient('prorank_image_optimization_progress');
        $file_path = get_attached_file($attachment_id);
        $current_image = $file_path ? basename($file_path) : ('ID ' . $attachment_id);

        // Ensure base structure
        if (!is_array($progress)) {
            $progress = [
                'current' => 0,
                'total' => 0,
                'bytes_saved' => 0,
                'log' => [],
            ];
        }

        // Mark current image
        $progress['current_image'] = $current_image;
        $progress['active_index'] = ($progress['current'] ?? 0) + 1;
        $progress['log'][] = [
            'image' => $current_image,
            'status' => 'processing',
            'ts' => time(),
        ];
        if (count($progress['log']) > 50) {
            $progress['log'] = array_slice($progress['log'], -50);
        }
        $progress['updated_at'] = time();
        set_transient('prorank_image_optimization_progress', $progress, HOUR_IN_SECONDS);

        // Update bulk job progress (if a bulk job is running)
        $this->record_bulk_job_progress($current_image, 0, false);

        $this->generate_optimized_images($metadata, $attachment_id);

        // Update progress after processing
        $saved_bytes = (int) get_post_meta($attachment_id, '_prorank_bytes_saved', true);
        $progress['current'] = isset($progress['current']) ? ((int) $progress['current'] + 1) : 1;
        $progress['bytes_saved'] = ($progress['bytes_saved'] ?? 0) + $saved_bytes;
        $progress['current_image'] = '';
        $progress['active_index'] = null;
        $progress['log'][] = [
            'image' => $current_image,
            'status' => 'done',
            'bytes_saved' => $saved_bytes,
            'ts' => time(),
        ];
        if (count($progress['log']) > 50) {
            $progress['log'] = array_slice($progress['log'], -50);
        }
        // Summary aggregation
        if (!isset($progress['summary']) || !is_array($progress['summary'])) {
            $progress['summary'] = ['processed' => 0, 'saved_bytes' => 0];
        }
        $progress['summary']['processed'] = ($progress['summary']['processed'] ?? 0) + 1;
        $progress['summary']['saved_bytes'] = ($progress['summary']['saved_bytes'] ?? 0) + $saved_bytes;

        $progress['completed'] = isset($progress['total'], $progress['current']) && $progress['total'] > 0
            ? $progress['current'] >= $progress['total']
            : false;
        if (!empty($progress['total'])) {
            $progress['percent'] = round(($progress['current'] / $progress['total']) * 100, 2);
        }

        $progress['updated_at'] = time();
        set_transient('prorank_image_optimization_progress', $progress, HOUR_IN_SECONDS);

        // Update bulk job progress with completion (if a bulk job is running)
        $this->record_bulk_job_progress($current_image, $saved_bytes, true);

        if (!empty($progress['completed'])) {
            delete_transient('prorank_image_conversion_running');
        }
    }

    /**
     * Handle bulk optimization continuation action.
     *
     * @param mixed $job_id Job ID or array with job_id/next_offset.
     * @param mixed $next_offset Next offset.
     * @return void
     */
    public function handle_bulk_optimization_continue($job_id = '', $next_offset = 0): void {
        if (is_array($job_id)) {
            $next_offset = $job_id['next_offset'] ?? 0;
            $job_id = $job_id['job_id'] ?? '';
        }

        $job_id = is_scalar($job_id) ? (string) $job_id : '';
        $next_offset = is_numeric($next_offset) ? (int) $next_offset : 0;

        if ($job_id === '') {
            return;
        }

        $controller = new ImageOptimizationController();
        $controller->handle_bulk_optimization_continue($job_id, $next_offset);
    }

    /**
     * Record progress for active bulk job and sync UI data.
     *
     * @param string $current_image Current image label.
     * @param int    $saved_bytes Bytes saved for this image.
     * @param bool   $completed Whether this image finished processing.
     * @return void
     */
    private function record_bulk_job_progress(string $current_image, int $saved_bytes, bool $completed): void {
        $job_id = (string) get_option('prorank_bulk_job_active', '');
        if ($job_id === '') {
            return;
        }

        $job_data = get_option('prorank_bulk_job_' . $job_id, []);
        if (!is_array($job_data) || empty($job_data)) {
            return;
        }

        if (!empty($job_data['status']) && in_array($job_data['status'], ['completed', 'stopped', 'failed'], true)) {
            return;
        }

        $log = $job_data['log'] ?? [];
        $log[] = [
            'image' => $current_image,
            'status' => $completed ? 'done' : 'processing',
            'bytes_saved' => $saved_bytes,
            'ts' => time(),
        ];
        if (count($log) > 50) {
            $log = array_slice($log, -50);
        }

        $update = [
            'current_image' => $completed ? '' : $current_image,
            'log' => $log,
            'status' => 'running',
        ];

        if ($completed) {
            $processed = (int) ($job_data['processed'] ?? 0) + 1;
            $bytes_total = (int) ($job_data['bytes_saved'] ?? 0) + $saved_bytes;
            $update['processed'] = $processed;
            $update['bytes_saved'] = $bytes_total;

            $total = (int) ($job_data['total'] ?? 0);
            if ($total > 0 && $processed >= $total) {
                $update['status'] = 'completed';
                $update['completed_at'] = time();
                update_option('prorank_bulk_job_last', $job_id, false);
                delete_option('prorank_bulk_job_active');
            }
        }

        $controller = new ImageOptimizationController();
        $controller->record_bulk_job_update($job_id, $update);
    }
    
    /**
     * Queue failed optimization for later retry
     *
     * @param string $file_path Source file path
     * @param string $format Target format
     * @param int $quality Quality setting
     * @param string $error_message Error message
     * @return void
     */
    private function queue_failed_optimization(string $file_path, string $format, int $quality, string $error_message): void {
        $queue = get_option('prorank_failed_optimizations', []);
        
        // Add to queue with metadata
        $queue[] = [
            'file_path' => $file_path,
            'format' => $format,
            'quality' => $quality,
            'error' => $error_message,
            'attempts' => 1,
            'queued_at' => current_time('mysql'),
            'last_attempt' => current_time('mysql'),
        ];
        
        // Keep only last 100 failed items
        if (count($queue) > 100) {
            $queue = array_slice($queue, -100);
        }
        
        update_option('prorank_failed_optimizations', $queue, false);
        
        // Schedule retry job if not already scheduled
        if (!wp_next_scheduled('prorank_retry_failed_optimizations')) {
            wp_schedule_single_event(time() + 3600, 'prorank_retry_failed_optimizations');
        }
    }
    
    /**
     * Process failed optimization queue
     *
     * @return void
     */
    public function process_failed_optimization_queue(): void {
        delete_option('prorank_failed_optimizations');
    }
    
    /**
     * Backup original image before optimization
     *
     * @param string $file_path     Original file path
     * @param int    $attachment_id Attachment ID
     * @return bool Success status
     */
    private function backup_original_image(string $file_path, int $attachment_id): bool {
        $upload_dir = wp_upload_dir();
        $backup_dir = $upload_dir['basedir'] . '/prorank-backups';
        
        // Create backup directory if it doesn't exist
        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);
            
            // Add .htaccess to protect backups
            $htaccess = $backup_dir . '/.htaccess';
            if (!file_exists($htaccess)) {
                file_put_contents($htaccess, "Order Deny,Allow\nDeny from all");
            }
        }
        
        // Create year/month subdirectory
        $time = current_time('mysql');
        $y = substr($time, 0, 4);
        $m = substr($time, 5, 2);
        $backup_subdir = $backup_dir . "/$y/$m";
        
        if (!file_exists($backup_subdir)) {
            wp_mkdir_p($backup_subdir);
        }
        
        // Generate backup filename
        $filename = basename($file_path);
        $backup_filename = $attachment_id . '-' . $filename;
        $backup_path = $backup_subdir . '/' . $backup_filename;
        
        // Copy file to backup location
        if (copy($file_path, $backup_path)) {
            // Store backup path in metadata
            update_post_meta($attachment_id, '_prorank_backup_path', $backup_path);
            return true;
        }
        
        return false;
    }
    
    /**
     * Schedule background optimization cron job
     *
     * @return void
     */
    public function schedule_background_optimization(): void {
        if (!wp_next_scheduled('prorank_background_image_optimization')) {
            wp_schedule_event(time(), 'hourly', 'prorank_background_image_optimization');
        }
    }
    
    /**
     * Run background optimization
     *
     * @return void
     */
    public function run_background_optimization(): void {
        // Check if already running
        if (get_transient('prorank_background_optimization_running')) {
            return;
        }
        
        // Set running flag
        set_transient('prorank_background_optimization_running', true, HOUR_IN_SECONDS);
        
        // Get unoptimized images
        $args = [
            'post_type' => 'attachment',
            'post_mime_type' => 'image',
            'post_status' => 'inherit',
            'posts_per_page' => 10, // Process 10 images per cron run
            'meta_query' => [
                [
                    'key' => '_prorank_optimized',
                    'compare' => 'NOT EXISTS',
                ],
            ],
        ];
        
        $attachments = get_posts($args);
        
        foreach ($attachments as $attachment) {
            $metadata = wp_get_attachment_metadata($attachment->ID);
            if ($metadata) {
                $this->generate_optimized_images($metadata, $attachment->ID);
            }
        }
        
        // Clear running flag
        delete_transient('prorank_background_optimization_running');
    }

    /**
     * Process bulk optimization (triggered by REST API)
     *
     * Handles the prorank_bulk_optimize_images cron action scheduled by
     * ImageOptimizationSettingsController::start_bulk_optimization()
     *
     * @return void
     */
    public function process_bulk_optimization(): void {
        // Get unoptimized images - process in batches of 20
        $batch_size = 20;
        $processed = 0;
        $max_execution = 50; // Max images per cron run

        do {
            $args = [
                'post_type' => 'attachment',
                'post_mime_type' => 'image',
                'post_status' => 'inherit',
                'posts_per_page' => $batch_size,
                'meta_query' => [
                    'relation' => 'OR',
                    [
                        'key' => '_prorank_optimized',
                        'compare' => 'NOT EXISTS',
                    ],
                    [
                        'key' => '_prorank_optimized',
                        'value' => '',
                    ],
                ],
            ];

            $attachments = get_posts($args);

            if (empty($attachments)) {
                break;
            }

            foreach ($attachments as $attachment) {
                // Check memory and time limits
                if (memory_get_usage(true) > (wp_convert_hr_to_bytes(ini_get('memory_limit')) * 0.8)) {
                    break 2;
                }

                $metadata = wp_get_attachment_metadata($attachment->ID);
                if ($metadata) {
                    $this->generate_optimized_images($metadata, $attachment->ID);
                } else {
                    // Mark as processed even without metadata to avoid infinite loop
                    update_post_meta($attachment->ID, '_prorank_optimized', '0');
                }

                $processed++;

                if ($processed >= $max_execution) {
                    break 2;
                }
            }
        } while (!empty($attachments) && $processed < $max_execution);

        // Check if more images remain
        $remaining = get_posts([
            'post_type' => 'attachment',
            'post_mime_type' => 'image',
            'post_status' => 'inherit',
            'posts_per_page' => 1,
            'fields' => 'ids',
            'meta_query' => [
                'relation' => 'OR',
                [
                    'key' => '_prorank_optimized',
                    'compare' => 'NOT EXISTS',
                ],
                [
                    'key' => '_prorank_optimized',
                    'value' => '',
                ],
            ],
        ]);

        if (!empty($remaining)) {
            // Schedule next batch
            if (!wp_next_scheduled('prorank_bulk_optimize_images')) {
                wp_schedule_single_event(time() + 30, 'prorank_bulk_optimize_images');
            }
        } else {
            // All done - clear running flag
            delete_transient('prorank_bulk_optimization_running');
        }
    }

    /**
     * Add custom media menu
     *
     * @return void
     */
    public function add_custom_media_menu(): void {
        add_media_page(
            __('Custom Media', 'prorank-seo'),
            __('Custom Media', 'prorank-seo'),
            'manage_options',
            'prorank-custom-media',
            [$this, 'render_custom_media_page']
        );
    }
    
    /**
     * Render custom media page
     *
     * @return void
     */
    public function render_custom_media_page(): void {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Custom Media Optimization', 'prorank-seo'); ?></h1>
            <p><?php echo esc_html__('Optimize images from custom folders not managed by WordPress Media Library.', 'prorank-seo'); ?></p>
            
            <div id="prorank-custom-media-root"></div>
        </div>
        <?php
    }
    
    /**
     * Restore original image from backup
     *
     * @param int $attachment_id Attachment ID
     * @return bool Success status
     */
    public function restore_from_backup(int $attachment_id): bool {
        $backup_path = get_post_meta($attachment_id, '_prorank_backup_path', true);
        
        if (!$backup_path || !file_exists($backup_path)) {
            return false;
        }
        
        $file_path = get_attached_file($attachment_id);
        if (!$file_path) {
            return false;
        }
        
        // Restore original file
        if (copy($backup_path, $file_path)) {
            // Delete optimized versions
            $this->delete_optimized_images($attachment_id);
            
            // Remove optimization flag
            delete_post_meta($attachment_id, '_prorank_optimized');
            
            // Regenerate thumbnails
            $metadata = wp_get_attachment_metadata($attachment_id);
            if ($metadata) {
                wp_update_attachment_metadata($attachment_id, $metadata);
            }
            
            return true;
        }
        
        return false;
    }
    
    /**
     * Get directory size recursively
     *
     * @param string $dir Directory path
     * @return int Size in bytes
     */
    private function get_directory_size(string $dir): int {
        $size = 0;
        
        if (!is_dir($dir)) {
            return $size;
        }
        
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($dir, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $size += $file->getSize();
            }
        }
        
        return $size;
    }
    
    /**
     * Delete all backup files
     *
     * @return bool Success status
     */
    public function delete_all_backups(): bool {
        $upload_dir = wp_upload_dir();
        $backup_dir = $upload_dir['basedir'] . '/prorank-backups';
        
        if (!is_dir($backup_dir)) {
            return true;
        }
        
        // Use WordPress filesystem API
        global $wp_filesystem;
        if (!function_exists('WP_Filesystem')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        
        WP_Filesystem();
        
        if ($wp_filesystem->exists($backup_dir)) {
            return $wp_filesystem->delete($backup_dir, true);
        }
        
        return true;
    }

    /**
     * Restore all images from backups
     *
     * @return array{restored:int,failed:int}|bool
     */
    public function restore_all_backups() {
        $args = [
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'posts_per_page' => -1,
            'fields'         => 'ids',
            'meta_query'     => [
                [
                    'key'     => '_prorank_backup_path',
                    'compare' => 'EXISTS',
                ],
            ],
        ];

        $ids = get_posts($args);
        $restored = 0;
        $failed = 0;

        foreach ($ids as $attachment_id) {
            $ok = $this->restore_from_backup((int) $attachment_id);
            if ($ok) {
                $restored++;
            } else {
                $failed++;
            }
        }

        return [
            'restored' => $restored,
            'failed' => $failed,
        ];
    }
    
    /**
     * Check if file should be excluded from optimization
     *
     * @param string $file_path     File path
     * @param int    $attachment_id Attachment ID
     * @param string $size          Image size (optional)
     * @return bool True if should be excluded
     */
    private function should_exclude_file(string $file_path, int $attachment_id, string $size = ''): bool {
        // Check size exclusions
        $exclude_sizes = $this->get_setting('exclude_sizes', []);
        if (!empty($size) && in_array($size, $exclude_sizes, true)) {
            return true;
        }
        
        // Check pattern exclusions
        $exclude_patterns = $this->get_setting('exclude_patterns', []);
        if (empty($exclude_patterns)) {
            return false;
        }
        
        // Get file info
        $filename = basename($file_path);
        $filesize = filesize($file_path);
        
        foreach ($exclude_patterns as $pattern) {
            $matches = false;
            
            switch ($pattern['type']) {
                case 'filename':
                    $matches = strpos($filename, $pattern['pattern']) !== false;
                    break;
                    
                case 'path':
                    $matches = strpos($file_path, $pattern['pattern']) !== false;
                    break;
                    
                case 'size':
                    $operator = substr($pattern['pattern'], 0, 1);
                    $kb = (int) substr($pattern['pattern'], 1);
                    $bytes = $kb * 1024;
                    
                    if ($operator === '<') {
                        $matches = $filesize < $bytes;
                    } elseif ($operator === '>') {
                        $matches = $filesize > $bytes;
                    }
                    break;
            }
            
            if ($matches) {
                // Check target applicability
                switch ($pattern['target']) {
                    case 'all':
                        return true;
                        
                    case 'thumbnails':
                        // Exclude only if it's a thumbnail (not full/scaled)
                        return !empty($size) && $size !== 'full' && $size !== 'scaled';
                        
                    case 'custom_media':
                        // This would need additional context to determine
                        return false;
                        
                    case 'specific_sizes':
                        return !empty($size) && in_array($size, $pattern['sizes'], true);
                }
            }
        }
        
        return false;
    }
    
    /**
     * Get all registered image sizes
     *
     * @return array Array of image sizes with details
     */
    public function get_image_sizes(): array {
        global $_wp_additional_image_sizes;
        
        $sizes = [];
        
        // Default WordPress sizes
        $default_sizes = ['thumbnail', 'medium', 'medium_large', 'large'];
        
        foreach ($default_sizes as $size) {
            $sizes[] = [
                'name' => $size,
                'label' => ucfirst(str_replace('_', ' ', $size)),
                'width' => get_option($size . '_size_w'),
                'height' => get_option($size . '_size_h'),
                'crop' => get_option($size . '_crop', false),
            ];
        }
        
        // Additional sizes
        if (isset($_wp_additional_image_sizes) && is_array($_wp_additional_image_sizes)) {
            foreach ($_wp_additional_image_sizes as $name => $data) {
                $sizes[] = [
                    'name' => $name,
                    'label' => ucfirst(str_replace(['_', '-'], ' ', $name)),
                    'width' => $data['width'],
                    'height' => $data['height'],
                    'crop' => $data['crop'],
                ];
            }
        }
        
        return $sizes;
    }
    
    /**
     * Convert images in content to picture elements
     *
     * @param string $content Post content
     * @return string Modified content
     */
    public function convert_content_images(string $content): string {
        // Skip if delivery is disabled
        if (!$this->get_setting('enable_picture_element', true)) {
            return $content;
        }
        
        // Skip if no images
        if (stripos($content, '<img') === false) {
            return $content;
        }
        
        // Use DOMDocument to parse HTML
        $doc = new \DOMDocument();
        libxml_use_internal_errors(true);
        @$doc->loadHTML('<?xml encoding="UTF-8"><div>' . $content . '</div>', LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        libxml_clear_errors();
        
        $xpath = new \DOMXPath($doc);
        $images = $xpath->query('//img[contains(@class, "wp-image-")]');
        
        if ($images->length === 0) {
            return $content;
        }
        
        $modified = false;
        
        foreach ($images as $img) {
            // Extract attachment ID from class
            $classes = $img->getAttribute('class');
            if (preg_match('/wp-image-(\d+)/', $classes, $matches)) {
                $attachment_id = (int) $matches[1];
                
                // Get original HTML for this image
                $img_html = $doc->saveHTML($img);
                
                // Convert to picture element
                $picture_html = $this->convert_to_picture_element($img_html, $attachment_id, '', false, []);
                
                // If conversion happened, replace in DOM
                if ($picture_html !== $img_html) {
                    $fragment = $doc->createDocumentFragment();
                    @$fragment->appendXML($picture_html);
                    if ($fragment->firstChild) {
                        $img->parentNode->replaceChild($fragment->firstChild, $img);
                        $modified = true;
                    }
                }
            }
        }
        
        if (!$modified) {
            return $content;
        }
        
        // Get the modified content
        $body = $xpath->query('//div')->item(0);
        if ($body) {
            $content = '';
            foreach ($body->childNodes as $child) {
                $content .= $doc->saveHTML($child);
            }
        }
        
        return $content;
    }
    
    /**
     * Get optimized image URL
     *
     * @param string $original_url Original image URL
     * @param string $format       Target format (webp/avif)
     * @return string Optimized URL
     */
    private function get_optimized_url(string $original_url, string $format): string {
        $pathinfo = pathinfo($original_url);
        return $pathinfo['dirname'] . '/' . $pathinfo['filename'] . '.' . $format;
    }
    
    /**
     * Check if optimized file exists
     * Also checks for WordPress-suffixed versions (e.g., filename-2.webp)
     *
     * @param string $url Optimized file URL
     * @return bool
     */
    private function optimized_file_exists(string $url): bool {
        // Convert URL to path
        $upload_dir = wp_upload_dir();
        $path = str_replace($upload_dir['baseurl'], $upload_dir['basedir'], $url);

        // Check exact match first
        if (file_exists($path)) {
            return true;
        }

        // Check for WordPress-suffixed versions (e.g., filename-2.webp, filename-3.webp)
        // WordPress adds numeric suffixes to prevent filename collisions
        $pathinfo = pathinfo($path);
        for ($i = 2; $i <= 10; $i++) {
            $suffixed_path = $pathinfo['dirname'] . '/' . $pathinfo['filename'] . '-' . $i . '.' . $pathinfo['extension'];
            if (file_exists($suffixed_path)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get actual optimized file URL (handles WordPress suffixes)
     *
     * @param string $url Expected optimized URL
     * @return string|null Actual URL if exists, null otherwise
     */
    private function get_actual_optimized_url(string $url): ?string {
        $upload_dir = wp_upload_dir();
        $path = str_replace($upload_dir['baseurl'], $upload_dir['basedir'], $url);

        // Check exact match first
        if (file_exists($path)) {
            return $url;
        }

        // Check for WordPress-suffixed versions
        // WordPress adds -N suffix BEFORE dimensions, e.g., image-2-300x187.webp (not image-300x187-2.webp)
        $pathinfo = pathinfo($path);
        $url_pathinfo = pathinfo($url);
        $filename = $pathinfo['filename'];
        $url_filename = $url_pathinfo['filename'];

        // Check if filename has dimension suffix like -300x187
        $dimension_pattern = '/-(\d+x\d+)$/';
        $has_dimensions = preg_match($dimension_pattern, $filename, $dim_matches);

        for ($i = 2; $i <= 10; $i++) {
            if ($has_dimensions) {
                // Insert -N before the dimensions: image-300x187 -> image-2-300x187
                $base_filename = preg_replace($dimension_pattern, '', $filename);
                $dimensions = $dim_matches[1];
                $suffixed_path = $pathinfo['dirname'] . '/' . $base_filename . '-' . $i . '-' . $dimensions . '.' . $pathinfo['extension'];

                if (file_exists($suffixed_path)) {
                    $base_url_filename = preg_replace($dimension_pattern, '', $url_filename);
                    return $url_pathinfo['dirname'] . '/' . $base_url_filename . '-' . $i . '-' . $dimensions . '.' . $url_pathinfo['extension'];
                }
            }

            // Also check traditional suffix at end (for non-sized images)
            $suffixed_path = $pathinfo['dirname'] . '/' . $filename . '-' . $i . '.' . $pathinfo['extension'];
            if (file_exists($suffixed_path)) {
                return $url_pathinfo['dirname'] . '/' . $url_filename . '-' . $i . '.' . $url_pathinfo['extension'];
            }
        }

        return null;
    }
    
    /**
     * Convert PNG to JPEG
     *
     * @param array $metadata      Attachment metadata
     * @param int   $attachment_id Attachment ID
     * @return array Modified metadata
     */
    public function convert_png_to_jpeg(array $metadata, int $attachment_id): array {
        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return $metadata;
        }
        
        // Check if it's a PNG
        $mime_type = get_post_mime_type($attachment_id);
        if ($mime_type !== 'image/png') {
            return $metadata;
        }
        
        // Check if it has transparency
        if ($this->png_has_transparency($file_path)) {
            // Keep PNG if it has transparency
            return $metadata;
        }
        
        // Convert to JPEG
        $jpeg_path = preg_replace('/\.png$/i', '.jpg', $file_path);
        
        $success = false;
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick($file_path);
                $imagick->setImageFormat('jpeg');
                $imagick->setImageCompressionQuality($this->get_jpeg_quality());
                $imagick->setImageBackgroundColor('white');
                $imagick = $imagick->flattenImages();
                $success = $imagick->writeImage($jpeg_path);
                $imagick->destroy();
            } catch (\Exception $e) {
                // Fall back to GD
            }
        }
        
        if (!$success && function_exists('imagecreatefromstring')) {
            $image_data = prorank_read_file($file_path);
            $image = $image_data !== '' ? imagecreatefromstring($image_data) : false;
            if ($image) {
                // Create a white background
                $width = imagesx($image);
                $height = imagesy($image);
                $jpeg_image = imagecreatetruecolor($width, $height);
                $white = imagecolorallocate($jpeg_image, 255, 255, 255);
                imagefill($jpeg_image, 0, 0, $white);
                imagecopy($jpeg_image, $image, 0, 0, 0, 0, $width, $height);
                
                $success = imagejpeg($jpeg_image, $jpeg_path, $this->get_jpeg_quality());
                imagedestroy($image);
                imagedestroy($jpeg_image);
            }
        }
        
        if ($success) {
            // Update attachment
            update_attached_file($attachment_id, $jpeg_path);
            wp_update_post([
                'ID' => $attachment_id,
                'post_mime_type' => 'image/jpeg',
            ]);
            
            // Delete original PNG
            @wp_delete_file($file_path);
            
            // Update metadata
            $metadata = wp_generate_attachment_metadata($attachment_id, $jpeg_path);
        }
        
        return $metadata;
    }
    
    /**
     * Convert CMYK to RGB
     *
     * @param array $metadata      Attachment metadata
     * @param int   $attachment_id Attachment ID
     * @return array Modified metadata
     */
    public function convert_cmyk_to_rgb(array $metadata, int $attachment_id): array {
        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return $metadata;
        }
        
        // Check if it's CMYK
        if (!$this->is_cmyk_image($file_path)) {
            return $metadata;
        }
        
        $success = false;
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick($file_path);
                if ($imagick->getImageColorspace() == \Imagick::COLORSPACE_CMYK) {
                    $imagick->transformImageColorspace(\Imagick::COLORSPACE_RGB);
                    $success = $imagick->writeImage($file_path);
                    $imagick->destroy();
                    
                    if ($success) {
                        // Regenerate metadata
                        $metadata = wp_generate_attachment_metadata($attachment_id, $file_path);
                    }
                }
            } catch (\Exception $e) {
                // Log error
            }
        }
        
        return $metadata;
    }
    
    /**
     * Manage EXIF data
     *
     * @param array $metadata      Attachment metadata
     * @param int   $attachment_id Attachment ID
     * @return array Modified metadata
     */
    public function manage_exif_data(array $metadata, int $attachment_id): array {
        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return $metadata;
        }
        
        // Check if it's a JPEG
        $mime_type = get_post_mime_type($attachment_id);
        if ($mime_type !== 'image/jpeg') {
            return $metadata;
        }
        
        // Remove EXIF data
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick($file_path);
                
                // Keep orientation data if needed
                $orientation = $imagick->getImageOrientation();
                
                // Strip all EXIF data
                $imagick->stripImage();
                
                // Restore orientation
                if ($orientation !== \Imagick::ORIENTATION_UNDEFINED) {
                    $imagick->setImageOrientation($orientation);
                }
                
                $imagick->writeImage($file_path);
                $imagick->destroy();
            } catch (\Exception $e) {
                // Log error
            }
        }
        
        return $metadata;
    }
    
    /**
     * Optimize PDF files
     *
     * @param array $metadata      Attachment metadata
     * @param int   $attachment_id Attachment ID
     * @return array Modified metadata
     */
    public function optimize_pdf(array $metadata, int $attachment_id): array {
        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return $metadata;
        }
        
        // Check if it's a PDF
        $mime_type = get_post_mime_type($attachment_id);
        if ($mime_type !== 'application/pdf') {
            return $metadata;
        }
        
        // Backup original
        if ($this->get_setting('backup_originals', true)) {
            $this->backup_original_image($file_path, $attachment_id);
        }
        
        // Use Imagick if available
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick();
                $imagick->setResolution(150, 150);
                $imagick->readImage($file_path);
                $imagick->setImageCompression(\Imagick::COMPRESSION_JPEG);
                $imagick->setImageCompressionQuality(85);
                $imagick->writeImage($file_path);
                $imagick->destroy();
                
                // Update optimization status
                update_post_meta($attachment_id, '_prorank_pdf_optimized', true);
            } catch (\Exception $e) {
                // Log error
            }
        }
        
        return $metadata;
    }
    
    /**
     * Generate retina images
     *
     * @param array $metadata      Attachment metadata
     * @param int   $attachment_id Attachment ID
     * @return array Modified metadata
     */
    public function generate_retina_images(array $metadata, int $attachment_id): array {
        $file_path = get_attached_file($attachment_id);
        if (!$file_path || !file_exists($file_path)) {
            return $metadata;
        }
        
        // Check if it's an image
        if (!wp_attachment_is_image($attachment_id)) {
            return $metadata;
        }
        
        // Generate 2x version for each size
        if (!empty($metadata['sizes'])) {
            $upload_dir = wp_upload_dir();
            $base_dir = dirname($file_path);
            
            foreach ($metadata['sizes'] as $size => $size_data) {
                $size_path = $base_dir . '/' . $size_data['file'];
                if (file_exists($size_path)) {
                    $this->generate_retina_version($size_path, $size_data['width'] * 2, $size_data['height'] * 2);
                }
            }
        }
        
        // Generate 2x for full size
        $size = wp_getimagesize($file_path);
        if ($size && $size[0] > 0 && $size[1] > 0) {
            $this->generate_retina_version($file_path, $size[0] * 2, $size[1] * 2);
        }
        
        return $metadata;
    }
    
    /**
     * Add AI training protection
     *
     * @param int $attachment_id Attachment ID
     * @return void
     */
    public function add_ai_training_protection(int $attachment_id): void {
        // Add AI training protection meta
        update_post_meta($attachment_id, '_prorank_no_ai_training', '1');
        
        // Add to protected content list
        $protected_images = get_option('prorank_ai_protected_images', []);
        if (!in_array($attachment_id, $protected_images)) {
            $protected_images[] = $attachment_id;
            update_option('prorank_ai_protected_images', $protected_images);
        }
    }
    
    /**
     * Add noai attributes to images
     *
     * @param array $attr       Image attributes
     * @param object $attachment Attachment object
     * @param string|array $size Image size
     * @return array Modified attributes
     */
    public function add_noai_attributes(array $attr, $attachment, $size): array {
        $no_ai = get_post_meta($attachment->ID, '_prorank_no_ai_training', true);
        
        if ($no_ai === '1') {
            // Add data attribute for AI training prevention
            $attr['data-noai'] = '1';
            $attr['data-no-ai-training'] = 'true';
            
            // Add to alt text if configured
            if ($this->get_setting('add_noai_to_alt', false)) {
                $attr['alt'] = ($attr['alt'] ?? '') . ' [No AI Training]';
            }
        }
        
        return $attr;
    }
    
    /**
     * Get JPEG quality based on compression type
     *
     * @return int Quality value
     */
    private function get_jpeg_quality(): int {
        $compression_type = $this->get_setting('compression_type', 'lossy');
        $quality = self::COMPRESSION_TYPES[$compression_type]['jpeg_quality'] ?? 85;
        
        // Override with custom quality if SmartCompression is disabled
        if (!$this->get_setting('smart_compression', true)) {
            $quality = $this->get_setting('jpeg_quality', 85);
        }
        
        return $quality;
    }
    
    /**
     * Check if PNG has transparency
     *
     * @param string $file_path File path
     * @return bool
     */
    private function png_has_transparency(string $file_path): bool {
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick($file_path);
                $has_alpha = $imagick->getImageAlphaChannel();
                $imagick->destroy();
                return $has_alpha === \Imagick::ALPHACHANNEL_ACTIVATE;
            } catch (\Exception $e) {
                // Fall back to GD
            }
        }

        if (function_exists('imagecreatefrompng')) {
            $image = @imagecreatefrompng($file_path);
            if ($image) {
                $width = imagesx($image);
                $height = imagesy($image);
                $total_pixels = $width * $height;

                // Use sampling for large images to avoid O(n²) performance issues
                // Sample up to 2000 random pixels - statistically sufficient to detect transparency
                $max_samples = min(2000, $total_pixels);

                if ($total_pixels <= 2000) {
                    // Small image - scan all pixels (still fast)
                    for ($x = 0; $x < $width; $x++) {
                        for ($y = 0; $y < $height; $y++) {
                            $rgba = imagecolorat($image, $x, $y);
                            $alpha = ($rgba & 0x7F000000) >> 24;
                            if ($alpha > 0) {
                                imagedestroy($image);
                                return true;
                            }
                        }
                    }
                } else {
                    // Large image - use random sampling + edge scanning
                    // First check corners and edges (common places for transparency)
                    $edge_points = [
                        [0, 0], [$width - 1, 0], [0, $height - 1], [$width - 1, $height - 1], // corners
                        [(int)($width / 2), 0], [(int)($width / 2), $height - 1], // top/bottom center
                        [0, (int)($height / 2)], [$width - 1, (int)($height / 2)], // left/right center
                    ];

                    foreach ($edge_points as $point) {
                        $rgba = imagecolorat($image, $point[0], $point[1]);
                        $alpha = ($rgba & 0x7F000000) >> 24;
                        if ($alpha > 0) {
                            imagedestroy($image);
                            return true;
                        }
                    }

                    // Then sample random pixels throughout the image
                    for ($i = 0; $i < $max_samples; $i++) {
                        $x = wp_rand(0, $width - 1);
                        $y = wp_rand(0, $height - 1);
                        $rgba = imagecolorat($image, $x, $y);
                        $alpha = ($rgba & 0x7F000000) >> 24;
                        if ($alpha > 0) {
                            imagedestroy($image);
                            return true;
                        }
                    }
                }
                imagedestroy($image);
            }
        }

        return false;
    }
    
    /**
     * Check if image is CMYK
     *
     * @param string $file_path File path
     * @return bool
     */
    private function is_cmyk_image(string $file_path): bool {
        if (extension_loaded('imagick') && class_exists('Imagick')) {
            try {
                $imagick = new \Imagick($file_path);
                $colorspace = $imagick->getImageColorspace();
                $imagick->destroy();
                return $colorspace === \Imagick::COLORSPACE_CMYK;
            } catch (\Exception $e) {
                // Log error
            }
        }
        
        return false;
    }
    
    /**
     * Generate retina version of image
     *
     * @param string $file_path Source file path
     * @param int $width Target width
     * @param int $height Target height
     * @return bool Success status
     */
    private function generate_retina_version(string $file_path, int $width, int $height): bool {
        $pathinfo = pathinfo($file_path);
        $retina_path = $pathinfo['dirname'] . '/' . $pathinfo['filename'] . '@2x.' . $pathinfo['extension'];
        
        // Skip if already exists
        if (file_exists($retina_path)) {
            return true;
        }
        
        // Skip if source is smaller than target
        $size = wp_getimagesize($file_path);
        if (!$size || $size[0] < $width || $size[1] < $height) {
            return false;
        }
        
        $editor = wp_get_image_editor($file_path);
        if (!is_wp_error($editor)) {
            $editor->resize($width, $height, true);
            $result = $editor->save($retina_path);
            return !is_wp_error($result);
        }
        
        return false;
    }
    
    /**
     * Smart crop and resize image
     *
     * @param string $file_path Source file path
     * @param int $width Target width
     * @param int $height Target height
     * @param array $options Crop options
     * @return bool Success status
     */
    public function smart_crop_resize(string $file_path, int $width, int $height, array $options = []): bool {
        $defaults = [
            'crop_focus' => 'center', // center, faces, entropy, attention
            'upscale' => false,
            'quality' => $this->get_jpeg_quality(),
        ];
        
        $options = wp_parse_args($options, $defaults);
        
        // Local crop only
        $editor = wp_get_image_editor($file_path);
        if (is_wp_error($editor)) {
            return false;
        }
        
        // Get current dimensions
        $size = $editor->get_size();
        $orig_width = $size['width'];
        $orig_height = $size['height'];
        
        // Calculate crop position based on focus
        $crop_x = 0;
        $crop_y = 0;
        
        switch ($options['crop_focus']) {
            case 'top':
                $crop_y = 0;
                break;
            case 'bottom':
                $crop_y = $orig_height - $height;
                break;
            case 'left':
                $crop_x = 0;
                break;
            case 'right':
                $crop_x = $orig_width - $width;
                break;
            case 'entropy':
                // Simple entropy-based crop (find area with most detail)
                list($crop_x, $crop_y) = $this->find_entropy_crop($file_path, $width, $height);
                break;
            default: // center
                $crop_x = ($orig_width - $width) / 2;
                $crop_y = ($orig_height - $height) / 2;
        }
        
        // Crop and resize
        $result = $editor->crop($crop_x, $crop_y, $width, $height, $width, $height, false);
        if (is_wp_error($result)) {
            return false;
        }
        
        // Save
        $filetype = wp_check_filetype($file_path);
        $saved = $editor->save($file_path, $filetype['type'] ?: 'image/jpeg');
        return !is_wp_error($saved);
    }
    
    /**
     * Find optimal crop position based on entropy
     *
     * @param string $file_path Source file path
     * @param int $width Target width
     * @param int $height Target height
     * @return array [x, y] coordinates
     */
    private function find_entropy_crop(string $file_path, int $width, int $height): array {
        // Simple entropy calculation - find area with most variation
        // This is a simplified version - real implementation would be more sophisticated
        
        if (!extension_loaded('gd')) {
            return [0, 0];
        }
        
        $image_data = prorank_read_file($file_path);
        $image = $image_data !== '' ? imagecreatefromstring($image_data) : false;
        if (!$image) {
            return [0, 0];
        }
        
        $orig_width = imagesx($image);
        $orig_height = imagesy($image);
        
        $best_x = 0;
        $best_y = 0;
        $max_entropy = 0;
        
        // Sample areas and find highest entropy
        $step = 10; // Sample every 10 pixels for performance
        
        for ($y = 0; $y <= $orig_height - $height; $y += $step) {
            for ($x = 0; $x <= $orig_width - $width; $x += $step) {
                $entropy = $this->calculate_area_entropy($image, $x, $y, $width, $height);
                if ($entropy > $max_entropy) {
                    $max_entropy = $entropy;
                    $best_x = $x;
                    $best_y = $y;
                }
            }
        }
        
        imagedestroy($image);
        
        return [$best_x, $best_y];
    }
    
    /**
     * Calculate entropy for an area of image
     *
     * @param resource $image GD image resource
     * @param int $x Start X
     * @param int $y Start Y
     * @param int $width Area width
     * @param int $height Area height
     * @return float Entropy value
     */
    private function calculate_area_entropy($image, int $x, int $y, int $width, int $height): float {
        $histogram = array_fill(0, 256, 0);
        $total_pixels = $width * $height;
        
        // Build histogram of grayscale values
        for ($py = $y; $py < $y + $height; $py++) {
            for ($px = $x; $px < $x + $width; $px++) {
                $rgb = imagecolorat($image, $px, $py);
                $r = ($rgb >> 16) & 0xFF;
                $g = ($rgb >> 8) & 0xFF;
                $b = $rgb & 0xFF;
                
                // Convert to grayscale
                $gray = (int)(0.299 * $r + 0.587 * $g + 0.114 * $b);
                $histogram[$gray]++;
            }
        }
        
        // Calculate entropy
        $entropy = 0;
        foreach ($histogram as $count) {
            if ($count > 0) {
                $probability = $count / $total_pixels;
                $entropy -= $probability * log($probability, 2);
            }
        }
        
        return $entropy;
    }
    
    /**
     * Rewrite image URLs to use CDN
     *
     * @param string $url Original URL
     * @param int $attachment_id Attachment ID
     * @return string Modified URL
     */
    public function rewrite_cdn_url(string $url, int $attachment_id): string {
        $cdn_url = $this->get_setting('cdn_url', '');
        
        if (empty($cdn_url)) {
            return $url;
        }
        
        // Only rewrite for frontend
        if (is_admin()) {
            return $url;
        }
        
        // Parse URLs
        $site_url = get_site_url();
        $cdn_url = rtrim($cdn_url, '/');
        
        // Replace site URL with CDN URL
        if (strpos($url, $site_url) === 0) {
            $url = str_replace($site_url, $cdn_url, $url);
        }
        
        return $url;
    }
    
    /**
     * Rewrite srcset URLs to use CDN
     *
     * @param array $sources Sources array
     * @param array $size_array Size array
     * @param string $image_src Image source
     * @param array $image_meta Image metadata
     * @param int $attachment_id Attachment ID
     * @return array Modified sources
     */
    public function rewrite_srcset_cdn(array $sources, array $size_array, string $image_src, array $image_meta, int $attachment_id): array {
        $cdn_url = $this->get_setting('cdn_url', '');
        
        if (empty($cdn_url) || is_admin()) {
            return $sources;
        }
        
        $site_url = get_site_url();
        $cdn_url = rtrim($cdn_url, '/');
        
        foreach ($sources as &$source) {
            if (isset($source['url']) && strpos($source['url'], $site_url) === 0) {
                $source['url'] = str_replace($site_url, $cdn_url, $source['url']);
            }
        }
        
        return $sources;
    }
    
    /**
     * Rewrite image URLs in content to use CDN
     *
     * @param string $content Post content
     * @return string Modified content
     */
    public function rewrite_content_cdn_urls(string $content): string {
        $cdn_url = $this->get_setting('cdn_url', '');
        
        if (empty($cdn_url) || is_admin()) {
            return $content;
        }
        
        $site_url = get_site_url();
        $cdn_url = rtrim($cdn_url, '/');
        $upload_dir = wp_upload_dir();
        
        // Replace upload URLs with CDN
        $content = str_replace($upload_dir['baseurl'], str_replace($site_url, $cdn_url, $upload_dir['baseurl']), $content);
        
        return $content;
    }

}
