<?php
/**
 * Schema Markup Module
 *
 * @package ProRank\SEO\Modules\OnPageSEO
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\OnPageSEO;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Modules\BaseModule;

/**
 * Schema Markup module class
 *
 * Handles basic structured data and rich snippets for FREE tier
 */
class SchemaMarkupModule extends BaseModule {
    /**
     * Module slug
     *
     * @var string
     */
    protected string $slug = 'schema-markup';
    
    /**
     * Module name
     *
     * @var string
     */
    protected string $name = 'Schema Markup';
    
    /**
     * Module description
     *
     * @var string
     */
    protected string $description = 'Basic structured data and rich snippets (Article, Organization, WebSite)';
    
    /**
     * Required tier
     *
     * @var string
     */
    protected string $feature_tier = 'free';
    
    /**
     * Parent module
     *
     * @var string|null
     */
    protected ?string $parent_slug = 'on-page-seo';
    
    /**
     * Initialize module hooks
     *
     * @return void
     */
    public function init_hooks(): void {
        if (!$this->is_enabled()) {
            return;
        }

        // 2025 Enhancement: Register schemas with centralized output manager
        add_action('prorank_seo_register_schemas', [$this, 'register_basic_schemas'], 5);

        // Add settings
        add_action('rest_api_init', [$this, 'register_rest_routes']);
    }
    
    /**
     * Register REST routes for settings
     *
     * @return void
     */
    public function register_rest_routes(): void {
        register_rest_route('prorank-seo/v1', '/settings/basic-schema', [
            'methods' => 'GET',
            'callback' => [$this, 'get_settings'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            }
        ]);
        
        register_rest_route('prorank-seo/v1', '/settings/basic-schema', [
            'methods' => 'POST',
            'callback' => [$this, 'update_settings'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            }
        ]);
    }
    
    /**
     * Get settings
     *
     * @return array
     */
    public function get_settings(): array {
        $defaults = $this->get_default_settings();
        $settings = get_option('prorank_basic_schema_settings', $defaults);

        if (!is_array($settings)) {
            $settings = $defaults;
        }

        $settings = array_merge($defaults, $settings);

        if (!empty($settings['disable_all_schema'])) {
            $settings['enable_website_schema'] = false;
            $settings['enable_organization_schema'] = false;
            $settings['enable_article_schema'] = false;
            $settings['enable_breadcrumb_schema'] = false;
        }

        return $settings;
    }
    
    /**
     * Update settings
     *
     * @param \WP_REST_Request $request Request object
     * @return array
     */
    public function update_settings(\WP_REST_Request $request): array {
        $defaults = $this->get_default_settings();
        $settings = $request->get_json_params();

        if (!is_array($settings)) {
            $settings = [];
        }

        $settings = array_merge($defaults, $settings);

        $settings['disable_all_schema'] = !empty($settings['disable_all_schema']);
        $settings['enable_website_schema'] = !empty($settings['enable_website_schema']);
        $settings['enable_organization_schema'] = !empty($settings['enable_organization_schema']);
        $settings['enable_article_schema'] = !empty($settings['enable_article_schema']);
        $settings['enable_breadcrumb_schema'] = !empty($settings['enable_breadcrumb_schema']);

        $settings['organization_name'] = sanitize_text_field($settings['organization_name']);
        $settings['organization_logo'] = esc_url_raw($settings['organization_logo']);
        $settings['organization_type'] = sanitize_text_field($settings['organization_type']);

        if ($settings['disable_all_schema']) {
            $settings['enable_website_schema'] = false;
            $settings['enable_organization_schema'] = false;
            $settings['enable_article_schema'] = false;
            $settings['enable_breadcrumb_schema'] = false;
        }

        update_option('prorank_basic_schema_settings', $settings);
        return $settings;
    }
    
    /**
     * Get default settings
     *
     * @return array
     */
    private function get_default_settings(): array {
        return [
            'disable_all_schema' => false,
            'enable_website_schema' => true,
            'enable_organization_schema' => true,
            'enable_article_schema' => true,
            'enable_breadcrumb_schema' => true,
            'organization_name' => get_bloginfo('name'),
            'organization_logo' => '',
            'organization_type' => 'Organization',
        ];
    }
    
    /**
     * Register basic schemas with output manager
     *
     * 2025 Enhancement: Uses SchemaOutputManager for consolidated output
     *
     * @param \ProRank\SEO\Core\Schema\SchemaOutputManager $manager Schema output manager
     * @return void
     */
    public function register_basic_schemas($manager): void {
        $settings = $this->get_settings();

        if (!empty($settings['disable_all_schema'])) {
            return;
        }

        // Website Schema
        if ($settings['enable_website_schema']) {
            $schema = $this->get_website_schema();
            if ($schema) {
                $manager->register_schema($schema, 5, 'SchemaMarkupModule::WebSite');
            }
        }

        // Organization Schema
        if ($settings['enable_organization_schema'] && is_front_page()) {
            $schema = $this->get_organization_schema($settings);
            if ($schema) {
                $manager->register_schema($schema, 5, 'SchemaMarkupModule::Organization');
            }
        }

        // Article Schema for posts
        if ($settings['enable_article_schema'] && is_singular('post')) {
            $schema = $this->get_article_schema();
            if ($schema) {
                $manager->register_schema($schema, 5, 'SchemaMarkupModule::Article');
            }
        }

        // Breadcrumb Schema
        if ($settings['enable_breadcrumb_schema'] && !is_front_page()) {
            $schema = $this->get_breadcrumb_schema();
            if ($schema) {
                $manager->register_schema($schema, 5, 'SchemaMarkupModule::BreadcrumbList');
            }
        }
    }
    
    /**
     * Get the current page language, accounting for multilingual plugins
     *
     * @return string Language code (e.g., 'en-US', 'de-DE')
     */
    private function get_page_language(): string {
        // Check for WPML
        if (defined('ICL_LANGUAGE_CODE') && ICL_LANGUAGE_CODE) {
            global $sitepress;
            if ($sitepress && method_exists($sitepress, 'get_locale')) {
                return str_replace('_', '-', $sitepress->get_locale(ICL_LANGUAGE_CODE));
            }
            return ICL_LANGUAGE_CODE;
        }

        // Check for Polylang
        if (function_exists('pll_current_language')) {
            $locale = pll_current_language('locale');
            if ($locale) {
                return str_replace('_', '-', $locale);
            }
        }

        // Check for TranslatePress
        global $TRP_LANGUAGE;
        if (!empty($TRP_LANGUAGE)) {
            return str_replace('_', '-', $TRP_LANGUAGE);
        }

        // Fallback to WordPress locale
        return str_replace('_', '-', get_locale());
    }

    /**
     * Get website schema
     *
     * @return array
     */
    private function get_website_schema(): array {
        $schema = [
            '@type' => 'WebSite',
            '@id' => home_url('/#website'),
            'url' => home_url('/'),
            'name' => get_bloginfo('name'),
            'description' => get_bloginfo('description'),
            'inLanguage' => $this->get_page_language(),
        ];
        
        // Add search action
        $schema['potentialAction'] = [
            '@type' => 'SearchAction',
            'target' => [
                '@type' => 'EntryPoint',
                'urlTemplate' => home_url('/?s={search_term_string}')
            ],
            'query-input' => 'required name=search_term_string'
        ];
        
        return $schema;
    }
    
    /**
     * Get organization schema
     *
     * @param array $settings Module settings
     * @return array
     */
    private function get_organization_schema(array $settings): array {
        $schema = [
            '@type' => $settings['organization_type'] ?? 'Organization',
            '@id' => home_url('/#organization'),
            'name' => $settings['organization_name'] ?? get_bloginfo('name'),
            'url' => home_url('/'),
        ];
        
        // Add logo if available
        if (!empty($settings['organization_logo'])) {
            $schema['logo'] = [
                '@type' => 'ImageObject',
                'url' => $settings['organization_logo']
            ];
        } elseif (has_custom_logo()) {
            $logo_id = get_theme_mod('custom_logo');
            $logo_url = wp_get_attachment_image_url($logo_id, 'full');
            if ($logo_url) {
                $schema['logo'] = [
                    '@type' => 'ImageObject',
                    'url' => $logo_url
                ];
            }
        }
        
        // Add same as (social profiles)
        $social_profiles = [];
        $social_settings = get_option('prorank_seo_social_profiles', []);
        
        if (!empty($social_settings['facebook'])) {
            $social_profiles[] = $social_settings['facebook'];
        }
        if (!empty($social_settings['twitter'])) {
            $social_profiles[] = 'https://twitter.com/' . ltrim($social_settings['twitter'], '@');
        }
        if (!empty($social_settings['linkedin'])) {
            $social_profiles[] = $social_settings['linkedin'];
        }
        if (!empty($social_settings['youtube'])) {
            $social_profiles[] = $social_settings['youtube'];
        }
        
        if (!empty($social_profiles)) {
            $schema['sameAs'] = $social_profiles;
        }
        
        return $schema;
    }
    
    /**
     * Get article schema
     *
     * @return array
     */
    private function get_article_schema(): array {
        global $post;
        
        $schema = [
            '@type' => 'Article',
            '@id' => get_permalink() . '#article',
            'headline' => get_the_title(),
            'datePublished' => get_the_date('c'),
            'dateModified' => get_the_modified_date('c'),
            'mainEntityOfPage' => [
                '@type' => 'WebPage',
                '@id' => get_permalink()
            ],
            'author' => [
                '@type' => 'Person',
                'name' => get_the_author(),
                'url' => get_author_posts_url($post->post_author)
            ],
            'publisher' => [
                '@type' => 'Organization',
                'name' => get_bloginfo('name'),
                '@id' => home_url('/#organization')
            ]
        ];
        
        // Add description
        if (has_excerpt()) {
            $schema['description'] = get_the_excerpt();
        }
        
        // Add featured image
        if (has_post_thumbnail()) {
            $image_id = get_post_thumbnail_id();
            $image_url = wp_get_attachment_image_url($image_id, 'full');
            $image_meta = wp_get_attachment_metadata($image_id);
            
            if ($image_url) {
                $schema['image'] = [
                    '@type' => 'ImageObject',
                    'url' => $image_url,
                    'width' => $image_meta['width'] ?? '',
                    'height' => $image_meta['height'] ?? ''
                ];
            }
        }
        
        // Add word count
        $content = get_post_field('post_content', $post->ID);
        $word_count = str_word_count(wp_strip_all_tags($content));
        $schema['wordCount'] = $word_count;
        
        // Add article section (categories)
        $categories = get_the_category();
        if (!empty($categories)) {
            $schema['articleSection'] = $categories[0]->name;
        }
        
        // Add keywords (tags)
        $tags = get_the_tags();
        if ($tags) {
            $schema['keywords'] = implode(', ', wp_list_pluck($tags, 'name'));
        }
        
        return $schema;
    }
    
    /**
     * Get breadcrumb schema
     *
     * @return array|null
     */
    private function get_breadcrumb_schema(): ?array {
        // Don't show on homepage
        if (is_front_page()) {
            return null;
        }
        
        $items = [];
        $position = 1;
        
        // Home
        $items[] = [
            '@type' => 'ListItem',
            'position' => $position++,
            'name' => 'Home',
            'item' => home_url('/')
        ];
        
        // Category/Archive pages
        if (is_category() || is_tag() || is_tax()) {
            $term = get_queried_object();
            $items[] = [
                '@type' => 'ListItem',
                'position' => $position++,
                'name' => $term->name,
                'item' => get_term_link($term)
            ];
        }
        
        // Single post/page
        if (is_singular()) {
            // Add categories for posts
            if (is_single() && !is_singular('page')) {
                $categories = get_the_category();
                if (!empty($categories)) {
                    $items[] = [
                        '@type' => 'ListItem',
                        'position' => $position++,
                        'name' => $categories[0]->name,
                        'item' => get_category_link($categories[0])
                    ];
                }
            }
            
            // Add current page/post
            $items[] = [
                '@type' => 'ListItem',
                'position' => $position++,
                'name' => get_the_title(),
                'item' => get_permalink()
            ];
        }
        
        if (empty($items)) {
            return null;
        }
        
        return [
            '@type' => 'BreadcrumbList',
            '@id' => get_permalink() . '#breadcrumb',
            'itemListElement' => $items
        ];
    }
}
