<?php
/**
 * Meta Module
 *
 * Manages SEO meta tags and social media meta tags.
 *
 * @package ProRank\SEO\Modules\Meta
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\Meta;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Modules\BaseModule;
use ProRank\SEO\Core\RestApi\MetaController;
use ProRank\SEO\Frontend\HeadOutput;

/**
 * MetaModule class
 * 
 * Provides comprehensive meta tag management including:
 * - SEO title and description
 * - Canonical URLs
 * - Meta robots directives
 * - Open Graph tags
 * - Twitter Card tags
 */
class MetaModule extends BaseModule {

    /**
     * Required tier for this module
     */
    protected string $feature_tier = 'free';
    
    /**
     * MetaController instance
     *
     * @var MetaController
     */
    private MetaController $meta_controller;
    
    /**
     * HeadOutput instance
     *
     * @var HeadOutput|null
     */
    private ?HeadOutput $head_output = null;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->slug = 'titles-meta';
        $this->name = 'Titles & Meta';
        $this->description = 'Manage page titles and meta descriptions';
$this->parent_slug = 'on-page-seo';
        
        $this->meta_controller = new MetaController();
    }

    /**
     * Get all Titles & Meta settings.
     *
     * @return array
     */
    public function get_all_settings(): array {
        $settings = get_option('prorank_seo_titles_meta', []);
        return is_array($settings) ? $settings : [];
    }

    /**
     * Get a specific Titles & Meta setting.
     *
     * @param string $key Setting key.
     * @param mixed  $default Default value.
     * @return mixed
     */
    public function get_setting(string $key, $default = null) {
        $settings = $this->get_all_settings();
        return array_key_exists($key, $settings) ? $settings[$key] : $default;
    }

    /**
     * Get a specific Titles & Meta setting (compat helper).
     *
     * @param string $key Setting key.
     * @param mixed  $default Default value.
     * @return mixed
     */
    public function get_setting_value(string $key, $default = null) {
        return $this->get_setting($key, $default);
    }

    /**
     * Process a template string with dynamic variables.
     *
     * @param string $template Template string.
     * @param string $context Context identifier.
     * @return string
     */
    public function process_template(string $template, string $context = ''): string {
        if ($template === '') {
            return '';
        }

        try {
            $parser = new \ProRank\SEO\Core\DynamicVariables();
            $context_object = null;
            $context_type = '';

            if (in_array($context, ['post', 'page', 'singular'], true)) {
                $context_object = get_post();
                $context_type = 'post';
            } elseif ($context === 'home') {
                $posts_page_id = (int) get_option('page_for_posts');
                if ($posts_page_id > 0) {
                    $context_object = get_post($posts_page_id);
                    $context_type = 'page';
                } else {
                    $context_type = 'archive';
                }
            } elseif (in_array($context, ['term', 'taxonomy'], true)) {
                $term = get_queried_object();
                if ($term instanceof \WP_Term) {
                    $context_object = $term;
                    $context_type = 'term';
                }
            } elseif ($context === 'author') {
                $user_id = get_queried_object_id();
                $user = get_user_by('id', $user_id);
                if ($user instanceof \WP_User) {
                    $context_object = $user;
                    $context_type = 'author';
                }
            } elseif ($context === 'search') {
                $context_type = 'search';
            } elseif ($context === 'archive') {
                $context_object = get_queried_object();
                $context_type = 'archive';
            }

            return $parser->parse($template, $context_object, $context_type);
        } catch (\Throwable $e) {
            return $template;
        }
    }

    /**
     * Get robots defaults from settings.
     *
     * @return array
     */
    public function get_robots_defaults(): array {
        return [
            'index' => (bool) $this->get_setting('robots_default_index', true),
            'follow' => (bool) $this->get_setting('robots_default_follow', true),
            'archive' => (bool) $this->get_setting('robots_default_archive', true),
            'imageindex' => (bool) $this->get_setting('robots_default_imageindex', true),
            'snippet' => (bool) $this->get_setting('robots_default_snippet', true),
            'max-snippet' => (int) $this->get_setting('robots_max_snippet', -1),
            'max-video-preview' => (int) $this->get_setting('robots_max_video_preview', -1),
            'max-image-preview' => $this->get_setting('robots_max_image_preview', 'large'),
        ];
    }
    
    /**
     * {@inheritDoc}
     */
    public function init_hooks(): void {
        // Register REST API routes
        add_action('rest_api_init', [$this->meta_controller, 'register_routes']);
        
        // Add meta box save functionality
        add_action('save_post', [$this, 'save_meta_box_data'], 10, 2);
        
        // Add default meta tags filter
        add_filter('prorank_seo_default_meta_tags', [$this, 'get_default_meta_tags'], 10, 2);
        
        // Add social meta tags filter
        add_filter('prorank_seo_social_meta_tags', [$this, 'get_social_meta_tags'], 10, 2);

        // Apply author base and archive redirects
        add_action('init', [$this, 'apply_author_base']);
        add_action('template_redirect', [$this, 'maybe_redirect_archives']);
        
        // HeadOutput is now initialized directly in the Plugin class
        // to ensure it's always loaded for frontend requests
        
        // No internal logging for free builds.
    }

    /**
     * Apply custom author base when configured.
     *
     * @return void
     */
    public function apply_author_base(): void {
        $settings = $this->get_all_settings();
        $author_base = isset($settings['author_base']) ? sanitize_title((string) $settings['author_base']) : '';

        if ($author_base === '') {
            return;
        }

        global $wp_rewrite;
        $wp_rewrite->author_base = $author_base;
    }

    /**
     * Redirect disabled archives to the homepage.
     *
     * @return void
     */
    public function maybe_redirect_archives(): void {
        if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
            return;
        }

        $settings = $this->get_all_settings();

        if (!empty($settings['disable_author_archives']) && is_author()) {
            wp_safe_redirect(home_url('/'), 301);
            exit;
        }

        if (!empty($settings['disable_date_archives']) && is_date()) {
            wp_safe_redirect(home_url('/'), 301);
            exit;
        }
    }
    
    
    /**
     * Save meta box data
     *
     * @param int      $post_id Post ID.
     * @param \WP_Post $post    Post object.
     * @return void
     */
    public function save_meta_box_data(int $post_id, \WP_Post $post): void {
        // Skip autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Verify nonce
        if (!isset($_POST['prorank_seo_meta_nonce']) || !check_admin_referer('prorank_seo_save_meta', 'prorank_seo_meta_nonce')) {
            return;
        }
        
        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Save SEO meta fields
        $seo_title = '';
        if (isset($_POST['prorank_seo_title']) && !is_array($_POST['prorank_seo_title'])) {
            $seo_title = sanitize_text_field(wp_unslash($_POST['prorank_seo_title']));
        }
        $this->save_meta_field($post_id, 'prorank_seo_title', $seo_title);

        $seo_description = '';
        if (isset($_POST['prorank_seo_description']) && !is_array($_POST['prorank_seo_description'])) {
            $seo_description = sanitize_textarea_field(wp_unslash($_POST['prorank_seo_description']));
        }
        $this->save_meta_field($post_id, 'prorank_seo_description', $seo_description);

        $canonical_url = '';
        if (isset($_POST['prorank_seo_canonical_url']) && !is_array($_POST['prorank_seo_canonical_url'])) {
            $canonical_url = esc_url_raw(wp_unslash($_POST['prorank_seo_canonical_url']));
        }
        $this->save_meta_field($post_id, 'prorank_seo_canonical_url', $canonical_url);
        
        // Save meta robots
        $meta_robots = filter_input(INPUT_POST, 'prorank_meta_robots', FILTER_SANITIZE_FULL_SPECIAL_CHARS, FILTER_REQUIRE_ARRAY);
        if (is_array($meta_robots)) {
            $robots = array_map('sanitize_key', $meta_robots);
            update_post_meta($post_id, '_prorank_meta_robots', $robots);
        } else {
            delete_post_meta($post_id, '_prorank_meta_robots');
        }
        
        // Save social meta fields
        $og_title = '';
        if (isset($_POST['prorank_og_title']) && !is_array($_POST['prorank_og_title'])) {
            $og_title = sanitize_text_field(wp_unslash($_POST['prorank_og_title']));
        }
        $this->save_meta_field($post_id, 'prorank_og_title', $og_title);

        $og_description = '';
        if (isset($_POST['prorank_og_description']) && !is_array($_POST['prorank_og_description'])) {
            $og_description = sanitize_textarea_field(wp_unslash($_POST['prorank_og_description']));
        }
        $this->save_meta_field($post_id, 'prorank_og_description', $og_description);

        $og_image_id = '';
        if (isset($_POST['prorank_og_image_id']) && !is_array($_POST['prorank_og_image_id'])) {
            $og_image_id = absint(wp_unslash($_POST['prorank_og_image_id']));
        }
        $this->save_meta_field($post_id, 'prorank_og_image_id', $og_image_id);

        $twitter_title = '';
        if (isset($_POST['prorank_twitter_title']) && !is_array($_POST['prorank_twitter_title'])) {
            $twitter_title = sanitize_text_field(wp_unslash($_POST['prorank_twitter_title']));
        }
        $this->save_meta_field($post_id, 'prorank_twitter_title', $twitter_title);

        $twitter_description = '';
        if (isset($_POST['prorank_twitter_description']) && !is_array($_POST['prorank_twitter_description'])) {
            $twitter_description = sanitize_textarea_field(wp_unslash($_POST['prorank_twitter_description']));
        }
        $this->save_meta_field($post_id, 'prorank_twitter_description', $twitter_description);

        $twitter_image_id = '';
        if (isset($_POST['prorank_twitter_image_id']) && !is_array($_POST['prorank_twitter_image_id'])) {
            $twitter_image_id = absint(wp_unslash($_POST['prorank_twitter_image_id']));
        }
        $this->save_meta_field($post_id, 'prorank_twitter_image_id', $twitter_image_id);

        $twitter_card_type = '';
        if (isset($_POST['prorank_twitter_card_type']) && !is_array($_POST['prorank_twitter_card_type'])) {
            $twitter_card_type = sanitize_key(wp_unslash($_POST['prorank_twitter_card_type']));
        }
        $this->save_meta_field($post_id, 'prorank_twitter_card_type', $twitter_card_type);
    }
    
    /**
     * Save individual meta field
     *
     * @param int         $post_id    Post ID.
     * @param string      $field_name Field name (without underscore prefix).
     * @param string|int  $value      Sanitized value.
     * @return void
     */
    private function save_meta_field(int $post_id, string $field_name, $value): void {
        if ($value !== '' && $value !== null) {
            update_post_meta($post_id, '_' . $field_name, $value);
        } else {
            delete_post_meta($post_id, '_' . $field_name);
        }
    }
    
    /**
     * Get default meta tags for a post
     *
     * @param array $meta_tags Existing meta tags.
     * @param int   $post_id   Post ID.
     * @return array
     */
    public function get_default_meta_tags(array $meta_tags, int $post_id): array {
        // SEO Title
        $seo_title = get_post_meta($post_id, '_prorank_seo_title', true);
        if (!empty($seo_title)) {
            $meta_tags['title'] = $seo_title;
        }
        
        // Meta Description
        $seo_description = get_post_meta($post_id, '_prorank_seo_description', true);
        if (!empty($seo_description)) {
            $meta_tags['description'] = $seo_description;
        }
        
        // Canonical URL
        $canonical_url = get_post_meta($post_id, '_prorank_seo_canonical_url', true);
        if (!empty($canonical_url)) {
            $meta_tags['canonical'] = $canonical_url;
        }
        
        // Meta Robots
        $meta_robots = get_post_meta($post_id, '_prorank_meta_robots', true);
        if (!empty($meta_robots) && is_array($meta_robots)) {
            $meta_tags['robots'] = implode(', ', $meta_robots);
        }
        
        return $meta_tags;
    }
    
    /**
     * Get social meta tags for a post
     *
     * @param array $social_tags Existing social tags.
     * @param int   $post_id     Post ID.
     * @return array
     */
    public function get_social_meta_tags(array $social_tags, int $post_id): array {
        $post = get_post($post_id);
        if (!$post) {
            return $social_tags;
        }
        
        // Open Graph tags
        $og_title = get_post_meta($post_id, '_prorank_og_title', true);
        if (!empty($og_title)) {
            $social_tags['og:title'] = $og_title;
        } elseif (!isset($social_tags['og:title'])) {
            // Fallback to SEO title or post title
            $seo_title = get_post_meta($post_id, '_prorank_seo_title', true);
            $social_tags['og:title'] = !empty($seo_title) ? $seo_title : $post->post_title;
        }
        
        $og_description = get_post_meta($post_id, '_prorank_og_description', true);
        if (!empty($og_description)) {
            $social_tags['og:description'] = $og_description;
        } elseif (!isset($social_tags['og:description'])) {
            // Fallback to SEO description or excerpt
            $seo_description = get_post_meta($post_id, '_prorank_seo_description', true);
            if (!empty($seo_description)) {
                $social_tags['og:description'] = $seo_description;
            } elseif (!empty($post->post_excerpt)) {
                $social_tags['og:description'] = $post->post_excerpt;
            } else {
                $social_tags['og:description'] = wp_trim_words($post->post_content, 30, '...');
            }
        }
        
        $og_image_id = get_post_meta($post_id, '_prorank_og_image_id', true);
        if (!empty($og_image_id)) {
            $image_url = wp_get_attachment_url((int) $og_image_id);
            if ($image_url) {
                $social_tags['og:image'] = $image_url;
            }
        } elseif (!isset($social_tags['og:image'])) {
            // Fallback to featured image
            $featured_image_id = get_post_thumbnail_id($post_id);
            if ($featured_image_id) {
                $image_url = wp_get_attachment_url($featured_image_id);
                if ($image_url) {
                    $social_tags['og:image'] = $image_url;
                }
            }
        }
        
        // Twitter Card tags
        $twitter_card_type = get_post_meta($post_id, '_prorank_twitter_card_type', true);
        $social_tags['twitter:card'] = !empty($twitter_card_type) ? $twitter_card_type : 'summary';
        
        $twitter_title = get_post_meta($post_id, '_prorank_twitter_title', true);
        if (!empty($twitter_title)) {
            $social_tags['twitter:title'] = $twitter_title;
        } elseif (!isset($social_tags['twitter:title'])) {
            // Fallback to OG title
            $social_tags['twitter:title'] = $social_tags['og:title'] ?? $post->post_title;
        }
        
        $twitter_description = get_post_meta($post_id, '_prorank_twitter_description', true);
        if (!empty($twitter_description)) {
            $social_tags['twitter:description'] = $twitter_description;
        } elseif (!isset($social_tags['twitter:description'])) {
            // Fallback to OG description
            $social_tags['twitter:description'] = $social_tags['og:description'] ?? '';
        }
        
        $twitter_image_id = get_post_meta($post_id, '_prorank_twitter_image_id', true);
        if (!empty($twitter_image_id)) {
            $image_url = wp_get_attachment_url((int) $twitter_image_id);
            if ($image_url) {
                $social_tags['twitter:image'] = $image_url;
            }
        } elseif (!isset($social_tags['twitter:image']) && isset($social_tags['og:image'])) {
            // Fallback to OG image
            $social_tags['twitter:image'] = $social_tags['og:image'];
        }
        
        // Default OG type and URL
        if (!isset($social_tags['og:type'])) {
            $social_tags['og:type'] = 'article';
        }
        if (!isset($social_tags['og:url'])) {
            $social_tags['og:url'] = get_permalink($post_id);
        }
        
        return $social_tags;
    }
    
    /**
     * Get meta data for a post
     *
     * @param int $post_id Post ID.
     * @return array
     */
    public function get_post_meta_data(int $post_id): array {
        return [
            'seo_title' => get_post_meta($post_id, '_prorank_seo_title', true) ?: '',
            'seo_description' => get_post_meta($post_id, '_prorank_seo_description', true) ?: '',
            'canonical_url' => get_post_meta($post_id, '_prorank_seo_canonical_url', true) ?: '',
            'meta_robots' => get_post_meta($post_id, '_prorank_meta_robots', true) ?: [],
            'og_title' => get_post_meta($post_id, '_prorank_og_title', true) ?: '',
            'og_description' => get_post_meta($post_id, '_prorank_og_description', true) ?: '',
            'og_image_id' => (int) get_post_meta($post_id, '_prorank_og_image_id', true),
            'twitter_title' => get_post_meta($post_id, '_prorank_twitter_title', true) ?: '',
            'twitter_description' => get_post_meta($post_id, '_prorank_twitter_description', true) ?: '',
            'twitter_image_id' => (int) get_post_meta($post_id, '_prorank_twitter_image_id', true),
            'twitter_card_type' => get_post_meta($post_id, '_prorank_twitter_card_type', true) ?: 'summary',
        ];
    }
}
