<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * Tag Sitemap Provider
 *
 * Generates sitemaps for tags
 *
 * @package ProRank\SEO\Modules\Indexing\Sitemaps
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\Indexing\Sitemaps;

defined( 'ABSPATH' ) || exit;

/**
 * Tag sitemap provider class
 */
class TagSitemapProvider extends CategorySitemapProvider {
    
    /**
     * Provider slug
     *
     * @var string
     */
    protected string $slug = 'post_tag';
    
    /**
     * Provider name
     *
     * @var string
     */
    protected string $name = 'Tags';
    
    /**
     * Check if provider is enabled
     *
     * @return bool
     */
    public function is_enabled(): bool {
        if (!$this->settings_manager) {
            return true; // Default to enabled
        }
        
        $settings = $this->settings_manager->get_settings('sitemaps');
        $enabled_taxonomies = $settings['include_taxonomies'] ?? ['category', 'post_tag'];
        
        return in_array('post_tag', $enabled_taxonomies, true);
    }
    
    /**
     * Get total number of terms
     *
     * @return int
     */
    protected function get_total_terms(): int {
        $settings = $this->settings_manager ? $this->settings_manager->get_settings('sitemaps') : [];
        $exclude_terms = $settings['exclude_terms'] ?? '';
        
        $args = [
            'taxonomy' => 'post_tag',
            'hide_empty' => $this->should_exclude_empty_terms(),
            'fields' => 'count',
        ];
        
        // Add term exclusions
        if (!empty($exclude_terms)) {
            $excluded_ids = array_map('intval', array_filter(explode(',', $exclude_terms)));
            if (!empty($excluded_ids)) {
                $args['exclude'] = $excluded_ids;
            }
        }
        
        $count = wp_count_terms($args);
        
        return is_numeric($count) ? (int) $count : 0;
    }
    
    /**
     * Get term URLs for sitemap
     *
     * @param int $page Page number
     * @return array
     */
    protected function get_term_urls(int $page): array {
        $offset = ($page - 1) * $this->max_entries;
        
        // Get settings
        $settings = $this->settings_manager ? $this->settings_manager->get_settings('sitemaps') : [];
        $exclude_terms = $settings['exclude_terms'] ?? '';
        
        $args = [
            'taxonomy' => 'post_tag',
            'hide_empty' => $this->should_exclude_empty_terms(),
            'number' => $this->max_entries,
            'offset' => $offset,
            'orderby' => 'count',
            'order' => 'DESC',
            'meta_query' => [
                'relation' => 'OR',
                [
                    'key' => '_prorank_seo_robots_noindex',
                    'value' => '1',
                    'compare' => '!='
                ],
                [
                    'key' => '_prorank_seo_robots_noindex',
                    'compare' => 'NOT EXISTS'
                ]
            ]
        ];
        
        // Add term exclusions
        if (!empty($exclude_terms)) {
            $excluded_ids = array_map('intval', array_filter(explode(',', $exclude_terms)));
            if (!empty($excluded_ids)) {
                $args['exclude'] = $excluded_ids;
            }
        }
        
        $terms = get_terms($args);
        $urls = [];
        
        if (!is_wp_error($terms) && !empty($terms)) {
            foreach ($terms as $term) {
                $url_data = [
                    'loc' => get_term_link($term),
                    'lastmod' => $this->get_term_last_modified($term->term_id),
                    'changefreq' => 'weekly',
                    'priority' => $this->get_term_priority($term)
                ];
                
                // Allow other modules to modify URL data
                $url_data = apply_filters('prorank_seo_sitemap_url_data', $url_data, $term, 'term');
                
                $urls[] = $url_data;
            }
        }
        
        return $urls;
    }
    
    /**
     * Get term priority based on post count
     *
     * @param \WP_Term $term Term object
     * @return float
     */
    protected function get_term_priority(\WP_Term $term): float {
        // Tags typically have lower priority than categories
        if ($term->count > 50) {
            return 0.6;
        } elseif ($term->count > 20) {
            return 0.5;
        } elseif ($term->count > 5) {
            return 0.4;
        } elseif ($term->count > 0) {
            return 0.3;
        }
        
        return 0.2;
    }
}