<?php
/**
 * Readability Analysis
 *
 * Provides readability scoring helpers.
 *
 * @package ProRank\SEO\Modules\Content
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\Content;

defined( 'ABSPATH' ) || exit;

/**
 * Readability class
 */
class Readability {
    /**
     * Calculate Flesch-Kincaid reading ease score
     *
     * @param string $text Input text.
     * @return float Readability score.
     */
    public static function calculate_flesch_kincaid(string $text): float {
        $clean_text = self::clean_text($text);

        if ($clean_text === '') {
            return 0.0;
        }

        $words = self::count_words($clean_text);
        $sentences = self::count_sentences($clean_text);
        $syllables = self::count_syllables($clean_text);

        if ($words === 0 || $sentences === 0) {
            return 0.0;
        }

        $score = 206.835 - (1.015 * ($words / $sentences)) - (84.6 * ($syllables / $words));

        return max(0.0, min(100.0, round($score, 1)));
    }

    /**
     * Get readability level description
     *
     * @param float $score Readability score.
     * @return string
     */
    public static function get_readability_level(float $score): string {
        if ($score >= 90) {
            return __('Very Easy', 'prorank-seo');
        }
        if ($score >= 80) {
            return __('Easy', 'prorank-seo');
        }
        if ($score >= 70) {
            return __('Fairly Easy', 'prorank-seo');
        }
        if ($score >= 60) {
            return __('Standard', 'prorank-seo');
        }
        if ($score >= 50) {
            return __('Fairly Difficult', 'prorank-seo');
        }
        if ($score >= 30) {
            return __('Difficult', 'prorank-seo');
        }
        return __('Very Difficult', 'prorank-seo');
    }

    /**
     * Get grade level (US) from score
     *
     * @param float $score Readability score.
     * @return string
     */
    public static function get_grade_level(float $score): string {
        if ($score >= 90) {
            return __('5th grade', 'prorank-seo');
        }
        if ($score >= 80) {
            return __('6th grade', 'prorank-seo');
        }
        if ($score >= 70) {
            return __('7th grade', 'prorank-seo');
        }
        if ($score >= 60) {
            return __('8th-9th grade', 'prorank-seo');
        }
        if ($score >= 50) {
            return __('10th-12th grade', 'prorank-seo');
        }
        if ($score >= 30) {
            return __('College', 'prorank-seo');
        }
        return __('College graduate', 'prorank-seo');
    }

    /**
     * Clean text for analysis
     *
     * @param string $text Input text.
     * @return string
     */
    private static function clean_text(string $text): string {
        $text = wp_strip_all_tags($text);
        $text = html_entity_decode($text, ENT_QUOTES);
        $text = preg_replace('/\s+/', ' ', $text);
        return trim((string) $text);
    }

    /**
     * Count words
     *
     * @param string $text Input text.
     * @return int
     */
    private static function count_words(string $text): int {
        $words = preg_split('/\s+/', trim($text));
        return is_array($words) ? count($words) : 0;
    }

    /**
     * Count sentences
     *
     * @param string $text Input text.
     * @return int
     */
    private static function count_sentences(string $text): int {
        $sentences = preg_split('/[.!?]+/', $text, -1, PREG_SPLIT_NO_EMPTY);
        return max(1, is_array($sentences) ? count($sentences) : 1);
    }

    /**
     * Count syllables
     *
     * @param string $text Input text.
     * @return int
     */
    private static function count_syllables(string $text): int {
        $text = strtolower($text);
        $text = preg_replace('/[^a-z]/', ' ', $text);
        $words = preg_split('/\s+/', trim($text));

        if (!is_array($words)) {
            return 0;
        }

        $count = 0;
        foreach ($words as $word) {
            $count += self::count_syllables_in_word($word);
        }

        return $count;
    }

    /**
     * Count syllables in a word
     *
     * @param string $word Word to analyze.
     * @return int
     */
    private static function count_syllables_in_word(string $word): int {
        if ($word === '') {
            return 0;
        }

        $word = preg_replace('/e$/', '', $word);
        $word = preg_replace('/[^aeiouy]/', ' ', $word);
        $syllables = preg_split('/\s+/', trim($word));

        if (!is_array($syllables)) {
            return 1;
        }

        $count = count($syllables);
        return $count > 0 ? $count : 1;
    }
}
