<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * Internal Linking Suggestions
 *
 * Provides basic internal linking suggestions for content.
 *
 * @package ProRank\SEO\Modules\Content
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\Content;

defined( 'ABSPATH' ) || exit;

use WP_Query;

/**
 * InternalLinking class
 */
class InternalLinking {
    /**
     * Maximum number of posts to scan for suggestions
     */
    private const MAX_POSTS_TO_SCAN = 50;

    /**
     * Maximum number of suggestions to return
     */
    private const MAX_SUGGESTIONS = 10;

    /**
     * Minimum anchor text length
     */
    private const MIN_ANCHOR_LENGTH = 3;

    /**
     * Minimum anchor word count
     */
    private const MIN_ANCHOR_WORDS = 2;

    /**
     * Maximum anchor words for suggestions
     */
    private const MAX_ANCHOR_WORDS = 6;

    /**
     * Maximum candidate anchors to evaluate
     */
    private const MAX_CANDIDATE_ANCHORS = 200;

    /**
     * Get basic internal linking suggestions
     *
     * @param int    $current_post_id      Current post ID.
     * @param string $current_post_content Current post content.
     * @param string $current_post_title   Current post title.
     * @return array Array of suggestions with title, url, and suggested_anchor.
     */
    public static function get_basic_suggestions(
        int $current_post_id,
        string $current_post_content,
        string $current_post_title
    ): array {
        $cleaned_content = self::prepare_content($current_post_content);

        $settings = self::get_internal_linking_settings();

        if (empty($settings['enabled'])) {
            return [];
        }

        $word_count = str_word_count($cleaned_content);
        if ($word_count < (int) ($settings['min_word_count'] ?? 0)) {
            return [];
        }

        $target_posts = self::get_target_posts($current_post_id, $settings);

        if (empty($target_posts)) {
            return [];
        }

        $suggestions = [];
        $max_suggestions = (int) ($settings['max_suggestions'] ?? self::MAX_SUGGESTIONS);
        if ($max_suggestions < 1) {
            $max_suggestions = 1;
        }
        $max_suggestions = min(self::MAX_SUGGESTIONS, $max_suggestions);
        $max_links_per_post = (int) ($settings['max_links_per_post'] ?? 0);
        if ($max_links_per_post > 0) {
            $max_suggestions = min($max_suggestions, $max_links_per_post);
        }
        $ignore_words = self::get_ignore_words($settings);

        $candidate_anchors = self::get_candidate_anchors($cleaned_content, self::MAX_CANDIDATE_ANCHORS);

        $current_tokens = self::get_content_tokens($cleaned_content);
        $current_terms = self::get_post_term_ids($current_post_id);
        $linked_urls = self::extract_linked_urls($current_post_content);

        foreach ($target_posts as $post) {
            if (!$post instanceof \WP_Post) {
                continue;
            }

            if ($post->ID === $current_post_id) {
                continue;
            }

            $target_url = get_permalink($post->ID);
            if ($target_url && self::is_url_already_linked($target_url, $linked_urls)) {
                continue;
            }

            if (self::is_noindex_post($post->ID)) {
                continue;
            }

            $profile = self::get_target_profile($post->ID, $post->post_title);
            $anchor_result = self::find_anchor_text(
                $cleaned_content,
                $post->post_title,
                $post->ID,
                $candidate_anchors,
                $profile
            );

            if (!$anchor_result || empty($anchor_result['text'])) {
                continue;
            }

            $anchor_text = $anchor_result['text'];
            if (self::is_anchor_ignored($anchor_text, $ignore_words)) {
                continue;
            }

            $content_score = self::calculate_token_overlap(
                $current_tokens,
                $profile['core_tokens'] ?? $profile['tokens']
            );
            $taxonomy_score = self::calculate_taxonomy_overlap($current_terms, $profile['term_ids']);
            $anchor_score = $anchor_result['score'];
            $final_score = (0.6 * $anchor_score) + (0.25 * $content_score) + (0.15 * $taxonomy_score);

            if ($final_score < 0.45) {
                continue;
            }

            $suggestions[] = [
                'title' => $post->post_title,
                'url' => $target_url,
                'suggested_anchor' => $anchor_text,
                'score' => $final_score,
            ];
        }

        usort($suggestions, static function($a, $b) {
            return ($b['score'] ?? 0) <=> ($a['score'] ?? 0);
        });

        return array_slice($suggestions, 0, $max_suggestions);
    }

    /**
     * Get target posts for linking suggestions
     *
     * @param int $exclude_post_id Post ID to exclude.
     * @return array
     */
    private static function get_target_posts(int $exclude_post_id, array $settings): array {
        $args = [
            'post_type'      => ['post', 'page'],
            'post_status'    => 'publish',
            'posts_per_page' => self::MAX_POSTS_TO_SCAN,
            'post__not_in'   => [$exclude_post_id],
            'orderby'        => 'modified',
            'order'          => 'DESC',
            'no_found_rows'  => true,
            'fields'         => 'ids',
        ];

        if (!empty($settings['link_post_types']) && is_array($settings['link_post_types'])) {
            $args['post_type'] = $settings['link_post_types'];
        }

        $args['post_type'] = apply_filters('prorank_seo_internal_linking_post_types', $args['post_type']);
        if (!is_array($args['post_type'])) {
            $args['post_type'] = array_filter((array) $args['post_type']);
        }
        if (empty($args['post_type'])) {
            $args['post_type'] = ['post', 'page'];
        }

        if (!empty($settings['exclude_categories']) && is_array($settings['exclude_categories'])) {
            $args['category__not_in'] = array_map('intval', $settings['exclude_categories']);
        }

        $query = new WP_Query($args);

        if (!$query->have_posts()) {
            return [];
        }

        $posts = [];
        foreach ($query->posts as $post_id) {
            $post = get_post($post_id);
            if ($post) {
                $posts[] = $post;
            }
        }

        return $posts;
    }

    /**
     * Find anchor text in content
     *
     * @param string $content    Content to search in.
     * @param string $post_title Target post title.
     * @param int    $post_id    Target post ID.
     * @return array|null
     */
    private static function find_anchor_text(
        string $content,
        string $post_title,
        int $post_id,
        array $candidate_anchors,
        array $profile
    ): ?array {
        $target_phrases = [];
        $target_tokens = [];

        $title_words = preg_split('/\s+/', trim($post_title));
        $title_word_count = is_array($title_words) ? count(array_filter($title_words)) : 0;

        if ($title_word_count > 0) {
            $target_phrases[] = $post_title;
            $target_phrases = array_merge($target_phrases, self::get_title_phrases($post_title, self::MAX_ANCHOR_WORDS));
        }

        $focus_keyword = get_post_meta($post_id, '_prorank_focus_keyword', true);
        if (empty($focus_keyword)) {
            $focus_keyword = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
        }
        if (!empty($focus_keyword)) {
            $target_phrases[] = (string) $focus_keyword;
        }

        $target_phrases = array_values(array_unique(array_filter(array_map('trim', $target_phrases))));

        $core_tokens = $profile['core_tokens'] ?? $profile['tokens'];
        $taxonomy_tokens = $profile['taxonomy_tokens'] ?? [];
        $target_tokens = array_values(
            array_unique(
                array_merge($core_tokens, self::get_tokens(implode(' ', $target_phrases)))
            )
        );

        $best_anchor = null;
        $best_score = 0.0;

        foreach ($candidate_anchors as $candidate) {
            if (empty($candidate['tokens'])) {
                continue;
            }
            $score = self::score_anchor_candidate($candidate, $target_phrases, $target_tokens, $taxonomy_tokens);
            if ($score > $best_score) {
                $best_score = $score;
                $best_anchor = $candidate['text'];
            }
        }

        if ($best_anchor !== null && $best_score >= 0.55) {
            return [
                'text' => $best_anchor,
                'score' => $best_score,
            ];
        }

        // Fallback: exact phrase match if present in content.
        foreach ($target_phrases as $phrase) {
            if ($phrase !== '' && stripos($content, $phrase) !== false) {
                $min_words = max(self::MIN_ANCHOR_WORDS, str_word_count($phrase));
                $anchor = self::extract_anchor_context($content, $phrase, $min_words);
                if ($anchor !== null) {
                    return [
                        'text' => $anchor,
                        'score' => 0.5,
                    ];
                }
            }
        }

        return null;
    }

    /**
     * Extract anchor text with context
     *
     * @param string $content Content to search in.
     * @param string $phrase  Phrase to find.
     * @return string|null
     */
    private static function extract_anchor_context(string $content, string $phrase, int $min_words = self::MIN_ANCHOR_WORDS): ?string {
        $pos = stripos($content, $phrase);
        if ($pos === false) {
            return null;
        }

        $before_pos = $pos;
        $after_pos = $pos + strlen($phrase);

        while ($before_pos > 0 && !self::is_word_boundary($content[$before_pos - 1])) {
            $before_pos--;
        }

        $content_length = strlen($content);
        while ($after_pos < $content_length && !self::is_word_boundary($content[$after_pos])) {
            $after_pos++;
        }

        $anchor = substr($content, $before_pos, $after_pos - $before_pos);
        $anchor = trim($anchor);

        $word_count = str_word_count($anchor);
        if ($word_count < max(1, $min_words)) {
            return null;
        }

        if (strlen($anchor) < self::MIN_ANCHOR_LENGTH || strlen($anchor) > 100) {
            return null;
        }

        return $anchor;
    }

    /**
     * Prepare content for anchor extraction.
     *
     * @param string $content Raw content.
     * @return string
     */
    private static function prepare_content(string $content): string {
        $content = wp_strip_all_tags($content);
        $content = html_entity_decode($content, ENT_QUOTES | ENT_HTML5);
        return trim($content);
    }

    /**
     * Build candidate anchors from content.
     *
     * @param string $content Content to scan.
     * @param int    $limit   Max candidates.
     * @return array
     */
    private static function get_candidate_anchors(string $content, int $limit): array {
        if ($content === '') {
            return [];
        }

        $sentences = preg_split('/(?<=[.!?])\s+/', $content);
        $candidates = [];
        $seen = [];

        foreach ($sentences as $sentence) {
            $sentence = trim($sentence);
            if ($sentence === '') {
                continue;
            }
            $words = preg_split('/\s+/', $sentence);
            $word_count = is_array($words) ? count($words) : 0;
            if ($word_count < self::MIN_ANCHOR_WORDS) {
                continue;
            }

            $max_words = min(self::MAX_ANCHOR_WORDS, $word_count);
            for ($length = $max_words; $length >= self::MIN_ANCHOR_WORDS; $length--) {
                for ($i = 0; $i <= $word_count - $length; $i++) {
                    $slice = array_slice($words, $i, $length);
                    $phrase = trim(implode(' ', $slice));
                    if ($phrase === '') {
                        continue;
                    }
                    $tokens = self::get_tokens($phrase);
                    if (count($tokens) < self::MIN_ANCHOR_WORDS) {
                        continue;
                    }
                    $key = strtolower($phrase);
                    if (isset($seen[$key])) {
                        continue;
                    }
                    $seen[$key] = true;
                    $candidates[] = [
                        'text' => $phrase,
                        'tokens' => $tokens,
                    ];
                    if (count($candidates) >= $limit) {
                        return $candidates;
                    }
                }
            }
        }

        return $candidates;
    }

    /**
     * Tokenize text into significant tokens.
     *
     * @param string $text Input text.
     * @return array
     */
    private static function get_tokens(string $text): array {
        $raw_words = preg_split('/\s+/', strtolower($text));
        $tokens = [];
        foreach ($raw_words as $word) {
            $word = trim($word, ".,!?;:\"'()[]{}<>-");
            if ($word === '') {
                continue;
            }
            if (!self::is_significant_word($word)) {
                continue;
            }
            $tokens[] = $word;
        }
        return array_values(array_unique($tokens));
    }

    /**
     * Score candidate anchor against target phrases/tokens.
     *
     * @param array $candidate Candidate anchor data.
     * @param array $target_phrases Target phrases.
     * @param array $core_tokens Target core tokens.
     * @param array $taxonomy_tokens Target taxonomy tokens.
     * @return float
     */
    private static function score_anchor_candidate(
        array $candidate,
        array $target_phrases,
        array $core_tokens,
        array $taxonomy_tokens = []
    ): float {
        $candidate_tokens = $candidate['tokens'] ?? [];
        if (empty($candidate_tokens) || (empty($core_tokens) && empty($taxonomy_tokens))) {
            return 0.0;
        }

        $core_overlap = array_intersect($candidate_tokens, $core_tokens);
        $core_overlap_count = count($core_overlap);

        // Require overlap with core tokens when available to avoid irrelevant anchors.
        if ($core_overlap_count === 0) {
            if (!empty($core_tokens)) {
                return 0.0;
            }
            $core_tokens = $taxonomy_tokens;
            $core_overlap = array_intersect($candidate_tokens, $core_tokens);
            $core_overlap_count = count($core_overlap);
            if ($core_overlap_count === 0) {
                return 0.0;
            }
        }

        if ($core_overlap_count < 2 && count($core_tokens) >= 3 && count($candidate_tokens) >= 3) {
            return 0.0;
        }

        $target_score = $core_overlap_count / max(1, count($core_tokens));
        $candidate_score = $core_overlap_count / max(1, count($candidate_tokens));
        $score = ($target_score * 0.7) + ($candidate_score * 0.3);

        if (!empty($taxonomy_tokens) && $core_overlap_count > 0) {
            $taxonomy_overlap = array_intersect($candidate_tokens, $taxonomy_tokens);
            if (!empty($taxonomy_overlap)) {
                $score += 0.05 * (count($taxonomy_overlap) / max(1, count($taxonomy_tokens)));
            }
        }

        $candidate_text = strtolower($candidate['text'] ?? '');
        foreach ($target_phrases as $phrase) {
            if ($phrase === '') {
                continue;
            }
            $phrase_lower = strtolower($phrase);
            if ($candidate_text === $phrase_lower) {
                $score += 0.35;
                break;
            }
            if (stripos($candidate_text, $phrase_lower) !== false) {
                $score += 0.15;
            }
        }

        $word_count = str_word_count($candidate_text);
        if ($word_count > self::MAX_ANCHOR_WORDS) {
            $score *= 0.8;
        } elseif ($word_count >= 2 && $word_count <= 5) {
            $score += 0.1;
        } elseif ($word_count === 1) {
            $score -= 0.1;
        }

        return $score;
    }

    /**
     * Build target profile tokens and taxonomy.
     *
     * @param int    $post_id Post ID.
     * @param string $title   Post title.
     * @return array
     */
    private static function get_target_profile(int $post_id, string $title): array {
        $core_tokens = self::get_tokens($title);
        $phrases = [];

        $focus_keyword = get_post_meta($post_id, '_prorank_focus_keyword', true);
        if (empty($focus_keyword)) {
            $focus_keyword = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
        }
        if (!empty($focus_keyword)) {
            $phrases[] = (string) $focus_keyword;
            $core_tokens = array_merge($core_tokens, self::get_tokens((string) $focus_keyword));
        }

        $slug = get_post_field('post_name', $post_id);
        if (!empty($slug)) {
            $core_tokens = array_merge($core_tokens, self::get_tokens(str_replace('-', ' ', (string) $slug)));
        }

        $term_names = [];
        foreach (['category', 'post_tag'] as $taxonomy) {
            $terms = get_the_terms($post_id, $taxonomy);
            if (!empty($terms) && !is_wp_error($terms)) {
                foreach ($terms as $term) {
                    $term_names[] = $term->name;
                }
            }
        }
        $taxonomy_tokens = [];
        foreach ($term_names as $name) {
            $taxonomy_tokens = array_merge($taxonomy_tokens, self::get_tokens($name));
        }

        $core_tokens = array_values(array_unique($core_tokens));
        $taxonomy_tokens = array_values(array_unique($taxonomy_tokens));
        $tokens = array_values(array_unique(array_merge($core_tokens, $taxonomy_tokens)));

        return [
            'tokens' => $tokens,
            'core_tokens' => $core_tokens,
            'taxonomy_tokens' => $taxonomy_tokens,
            'phrases' => $phrases,
            'term_ids' => self::get_post_term_ids($post_id),
        ];
    }

    /**
     * Get content tokens ranked by frequency.
     *
     * @param string $content Clean content.
     * @param int    $limit   Max tokens.
     * @return array
     */
    private static function get_content_tokens(string $content, int $limit = 80): array {
        $tokens = self::get_tokens($content);
        if (empty($tokens)) {
            return [];
        }
        $counts = array_count_values($tokens);
        arsort($counts);
        return array_slice(array_keys($counts), 0, $limit);
    }

    /**
     * Calculate overlap score between token sets.
     *
     * @param array $tokens_a Tokens A.
     * @param array $tokens_b Tokens B.
     * @return float
     */
    private static function calculate_token_overlap(array $tokens_a, array $tokens_b): float {
        if (empty($tokens_a) || empty($tokens_b)) {
            return 0.0;
        }
        $overlap = array_intersect($tokens_a, $tokens_b);
        return count($overlap) / max(1, count(array_unique(array_merge($tokens_a, $tokens_b))));
    }

    /**
     * Get term IDs for categories and tags.
     *
     * @param int $post_id Post ID.
     * @return array
     */
    private static function get_post_term_ids(int $post_id): array {
        $term_ids = [];
        foreach (['category', 'post_tag'] as $taxonomy) {
            $terms = get_the_terms($post_id, $taxonomy);
            if (!empty($terms) && !is_wp_error($terms)) {
                foreach ($terms as $term) {
                    $term_ids[] = (int) $term->term_id;
                }
            }
        }
        return array_values(array_unique(array_filter($term_ids)));
    }

    /**
     * Calculate taxonomy overlap between current and target.
     *
     * @param array $current_terms Current term IDs.
     * @param array $target_terms  Target term IDs.
     * @return float
     */
    private static function calculate_taxonomy_overlap(array $current_terms, array $target_terms): float {
        if (empty($current_terms) || empty($target_terms)) {
            return 0.0;
        }
        $overlap = array_intersect($current_terms, $target_terms);
        return count($overlap) / max(1, count($current_terms));
    }

    /**
     * Extract linked URLs from content.
     *
     * @param string $content Raw content.
     * @return array
     */
    private static function extract_linked_urls(string $content): array {
        $urls = [];
        if (preg_match_all('/href=[\"\\\']([^\"\\\']+)[\"\\\']/i', $content, $matches)) {
            foreach ($matches[1] as $url) {
                $normalized = self::normalize_url((string) $url);
                if ($normalized) {
                    $urls[] = $normalized;
                }
            }
        }
        return array_values(array_unique($urls));
    }

    /**
     * Check if a URL is already linked in content.
     *
     * @param string $url   Target URL.
     * @param array  $links Linked URLs.
     * @return bool
     */
    private static function is_url_already_linked(string $url, array $links): bool {
        $normalized = self::normalize_url($url);
        if (!$normalized) {
            return false;
        }
        return in_array($normalized, $links, true);
    }

    /**
     * Normalize URL to host + path.
     *
     * @param string $url URL to normalize.
     * @return string|null
     */
    private static function normalize_url(string $url): ?string {
        $parsed = wp_parse_url($url);
        if (empty($parsed)) {
            return null;
        }
        $host = $parsed['host'] ?? '';
        $path = $parsed['path'] ?? '';
        if ($host === '' && $path === '') {
            return null;
        }
        $normalized = strtolower($host . $path);
        return rtrim($normalized, '/');
    }

    /**
     * Determine if a post is noindex.
     *
     * @param int $post_id Post ID.
     * @return bool
     */
    private static function is_noindex_post(int $post_id): bool {
        $robots = get_post_meta($post_id, '_prorank_meta_robots', true);
        if (is_array($robots) && in_array('noindex', $robots, true)) {
            return true;
        }
        $prorank_noindex = get_post_meta($post_id, '_prorank_seo_noindex', true);
        if ($prorank_noindex === '1' || $prorank_noindex === 'on') {
            return true;
        }
        $robots_noindex = get_post_meta($post_id, '_prorank_seo_robots_noindex', true);
        if ($robots_noindex === '1') {
            return true;
        }
        $yoast_noindex = get_post_meta($post_id, '_yoast_wpseo_meta-robots-noindex', true);
        if ($yoast_noindex === '1') {
            return true;
        }
        return false;
    }

    /**
     * Get significant words from a title
     *
     * @param string $title Post title.
     * @return array
     */
    private static function get_significant_words(string $title): array {
        $stop_words = self::get_stop_words();
        $words = preg_split('/\s+/', strtolower($title));

        $significant_words = [];
        foreach ($words as $word) {
            $word = trim($word, '.,!?;:"\'-');
            if (strlen($word) >= self::MIN_ANCHOR_LENGTH && !in_array($word, $stop_words, true)) {
                $significant_words[] = $word;
            }
        }

        return $significant_words;
    }

    /**
     * Get title phrases (2-3 word chunks) to match more descriptive anchors.
     *
     * @param string $title Title to extract phrases from.
     * @param int    $max_words Maximum words per phrase.
     * @return array
     */
    private static function get_title_phrases(string $title, int $max_words = 3): array {
        $raw_words = preg_split('/\s+/', $title);
        $words = [];
        foreach ($raw_words as $word) {
            $word = trim($word, '.,!?;:"\'-');
            if ($word !== '') {
                $words[] = $word;
            }
        }

        $count = count($words);
        if ($count < 2) {
            return [];
        }

        $stop_words = self::get_stop_words();
        $phrases = [];
        $max_words = min($max_words, $count);

        for ($length = $max_words; $length >= 2; $length--) {
            for ($i = 0; $i <= $count - $length; $i++) {
                $slice = array_slice($words, $i, $length);
                $significant_count = 0;
                foreach ($slice as $word) {
                    $lower = strtolower($word);
                    if (self::is_significant_word($lower)) {
                        $significant_count++;
                    }
                }
                if ($significant_count < 2) {
                    continue;
                }
                $phrases[] = implode(' ', $slice);
            }
        }

        return array_values(array_unique($phrases));
    }

    /**
     * Stop words list for anchor matching.
     *
     * @return array
     */
    private static function get_stop_words(): array {
        return [
            'a', 'an', 'and', 'are', 'as', 'at', 'be', 'been', 'being', 'by', 'for',
            'from', 'has', 'have', 'had', 'he', 'her', 'hers', 'his', 'i', 'in',
            'is', 'it', 'its', 'me', 'my', 'of', 'on', 'or', 'our', 'ours',
            'she', 'so', 'that', 'the', 'their', 'theirs', 'them', 'they', 'this',
            'those', 'to', 'was', 'were', 'will', 'with', 'you', 'your', 'yours',
            'what', 'when', 'where', 'who', 'whom', 'which', 'why', 'how', 'do',
            'does', 'did', 'can', 'could', 'should', 'would', 'may', 'might',
            'just', 'into', 'over', 'under', 'between', 'above', 'below', 'about',
            'after', 'before', 'again', 'further', 'then', 'once', 'here', 'there',
        ];
    }

    /**
     * Determine if a word is significant for anchor selection.
     *
     * @param string $word Word to check.
     * @return bool
     */
    private static function is_significant_word(string $word): bool {
        if (strlen($word) < self::MIN_ANCHOR_LENGTH) {
            return false;
        }
        $stop_words = self::get_stop_words();
        $weak_words = self::get_weak_words();
        if (in_array($word, $stop_words, true) || in_array($word, $weak_words, true)) {
            return false;
        }
        return true;
    }

    /**
     * Extra weak words beyond stop words.
     *
     * @return array
     */
    private static function get_weak_words(): array {
        return [
            'about', 'after', 'before', 'during', 'because', 'into', 'over', 'under',
            'between', 'within', 'without', 'very', 'more', 'most', 'less', 'best',
            'guide', 'tips', 'ways', 'things', 'simple', 'easy', 'quick',
            'review', 'reviews', 'tutorial', 'tutorials', 'introduction', 'overview',
            'comparison', 'comparisons', 'ultimate', 'complete', 'beginner', 'beginners',
            'learn', 'learning', 'using', 'use', 'vs', 'versus',
        ];
    }

    /**
     * Determine if a character is a word boundary
     *
     * @param string $char Character to check.
     * @return bool
     */
    private static function is_word_boundary(string $char): bool {
        return (bool) preg_match('/\s|[.,!?;:"\'()]/', $char);
    }

    /**
     * Get internal linking settings with defaults.
     *
     * @return array
     */
    private static function get_internal_linking_settings(): array {
        if (function_exists('prorank_get_internal_linking_settings')) {
            $settings = \prorank_get_internal_linking_settings();
        } else {
            $settings = get_option('prorank_seo_internal_linking', []);
            if (!is_array($settings)) {
                $settings = [];
            }
        }

        $defaults = [
            'enabled' => true,
            'max_suggestions' => self::MAX_SUGGESTIONS,
            'min_word_count' => 100,
            'link_post_types' => ['post', 'page'],
            'exclude_categories' => [],
            'ignore_words' => '',
            'ignored_words' => '',
        ];

        return array_merge($defaults, $settings);
    }

    /**
     * Parse ignore words list.
     *
     * @param array $settings Settings array.
     * @return array
     */
    private static function get_ignore_words(array $settings): array {
        $raw = $settings['ignore_words'] ?? $settings['ignored_words'] ?? '';
        $parts = preg_split('/[,\\n\\r]+/', (string) $raw);
        $words = [];
        foreach ($parts as $part) {
            $word = strtolower(trim($part));
            if ($word !== '') {
                $words[] = $word;
            }
        }
        return array_values(array_unique($words));
    }

    /**
     * Check if anchor should be ignored.
     *
     * @param string $anchor Anchor text.
     * @param array  $ignore_words Ignore words list.
     * @return bool
     */
    private static function is_anchor_ignored(string $anchor, array $ignore_words): bool {
        if (empty($ignore_words)) {
            return false;
        }

        $anchor_lower = strtolower($anchor);
        foreach ($ignore_words as $word) {
            if ($word === '') {
                continue;
            }
            if (preg_match('/\\b' . preg_quote($word, '/') . '\\b/i', $anchor_lower)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get link density (percentage of linked words)
     *
     * @param string $content Post content.
     * @return float
     */
    public static function get_link_density(string $content): float {
        $text = wp_strip_all_tags($content);
        $word_count = str_word_count($text);
        if ($word_count === 0) {
            return 0.0;
        }

        preg_match_all('/<a\s[^>]*>(.*?)<\/a>/i', $content, $matches);
        $linked_text = implode(' ', $matches[1] ?? []);
        $linked_words = str_word_count(wp_strip_all_tags($linked_text));

        return round(($linked_words / $word_count) * 100, 2);
    }
}
