<?php
/**
 * Virtual text file responses for root requests.
 *
 * @package ProRank\SEO\Frontend
 */

declare(strict_types=1);

namespace ProRank\SEO\Frontend;

use ProRank\SEO\Core\Config\Settings;
use ProRank\SEO\Core\Services\LlmsTxtService;

defined( 'ABSPATH' ) || exit;

class VirtualTxtFiles {
    /**
     * Initialize hooks.
     */
    public function init(): void {
        add_action( 'template_redirect', [ $this, 'maybe_serve_virtual_file' ], 0 );
    }

    /**
     * Serve virtual root files (llms.txt, llms-full.txt, IndexNow key).
     */
    public function maybe_serve_virtual_file(): void {
        if ( is_admin() || wp_doing_ajax() || ( defined( 'REST_REQUEST' ) && REST_REQUEST ) ) {
            return;
        }

        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        if ( $request_uri === '' ) {
            return;
        }

        $path = wp_parse_url( $request_uri, PHP_URL_PATH );
        if ( ! is_string( $path ) || $path === '' ) {
            return;
        }

        $path = trim( $path, '/' );
        if ( $path === '' ) {
            return;
        }

        if ( $path === 'llms.txt' ) {
            $content = $this->get_llms_content( 'index' );
            if ( $content !== '' ) {
                $this->send_text_response( $content );
            }
            return;
        }

        if ( $path === 'llms-full.txt' ) {
            $content = $this->get_llms_content( 'full' );
            if ( $content !== '' ) {
                $this->send_text_response( $content );
            }
            return;
        }

        $indexnow_key = $this->get_indexnow_key();
        if ( $indexnow_key !== '' && $path === $indexnow_key . '.txt' ) {
            $this->send_text_response( $indexnow_key );
        }
    }

    /**
     * Get IndexNow API key for virtual key file.
     *
     * @return string
     */
    private function get_indexnow_key(): string {
        $settings = Settings::get( 'robots_indexing', [] );
        if ( ! is_array( $settings ) ) {
            return '';
        }

        if ( empty( $settings['enable_indexnow'] ) ) {
            return '';
        }

        $key = $settings['indexnow_api_key'] ?? '';
        return is_string( $key ) ? trim( $key ) : '';
    }

    /**
     * Get LLMs content from cache or local storage.
     *
     * @param string $type index|full
     * @return string
     */
    private function get_llms_content( string $type ): string {
        $settings = get_option( 'prorank_llms_txt_settings', [] );
        if ( ! is_array( $settings ) || empty( $settings['enabled'] ) ) {
            return '';
        }

        $transient_key = $type === 'full' ? 'prorank_llms_full_txt' : 'prorank_llms_txt';
        $cached = get_transient( $transient_key );
        if ( is_string( $cached ) && $cached !== '' ) {
            return $cached;
        }

        $llms_dir = $this->get_llms_dir();
        $file = $type === 'full' ? $llms_dir . '/llms-full.txt' : $llms_dir . '/llms.txt';
        if ( is_readable( $file ) ) {
            global $wp_filesystem;
            if ( ! function_exists( 'WP_Filesystem' ) ) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }
            WP_Filesystem();
            if ( $wp_filesystem ) {
                $content = $wp_filesystem->get_contents( $file );
                if ( is_string( $content ) && $content !== '' ) {
                    set_transient( $transient_key, $content, WEEK_IN_SECONDS );
                    return $content;
                }
            }
        }

        $service = new LlmsTxtService();
        $content = $type === 'full' ? $service->generate_full() : $service->generate_index();
        if ( is_string( $content ) && $content !== '' ) {
            set_transient( $transient_key, $content, WEEK_IN_SECONDS );
            return $content;
        }

        return '';
    }

    /**
     * Get LLMs storage directory (legacy or new).
     *
     * @return string
     */
    private function get_llms_dir(): string {
        $upload_dir = wp_upload_dir();
        $legacy_dir = trailingslashit( $upload_dir['basedir'] ) . 'prorank-llms';

        if ( is_dir( $legacy_dir ) ) {
            return $legacy_dir;
        }

        return trailingslashit( $upload_dir['basedir'] ) . 'prorank-seo/llms';
    }

    /**
     * Send plain text response and exit.
     *
     * @param string $content
     * @return void
     */
    private function send_text_response( string $content ): void {
        nocache_headers();
        header( 'Content-Type: text/plain; charset=' . get_option( 'blog_charset' ) );
        echo $content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Plain text output.
        exit;
    }
}
