<?php
/**
 * Lightweight RUM beacon for INP/CLS/LCP
 *
 * Injects an inline script to collect web vitals and POST to a REST endpoint.
 *
 * @package ProRank\SEO\Frontend
 */

declare(strict_types=1);

namespace ProRank\SEO\Frontend;

defined( 'ABSPATH' ) || exit;

class RumBeacon {
    public function init(): void {
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_beacon_script' ], 3 );
    }

    public function enqueue_beacon_script(): void {
        if ( is_admin() ) {
            return;
        }
        if ( function_exists( 'prorank_is_rum_enabled' ) && ! prorank_is_rum_enabled() ) {
            return;
        }

        $endpoint = esc_url_raw( rest_url( 'prorank-seo/v1/performance/rum' ) );

        // Register placeholder script handle for inline script
        wp_register_script( 'prorank-rum-beacon', '', [], PRORANK_VERSION, false );
        wp_enqueue_script( 'prorank-rum-beacon' );

        $script = '
        (function() {
          if (!("PerformanceObserver" in window)) return;
          var lcp = 0, cls = 0, inp = 0;

          try {
            var lcpObs = new PerformanceObserver(function(list) {
              var entries = list.getEntries();
              var last = entries[entries.length - 1];
              if (last) lcp = last.renderTime || last.loadTime || last.startTime || 0;
            });
            lcpObs.observe({type:"largest-contentful-paint", buffered:true});
          } catch(e){}

          try {
            var clsObs = new PerformanceObserver(function(list) {
              for (var i = 0; i < list.getEntries().length; i++) {
                var entry = list.getEntries()[i];
                if (!entry.hadRecentInput) cls += entry.value || 0;
              }
            });
            clsObs.observe({type:"layout-shift", buffered:true});
          } catch(e){}

          try {
            var inpObs = new PerformanceObserver(function(list) {
              for (var i = 0; i < list.getEntries().length; i++) {
                var entry = list.getEntries()[i];
                var dur = entry.duration || entry.processingEnd - entry.startTime || 0;
                if (dur > inp) inp = dur;
              }
            });
            inpObs.observe({type:"event", buffered:true, durationThreshold:0});
          } catch(e){}

          function sendBeacon() {
            var payload = {
              url: location.href,
              lcp: Math.round(lcp),
              cls: Number(cls.toFixed(3)),
              inp: Math.round(inp)
            };
            var blob = new Blob([JSON.stringify(payload)], {type:"application/json"});
            navigator.sendBeacon("' . esc_js( $endpoint ) . '", blob);
          }

          document.addEventListener("visibilitychange", function() {
            if (document.visibilityState === "hidden") {
              sendBeacon();
            }
          });
          window.addEventListener("pagehide", sendBeacon);
        })();
        ';

        wp_add_inline_script( 'prorank-rum-beacon', $script );
    }
}
