<?php
/**
 * Wizard State REST API Controller
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Request;
use WP_REST_Response;

/**
 * Class WizardStateController
 *
 * Handles REST API endpoints for wizard state persistence
 */
class WizardStateController extends BaseController {
    /**
     * Option key for storing wizard state
     *
     * @var string
     */
    protected string $option_key = 'prorank_wizard_state_server';

    /**
     * Register REST API routes
     *
     * @return void
     */
    public function register_routes(): void {
        register_rest_route($this->namespace, '/wizard/state', [
            [
                'methods'             => \WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_state'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
            [
                'methods'             => \WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'save_state'],
                'permission_callback' => [$this, 'check_admin_permission'],
            ],
        ]);
    }

    /**
     * Get the current wizard state
     *
     * @param WP_REST_Request $request REST request object.
     * @return WP_REST_Response
     */
    public function get_state(WP_REST_Request $request): WP_REST_Response {
        $stored = get_option($this->option_key, []);
        return new WP_REST_Response([
            'success' => true,
            'data'    => $stored,
        ], 200);
    }

    /**
     * Save the wizard state
     *
     * @param WP_REST_Request $request REST request object.
     * @return WP_REST_Response
     */
    public function save_state(WP_REST_Request $request): WP_REST_Response {
        $body = $request->get_json_params();
        if (!is_array($body)) {
            return $this->error(__('Invalid payload', 'prorank-seo'), 'invalid_payload', 400);
        }

        $state = [
            'currentStep'    => $body['currentStep'] ?? 0,
            'completedSteps' => is_array($body['completedSteps'] ?? null) ? $body['completedSteps'] : [],
            'wizardData'     => is_array($body['wizardData'] ?? null) ? $body['wizardData'] : [],
            'updated_at'     => current_time('mysql'),
            'updated_by'     => get_current_user_id(),
        ];

        update_option($this->option_key, $state, false);

        return new WP_REST_Response([
            'success' => true,
            'data'    => $state,
        ], 200);
    }
}
