<?php

declare(strict_types=1);

/**
 * Test Content Creation Endpoint
 *
 * @package ProRank\SEO\Core\RestApi
 */

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Request;
use WP_REST_Response;

/**
 * Register test content endpoint
 */
add_action('rest_api_init', function() {
    register_rest_route('prorank-seo/v1', '/test/create-content', [
        'methods' => \WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\create_test_content',
        'permission_callback' => function() {
            return current_user_can('manage_options');
        },
    ]);
});

/**
 * Create test content for Visual Link Map
 */
function create_test_content(WP_REST_Request $request): WP_REST_Response {
    $created_items = [
        'pages' => [],
        'posts' => [],
        'orphans' => [],
    ];
    
    // Create main pages with interconnected links
    $pages = [
        [
            'post_title' => 'Home',
            'post_content' => 'Welcome to our website. Check out our About page and Services.',
            'post_type' => 'page',
            'post_status' => 'publish',
            'post_name' => 'home-test',
            'meta_input' => [
                '_prorank_focus_keyword' => 'home page website',
            ],
        ],
        [
            'post_title' => 'About Us',
            'post_content' => 'Learn about our company history and mission. Meet our Team.',
            'post_type' => 'page',
            'post_status' => 'publish',
            'post_name' => 'about-test',
            'meta_input' => [
                '_prorank_focus_keyword' => 'about company',
            ],
        ],
        [
            'post_title' => 'Services',
            'post_content' => 'We offer various professional services. View our Portfolio.',
            'post_type' => 'page',
            'post_status' => 'publish',
            'post_name' => 'services-test',
            'meta_input' => [
                '_prorank_focus_keyword' => 'services solutions',
            ],
        ],
        [
            'post_title' => 'Products',
            'post_content' => 'Browse our product catalog. Check Pricing for details.',
            'post_type' => 'page',
            'post_status' => 'publish',
            'post_name' => 'products-test',
            'meta_input' => [
                '_prorank_focus_keyword' => 'products catalog',
            ],
        ],
        [
            'post_title' => 'Contact',
            'post_content' => 'Get in touch with us. Find our Location and Support hours.',
            'post_type' => 'page',
            'post_status' => 'publish',
            'post_name' => 'contact-test',
            'meta_input' => [
                '_prorank_focus_keyword' => 'contact support',
            ],
        ],
        [
            'post_title' => 'Team',
            'post_content' => 'Meet our talented team members and their expertise.',
            'post_type' => 'page',
            'post_status' => 'publish',
            'post_name' => 'team-test',
            'meta_input' => [
                '_prorank_focus_keyword' => 'team members staff',
            ],
        ],
        [
            'post_title' => 'Portfolio',
            'post_content' => 'View our recent work and case studies.',
            'post_type' => 'page',
            'post_status' => 'publish',
            'post_name' => 'portfolio-test',
            'meta_input' => [
                '_prorank_focus_keyword' => 'portfolio work',
            ],
        ],
        [
            'post_title' => 'Pricing',
            'post_content' => 'Our competitive pricing plans and packages.',
            'post_type' => 'page',
            'post_status' => 'publish',
            'post_name' => 'pricing-test',
            'meta_input' => [
                '_prorank_focus_keyword' => 'pricing plans',
            ],
        ],
    ];
    
    // Create blog posts
    $posts = [
        [
            'post_title' => 'Getting Started with SEO in 2025',
            'post_content' => 'Learn the fundamentals of search engine optimization. This guide covers keyword research, on-page optimization, and link building strategies.',
            'post_type' => 'post',
            'post_status' => 'publish',
            'post_category' => [1], // Uncategorized
            'meta_input' => [
                '_prorank_focus_keyword' => 'SEO guide basics',
            ],
        ],
        [
            'post_title' => 'Internal Linking Best Practices',
            'post_content' => 'Discover effective internal linking strategies to improve your site structure and SEO rankings.',
            'post_type' => 'post',
            'post_status' => 'publish',
            'post_category' => [1],
            'meta_input' => [
                '_prorank_focus_keyword' => 'internal linking SEO',
            ],
        ],
        [
            'post_title' => 'Content Marketing Strategies That Work',
            'post_content' => 'Create engaging content that drives traffic and conversions. Learn about content planning and distribution.',
            'post_type' => 'post',
            'post_status' => 'publish',
            'post_category' => [1],
            'meta_input' => [
                '_prorank_focus_keyword' => 'content marketing strategy',
            ],
        ],
        [
            'post_title' => 'Technical SEO Checklist',
            'post_content' => 'Ensure your website meets all technical SEO requirements. From site speed to mobile optimization.',
            'post_type' => 'post',
            'post_status' => 'publish',
            'post_category' => [1],
            'meta_input' => [
                '_prorank_focus_keyword' => 'technical SEO checklist',
            ],
        ],
        [
            'post_title' => 'Local SEO Guide for Small Businesses',
            'post_content' => 'Optimize your business for local search results. Set up Google Business Profile and local citations.',
            'post_type' => 'post',
            'post_status' => 'publish',
            'post_category' => [1],
            'meta_input' => [
                '_prorank_focus_keyword' => 'local SEO business',
            ],
        ],
    ];
    
    // Create orphan pages (no links pointing to them)
    $orphans = [
        [
            'post_title' => 'Privacy Policy',
            'post_content' => 'Our privacy policy outlines how we collect, use, and protect your personal information.',
            'post_type' => 'page',
            'post_status' => 'publish',
            'post_name' => 'privacy-policy-test',
        ],
        [
            'post_title' => 'Terms of Service',
            'post_content' => 'Terms and conditions for using our services and website.',
            'post_type' => 'page',
            'post_status' => 'publish',
            'post_name' => 'terms-test',
        ],
        [
            'post_title' => 'Cookie Policy',
            'post_content' => 'Information about how we use cookies on our website.',
            'post_type' => 'page',
            'post_status' => 'publish',
            'post_name' => 'cookie-policy-test',
        ],
    ];
    
    // Create pages
    foreach ($pages as $page_data) {
        $page_id = wp_insert_post($page_data);
        if (!is_wp_error($page_id)) {
            $created_items['pages'][$page_data['post_name']] = $page_id;
        }
    }
    
    // Create posts
    foreach ($posts as $post_data) {
        $post_id = wp_insert_post($post_data);
        if (!is_wp_error($post_id)) {
            $created_items['posts'][] = $post_id;
        }
    }
    
    // Create orphan pages
    foreach ($orphans as $orphan_data) {
        $orphan_id = wp_insert_post($orphan_data);
        if (!is_wp_error($orphan_id)) {
            $created_items['orphans'][] = $orphan_id;
        }
    }
    
    // Now update content with real internal links
    if (!empty($created_items['pages'])) {
        $pages = $created_items['pages'];
        
        // Update Home page with links
        if (isset($pages['home-test'])) {
            $content = sprintf(
                'Welcome to our website. Check out our <a href="%s">About page</a> and <a href="%s">Services</a>. Browse our <a href="%s">Products</a> or <a href="%s">Contact us</a>.',
                get_permalink($pages['about-test']),
                get_permalink($pages['services-test']),
                get_permalink($pages['products-test']),
                get_permalink($pages['contact-test'])
            );
            wp_update_post(['ID' => $pages['home-test'], 'post_content' => $content]);
        }
        
        // Update About page with links
        if (isset($pages['about-test'])) {
            $content = sprintf(
                'Learn about our company history and mission. Meet our <a href="%s">Team</a> and view our <a href="%s">Portfolio</a>. Return to <a href="%s">Home</a>.',
                get_permalink($pages['team-test']),
                get_permalink($pages['portfolio-test']),
                get_permalink($pages['home-test'])
            );
            wp_update_post(['ID' => $pages['about-test'], 'post_content' => $content]);
        }
        
        // Update Services page with links
        if (isset($pages['services-test'])) {
            $content = sprintf(
                'We offer various professional services. View our <a href="%s">Portfolio</a> for examples. Check our <a href="%s">Pricing</a> or <a href="%s">Contact us</a> for a quote.',
                get_permalink($pages['portfolio-test']),
                get_permalink($pages['pricing-test']),
                get_permalink($pages['contact-test'])
            );
            wp_update_post(['ID' => $pages['services-test'], 'post_content' => $content]);
        }
        
        // Update Products page with links
        if (isset($pages['products-test'])) {
            $content = sprintf(
                'Browse our product catalog. Check <a href="%s">Pricing</a> for details. View our <a href="%s">Services</a> or <a href="%s">Contact sales</a>.',
                get_permalink($pages['pricing-test']),
                get_permalink($pages['services-test']),
                get_permalink($pages['contact-test'])
            );
            wp_update_post(['ID' => $pages['products-test'], 'post_content' => $content]);
        }
        
        // Update Contact page with links
        if (isset($pages['contact-test'])) {
            $content = sprintf(
                'Get in touch with us. Learn <a href="%s">About us</a> or view our <a href="%s">Services</a>. Return to <a href="%s">Home</a>.',
                get_permalink($pages['about-test']),
                get_permalink($pages['services-test']),
                get_permalink($pages['home-test'])
            );
            wp_update_post(['ID' => $pages['contact-test'], 'post_content' => $content]);
        }
    }
    
    // Add cross-links between blog posts
    if (!empty($created_items['posts']) && count($created_items['posts']) > 1) {
        $posts = $created_items['posts'];
        
        // Link first post to second
        if (isset($posts[0]) && isset($posts[1])) {
            $content = get_post_field('post_content', $posts[0]);
            $next_post = get_post($posts[1]);
            $content .= sprintf(' Also read: <a href="%s">%s</a>.', get_permalink($posts[1]), $next_post->post_title);
            wp_update_post(['ID' => $posts[0], 'post_content' => $content]);
        }
        
        // Link second post to third
        if (isset($posts[1]) && isset($posts[2])) {
            $content = get_post_field('post_content', $posts[1]);
            $next_post = get_post($posts[2]);
            $content .= sprintf(' Related: <a href="%s">%s</a>.', get_permalink($posts[2]), $next_post->post_title);
            wp_update_post(['ID' => $posts[1], 'post_content' => $content]);
        }
        
        // Link posts to pages
        if (isset($posts[2]) && isset($created_items['pages']['services-test'])) {
            $content = get_post_field('post_content', $posts[2]);
            $content .= sprintf(' Learn about our <a href="%s">Services</a>.', get_permalink($created_items['pages']['services-test']));
            wp_update_post(['ID' => $posts[2], 'post_content' => $content]);
        }
    }
    
    // Add one broken link for testing
    if (!empty($created_items['pages']['team-test'])) {
        $content = get_post_field('post_content', $created_items['pages']['team-test']);
        $content .= ' View our <a href="https://nonexistent-domain-12345.com/broken-page">awards</a>.';
        wp_update_post(['ID' => $created_items['pages']['team-test'], 'post_content' => $content]);
    }
    
    return rest_ensure_response([
        'success' => true,
        'message' => 'Test content created successfully',
        'data' => [
            'pages_created' => count($created_items['pages']),
            'posts_created' => count($created_items['posts']),
            'orphans_created' => count($created_items['orphans']),
            'total_created' => count($created_items['pages']) + count($created_items['posts']) + count($created_items['orphans']),
        ],
    ]);
}
