<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * Settings REST API Controller
 *
 * Handles settings endpoints for all modules
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use ProRank\SEO\Plugin;
use ProRank\SEO\Core\SettingsManager;

/**
 * Settings controller class
 */
class SettingsController extends BaseController {
    
    /**
     * Rest base
     *
     * @var string
     */
    protected $rest_base = 'settings';
    
    /**
     * Settings manager instance
     *
     * @var SettingsManager|null
     */
    private ?SettingsManager $settings_manager = null;
    
    /**
     * Constructor
     *
     * @param SettingsManager $settings_manager Settings manager instance
     */
    public function __construct(SettingsManager $settings_manager) {
        parent::__construct();
        $this->settings_manager = $settings_manager;
    }

    /**
     * Permission callback for Pro+ settings endpoints that live in this controller.
     *
     * @return bool|WP_Error
     */
    public function check_pro_permission() {
        return $this->check_permission();
    }
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        
        // Site Basics endpoint
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/site_basics',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_site_basics'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_site_basics'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'settings' => [
                            'required' => true,
                            'type'     => 'object',
                        ],
                    ],
                ],
            ]
        );
        
        // Breadcrumbs endpoint
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/breadcrumbs',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_breadcrumbs'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_breadcrumbs'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'data' => [
                            'required' => false,
                            'type'     => 'object',
                        ],
                    ],
                ],
            ]
        );
        
        // Internal Linking endpoints
        register_rest_route(
            'prorank-seo/v1',
            '/settings/internal-linking',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_internal_linking'],
                    'permission_callback' => [$this, 'check_pro_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_internal_linking'],
                    'permission_callback' => [$this, 'check_pro_permission'],
                    'args'                => [
                        'data' => [
                            'required' => false,
                            'type'     => 'object',
                        ],
                    ],
                ],
            ]
        );
        
        // Schema endpoints
        register_rest_route(
            'prorank-seo/v1',
            '/schemas',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_schemas'],
                    'permission_callback' => [$this, 'check_pro_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'create_schema'],
                    'permission_callback' => [$this, 'check_pro_permission'],
                ],
            ]
        );
        
        register_rest_route(
            'prorank-seo/v1',
            '/schemas/(?P<id>\d+)',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_schema'],
                    'permission_callback' => [$this, 'check_pro_permission'],
                ],
                [
                    'methods'             => 'PUT',
                    'callback'            => [$this, 'update_schema'],
                    'permission_callback' => [$this, 'check_pro_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::DELETABLE,
                    'callback'            => [$this, 'delete_schema'],
                    'permission_callback' => [$this, 'check_pro_permission'],
                ],
            ]
        );
        
        register_rest_route(
            'prorank-seo/v1',
            '/schemas/(?P<id>\d+)/toggle',
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'toggle_schema_status'],
                'permission_callback' => [$this, 'check_pro_permission'],
            ]
        );
        
        register_rest_route(
            'prorank-seo/v1',
            '/schema/custom-fields',
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_custom_fields'],
                'permission_callback' => [$this, 'check_pro_permission'],
            ]
        );
        
        // NOTE: Podcast settings endpoint is handled by PodcastModule.php
        // to avoid duplicate REST route registration

        // Titles & Meta endpoint
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/titles-meta',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_titles_meta'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_titles_meta'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'settings' => [
                            'required' => false,
                            'type'     => 'object',
                        ],
                    ],
                ],
            ]
        );
        
        // Content Analysis endpoint
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/content-analysis',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_content_analysis'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_content_analysis'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'data' => [
                            'required' => false,
                            'type'     => 'object',
                        ],
                    ],
                ],
            ]
        );
        
        // Performance Beta toggle endpoint - use a unique path
        register_rest_route(
            $this->namespace,
            '/performance-beta',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_performance_beta'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_performance_beta'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Get all settings (admin only)
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_all_settings'],
                    'permission_callback' => function() {
                        // Allow read access for logged-in users
                        return is_user_logged_in();
                    },
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'save_all_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Get settings for a specific group - register LAST as it's a catch-all
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<group>[a-zA-Z0-9_-]+)',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'group' => [
                            'required'          => true,
                            'sanitize_callback' => 'sanitize_key',
                            'validate_callback' => function($param) {
                                return !empty($param);
                            },
                        ],
                    ],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'group' => [
                            'required'          => true,
                            'sanitize_callback' => 'sanitize_key',
                        ],
                        'settings' => [
                            'required'          => true,
                            'type'              => 'object',
                        ],
                    ],
                ],
            ]
        );
    }
    
    /**
     * Get settings for a specific group
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_settings(WP_REST_Request $request) {
        // Sanitize settings group parameter
        $group = sanitize_key($request->get_param('group'));
        
        if (!$this->settings_manager) {
            return $this->get_error_response(
                __('Settings manager not available', 'prorank-seo'),
                500
            );
        }
        
        try {
            $settings = $this->settings_manager->get_settings($group);
            
            return rest_ensure_response([
                'group' => $group,
                'settings' => $settings,
            ]);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to get settings: %s', 'prorank-seo'), esc_html($e->getMessage())),
                500
            );
        }
    }
    
    /**
     * Update settings for a specific group
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_settings(WP_REST_Request $request) {
        // Sanitize settings group parameter
        $group = sanitize_key($request->get_param('group'));
        $settings = $request->get_param('settings'); // Sanitized by validate_settings()
        
        if (!$this->settings_manager) {
            return $this->get_error_response(
                __('Settings manager not available', 'prorank-seo'),
                500
            );
        }
        
        // Validate settings based on group
        $validated_settings = $this->validate_settings($group, $settings);
        if (is_wp_error($validated_settings)) {
            return $validated_settings;
        }
        
        try {
            // Update settings
            foreach ($validated_settings as $key => $value) {
                // Check if the key already has the group prefix
                if (strpos($key, $group . '_') === 0) {
                    // Key already has prefix, use update() directly
                    $this->settings_manager->update($key, $value);
                } else {
                    // Key doesn't have prefix, use update_setting()
                    $this->settings_manager->update_setting($group, $key, $value);
                }
            }
            
            // Trigger action after settings update
            do_action('prorank_seo_settings_updated', $group, $validated_settings);
            
            return rest_ensure_response([
                'success' => true,
                'message' => __('Settings saved successfully', 'prorank-seo'),
                'group' => $group,
                'settings' => $this->settings_manager->get_settings($group),
            ]);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to save settings: %s', 'prorank-seo'), esc_html($e->getMessage())),
                500
            );
        }
    }
    
    /**
     * Get all settings (for export/debug)
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_all_settings(WP_REST_Request $request) {
        if (!$this->settings_manager) {
            return $this->get_error_response(
                __('Settings manager not available', 'prorank-seo'),
                500
            );
        }
        
        try {
            $all_settings = [];
            
            // Get all registered setting groups
            $groups = $this->get_setting_groups();
            
            foreach ($groups as $group) {
                try {
                    $all_settings[$group] = $this->settings_manager->get_settings($group);
                } catch (\Exception $e) {
                    // If settings don't exist for this group, use empty array
                    $all_settings[$group] = [];
                }
            }
            
            return rest_ensure_response($all_settings);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to get all settings: %s', 'prorank-seo'), esc_html($e->getMessage())),
                500
            );
        }
    }
    
    /**
     * Save all settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function save_all_settings(WP_REST_Request $request) {
        if (!$this->settings_manager) {
            return $this->get_error_response(
                __('Settings manager not available', 'prorank-seo'),
                500
            );
        }
        
        $settings = $request->get_json_params();
        
        if (!is_array($settings)) {
            return $this->get_error_response(
                __('Invalid settings format', 'prorank-seo'),
                400
            );
        }
        
        try {
            foreach ($settings as $group => $group_settings) {
                if (!is_array($group_settings)) {
                    continue;
                }
                
                $validated_settings = $this->validate_settings($group, $group_settings);
                if (is_wp_error($validated_settings)) {
                    return $validated_settings;
                }
                
                foreach ($validated_settings as $key => $value) {
                    $this->settings_manager->update_setting($group, $key, $value);
                }
            }
            
            return rest_ensure_response([
                'success' => true,
                'message' => __('Settings saved successfully', 'prorank-seo'),
            ]);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to save settings: %s', 'prorank-seo'), esc_html($e->getMessage())),
                500
            );
        }
    }
    
    /**
     * Validate settings based on group
     *
     * @param string $group    Settings group
     * @param array  $settings Settings to validate
     * @return array|WP_Error Validated settings or error
     */
    private function validate_settings(string $group, array $settings) {
        $validated = [];
        
        // Group-specific validation
        switch ($group) {
            case 'sitemaps':
                $validated = $this->validate_sitemap_settings($settings);
                break;
                
            case 'redirects':
                $validated = $this->validate_redirect_settings($settings);
                break;
                
            case 'meta':
                $validated = $this->validate_meta_settings($settings);
                break;
                
            case 'social':
                $validated = $this->validate_social_settings($settings);
                break;
                
            case 'cache':
                $validated = $this->validate_cache_settings($settings);
                break;
                
            case 'image_optimization':
                $validated = $this->validate_image_optimization_settings($settings);
                break;
                
            case 'asset_optimization':
                $validated = $this->validate_asset_optimization_settings($settings);
                break;
                
            case 'site_audit':
                $validated = $this->validate_site_audit_settings($settings);
                break;

            case 'psi':
            case 'psi_api':
                $validated = $this->validate_psi_settings($settings);
                break;
                
            case 'clarity':
                $validated = $this->validate_clarity_settings($settings);
                break;
                
            default:
                // For unknown groups, pass through with basic sanitization
                foreach ($settings as $key => $value) {
                    $key = sanitize_key($key);
                    
                    if (is_string($value)) {
                        $validated[$key] = sanitize_text_field($value);
                    } elseif (is_bool($value)) {
                        $validated[$key] = (bool) $value;
                    } elseif (is_numeric($value)) {
                        $validated[$key] = is_float($value) ? (float) $value : (int) $value;
                    } elseif (is_array($value)) {
                        $validated[$key] = array_map('sanitize_text_field', $value);
                    } else {
                        $validated[$key] = $value;
                    }
                }
        }
        
        return $validated;
    }
    
    /**
     * Validate sitemap settings
     *
     * @param array $settings Settings to validate
     * @return array Validated settings
     */
    private function validate_sitemap_settings(array $settings): array {
        $validated = [];
        
        // Boolean settings
        $boolean_fields = [
            'enable_xml_sitemap',
            'exclude_empty_terms',
            'include_images',
            'include_author_archives',
            'exclude_noindex',
            'ping_search_engines',
        ];
        
        foreach ($boolean_fields as $field) {
            if (isset($settings[$field])) {
                $validated[$field] = (bool) $settings[$field];
            }
        }
        
        // Integer settings
        if (isset($settings['posts_per_sitemap'])) {
            $validated['posts_per_sitemap'] = min(50000, max(1, (int) $settings['posts_per_sitemap']));
        }
        
        // Array settings
        if (isset($settings['include_post_types']) && is_array($settings['include_post_types'])) {
            $validated['include_post_types'] = array_map('sanitize_key', $settings['include_post_types']);
        }
        
        if (isset($settings['include_taxonomies']) && is_array($settings['include_taxonomies'])) {
            $validated['include_taxonomies'] = array_map('sanitize_key', $settings['include_taxonomies']);
        }
        
        if (isset($settings['author_roles']) && is_array($settings['author_roles'])) {
            $validated['author_roles'] = array_map('sanitize_key', $settings['author_roles']);
        }
        
        // String settings
        if (isset($settings['sitemap_index_url'])) {
            $url = sanitize_file_name($settings['sitemap_index_url']);
            if (preg_match('/^[a-zA-Z0-9_-]+\.xml$/', $url)) {
                $validated['sitemap_index_url'] = $url;
            }
        }
        
        return $validated;
    }
    
    /**
     * Validate redirect settings
     *
     * @param array $settings Settings to validate
     * @return array Validated settings
     */
    private function validate_redirect_settings(array $settings): array {
        $validated = [];

        // Validate source URL
        if (isset($settings['source']) || isset($settings['source_url'])) {
            $source = $settings['source'] ?? $settings['source_url'] ?? '';
            $validated['source_url'] = sanitize_text_field(wp_unslash($source));
        }

        // Validate target URL
        if (isset($settings['target']) || isset($settings['target_url'])) {
            $target = $settings['target'] ?? $settings['target_url'] ?? '';
            $validated['target_url'] = esc_url_raw(wp_unslash($target));
        }

        // Validate redirect type
        $valid_types = ['301', '302', '307', '308', '410', '451'];
        if (isset($settings['type']) || isset($settings['redirect_type'])) {
            $type = (string) ($settings['type'] ?? $settings['redirect_type'] ?? '301');
            $validated['redirect_type'] = in_array($type, $valid_types, true) ? $type : '301';
        }

        // Validate enabled status
        if (isset($settings['enabled'])) {
            $validated['enabled'] = (bool) $settings['enabled'];
        }

        // Validate case sensitivity
        if (isset($settings['case_insensitive'])) {
            $validated['case_insensitive'] = (bool) $settings['case_insensitive'];
        }

        return $validated;
    }

    /**
     * Validate meta settings
     *
     * @param array $settings Settings to validate
     * @return array Validated settings
     */
    private function validate_meta_settings(array $settings): array {
        $validated = [];

        foreach ($settings as $key => $value) {
            $sanitized_key = sanitize_key($key);

            // Title fields
            if (str_starts_with($sanitized_key, 'title_') || str_ends_with($sanitized_key, '_title')) {
                $validated[$sanitized_key] = sanitize_text_field(wp_unslash($value));
            }
            // Description fields
            elseif (str_starts_with($sanitized_key, 'description_') ||
                    str_starts_with($sanitized_key, 'desc_') ||
                    str_ends_with($sanitized_key, '_description') ||
                    str_ends_with($sanitized_key, '_desc')) {
                $validated[$sanitized_key] = sanitize_textarea_field(wp_unslash($value));
            }
            // Boolean toggle fields
            elseif (str_starts_with($sanitized_key, 'enable_') ||
                    str_starts_with($sanitized_key, 'disable_') ||
                    str_starts_with($sanitized_key, 'show_') ||
                    str_starts_with($sanitized_key, 'hide_') ||
                    str_ends_with($sanitized_key, '_enabled') ||
                    str_ends_with($sanitized_key, '_noindex') ||
                    str_ends_with($sanitized_key, '_nofollow')) {
                $validated[$sanitized_key] = (bool) $value;
            }
            // Separator field
            elseif ($sanitized_key === 'title_separator' || $sanitized_key === 'separator') {
                $allowed_separators = ['-', '–', '—', ':', '·', '•', '*', '|', '~', '/'];
                $separator = sanitize_text_field($value);
                $validated[$sanitized_key] = in_array($separator, $allowed_separators, true) ? $separator : '-';
            }
            // Integer fields
            elseif (str_ends_with($sanitized_key, '_length') || str_ends_with($sanitized_key, '_limit')) {
                $validated[$sanitized_key] = absint($value);
            }
            // Default: sanitize as text
            else {
                if (is_array($value)) {
                    $validated[$sanitized_key] = array_map('sanitize_text_field', $value);
                } elseif (is_bool($value)) {
                    $validated[$sanitized_key] = $value;
                } else {
                    $validated[$sanitized_key] = sanitize_text_field(wp_unslash($value));
                }
            }
        }

        return $validated;
    }

    /**
     * Validate social settings
     *
     * @param array $settings Settings to validate
     * @return array Validated settings
     */
    private function validate_social_settings(array $settings): array {
        $validated = [];

        // URL fields that need URL validation
        $url_fields = [
            'facebook_url',
            'facebook_page_url',
            'twitter_url',
            'linkedin_url',
            'instagram_url',
            'youtube_url',
            'pinterest_url',
            'tiktok_url',
            'default_og_image',
            'default_twitter_image',
        ];

        // Username fields that need specific sanitization
        $username_fields = [
            'twitter_username',
            'facebook_admin_id',
            'facebook_app_id',
        ];

        foreach ($settings as $key => $value) {
            $sanitized_key = sanitize_key($key);

            if (in_array($sanitized_key, $url_fields, true)) {
                $validated[$sanitized_key] = esc_url_raw(wp_unslash($value));
            } elseif (in_array($sanitized_key, $username_fields, true)) {
                // Remove @ from usernames and sanitize
                $username = sanitize_text_field(wp_unslash($value));
                $validated[$sanitized_key] = ltrim($username, '@');
            } elseif (str_starts_with($sanitized_key, 'enable_') || str_ends_with($sanitized_key, '_enabled')) {
                $validated[$sanitized_key] = (bool) $value;
            } elseif ($sanitized_key === 'og_type' || $sanitized_key === 'twitter_card_type') {
                $allowed_og_types = ['website', 'article', 'product', 'profile'];
                $allowed_twitter_types = ['summary', 'summary_large_image', 'app', 'player'];

                if ($sanitized_key === 'og_type') {
                    $validated[$sanitized_key] = in_array($value, $allowed_og_types, true) ? $value : 'website';
                } else {
                    $validated[$sanitized_key] = in_array($value, $allowed_twitter_types, true) ? $value : 'summary_large_image';
                }
            } else {
                $validated[$sanitized_key] = sanitize_text_field(wp_unslash($value));
            }
        }

        return $validated;
    }
    
    /**
     * Get all setting groups
     *
     * @return array List of setting groups
     */
    private function get_setting_groups(): array {
        return [
            'general',
            'sitemaps',
            'redirects',
            'meta',
            'social',
            'schema',
            'performance',
            'cache',
            'tools',
        ];
    }
    
    /**
     * Validate cache settings
     *
     * @param array $settings Settings to validate
     * @return array Validated settings
     */
    private function validate_cache_settings(array $settings): array {
        $validated = [];
        
        // Boolean settings
        $boolean_fields = [
            'cache_enabled',
            'cache_gzip',
            'cache_mobile_separate',
            'preload_enabled',
            'preload_auto_save',
            'js_defer',
            'js_delay',
            'js_combine',
            'css_minify',
            'css_combine',
            'css_async',
        ];
        
        foreach ($boolean_fields as $field) {
            if (isset($settings[$field])) {
                $validated[$field] = (bool) $settings[$field];
            }
        }
        
        // Integer settings
        if (isset($settings['cache_lifetime'])) {
            $validated['cache_lifetime'] = min(86400, max(60, (int) $settings['cache_lifetime']));
        }
        
        // String settings
        $string_fields = [
            'cache_exclude_urls',
            'cache_exclude_cookies',
            'cache_exclude_agents',
            'js_exclude',
            'css_exclude',
        ];
        
        foreach ($string_fields as $field) {
            if (isset($settings[$field])) {
                $validated[$field] = sanitize_textarea_field($settings[$field]);
            }
        }
        
        // Select fields
        if (isset($settings['preload_mode'])) {
            $allowed = ['sitemap', 'recent'];
            $validated['preload_mode'] = in_array($settings['preload_mode'], $allowed) 
                ? $settings['preload_mode'] 
                : 'sitemap';
        }
        
        if (isset($settings['preload_schedule'])) {
            $allowed = ['hourly', 'daily', 'weekly'];
            $validated['preload_schedule'] = in_array($settings['preload_schedule'], $allowed) 
                ? $settings['preload_schedule'] 
                : 'daily';
        }
        
        return $validated;
    }
    
    
    /**
     * Validate image optimization settings
     *
     * @param array $settings Settings to validate
     * @return array Validated settings
     */
    private function validate_image_optimization_settings(array $settings): array {
        $validated = [];
        
        // Boolean settings
        $boolean_fields = [
            'lazyload_images',
            'lazyload_iframes',
            'lazyload_videos',
            'lazyload_background_images',
            'image_opt_webp_enabled',
            'image_opt_avif_enabled',
            'image_opt_delivery_enabled',
        ];
        
        foreach ($boolean_fields as $field) {
            if (isset($settings[$field])) {
                $validated[$field] = (bool) $settings[$field];
            }
        }
        
        // Integer settings
        if (isset($settings['image_opt_quality_webp'])) {
            $validated['image_opt_quality_webp'] = min(100, max(1, (int) $settings['image_opt_quality_webp']));
        }
        
        if (isset($settings['image_opt_quality_avif'])) {
            $validated['image_opt_quality_avif'] = min(100, max(1, (int) $settings['image_opt_quality_avif']));
        }
        
        // String settings
        if (isset($settings['lazyload_exclude'])) {
            $validated['lazyload_exclude'] = sanitize_textarea_field($settings['lazyload_exclude']);
        }
        
        return $validated;
    }
    
    /**
     * Validate asset optimization settings
     *
     * @param array $settings Settings to validate
     * @return array Validated settings
     */
    private function validate_asset_optimization_settings(array $settings): array {
        $validated = [];
        
        // Boolean settings
        $boolean_fields = [
            'critical_css_enabled',
            'critical_css_async',
            'critical_css_preload',
            'unused_css_enabled',
            'unused_css_collect',
            'unused_css_minify',
            'js_advanced_enabled',
            'font_opt_google_local',
            'font_opt_display_swap',
            'font_opt_subset',
        ];
        
        foreach ($boolean_fields as $field) {
            if (isset($settings[$field])) {
                $validated[$field] = (bool) $settings[$field];
            }
        }
        
        // String settings
        $string_fields = [
            'critical_css_mode',
            'critical_css_manual',
            'critical_css_exclude',
            'unused_css_exclude',
            'unused_css_exclude_files',
        ];
        
        foreach ($string_fields as $field) {
            if (isset($settings[$field])) {
                $validated[$field] = sanitize_textarea_field($settings[$field]);
            }
        }
        
        // Select fields
        if (isset($settings['critical_css_mode'])) {
            $allowed = ['manual', 'automatic'];
            $validated['critical_css_mode'] = in_array($settings['critical_css_mode'], $allowed) 
                ? $settings['critical_css_mode'] 
                : 'manual';
        }
        
        // Array fields
        if (isset($settings['js_advanced_rules']) && is_array($settings['js_advanced_rules'])) {
            $validated['js_advanced_rules'] = array_map(function($rule) {
                return [
                    'handle' => sanitize_key($rule['handle'] ?? ''),
                    'action' => in_array($rule['action'] ?? '', ['defer', 'delay', 'remove']) ? $rule['action'] : 'defer',
                ];
            }, $settings['js_advanced_rules']);
        }
        
        return $validated;
    }
    
    /**
     * Get performance beta setting
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_performance_beta(WP_REST_Request $request) {
        if (!$this->settings_manager) {
            return new \WP_Error(
                'no_settings_manager',
                __('Settings manager not available', 'prorank-seo'),
                ['status' => 500]
            );
        }
        
        try {
            // Get performance beta setting from general settings
            $value = $this->settings_manager->get('general', 'performance_beta', false);
            
            return rest_ensure_response([
                'value' => (bool) $value,
            ]);
        } catch (\Exception $e) {
            return new \WP_Error(
                'settings_error',
                esc_html($e->getMessage()),
                ['status' => 500]
            );
        }
    }
    
    /**
     * Update performance beta setting
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_performance_beta(WP_REST_Request $request) {
        if (!$this->settings_manager) {
            return new \WP_Error(
                'no_settings_manager',
                __('Settings manager not available', 'prorank-seo'),
                ['status' => 500]
            );
        }
        
        // Get value from request body or params
        $body = $request->get_json_params();
        $value = isset($body['value']) ? $body['value'] : $request->get_param('value');
        
        // Convert to boolean
        $bool_value = filter_var($value, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        if ($bool_value === null) {
            return new \WP_Error(
                'invalid_value',
                __('Invalid boolean value', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        try {
            // Update performance beta setting in general settings
            $this->settings_manager->update_setting('general', 'performance_beta', $bool_value);
            
            // Trigger action after update
            do_action('prorank_seo_performance_beta_updated', $bool_value);
            
            return rest_ensure_response([
                'success' => true,
                'message' => $bool_value 
                    ? __('Performance Beta enabled', 'prorank-seo')
                    : __('Performance Beta disabled', 'prorank-seo'),
                'value' => $bool_value,
            ]);
            
        } catch (\Exception $e) {
            return new \WP_Error(
                'update_error',
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to update performance beta: %s', 'prorank-seo'), esc_html($e->getMessage())),
                ['status' => 500]
            );
        }
    }
    
    /**
     * Validate site audit settings
     *
     * @param array $settings Settings to validate
     * @return array Validated settings
     */
    private function validate_site_audit_settings(array $settings): array {
        $validated = [];
        
        // Boolean settings
        $boolean_fields = [
            'site_audit_schedule_enabled',
            'site_audit_auto_fix_enabled',
            'site_audit_auto_fix_require_approval',
            'white_label_remove_branding',
            'site_audit_email_reports',
        ];
        
        foreach ($boolean_fields as $field) {
            if (isset($settings[$field])) {
                $validated[$field] = (bool) $settings[$field];
            }
        }
        
        // String settings
        $string_fields = [
            'site_audit_schedule_frequency',
            'white_label_logo',
            'white_label_company_name',
            'white_label_primary_color',
            'white_label_secondary_color',
            'google_pagespeed_api_key',
            'site_audit_email_recipients',
        ];
        
        foreach ($string_fields as $field) {
            if (isset($settings[$field])) {
                $validated[$field] = sanitize_text_field($settings[$field]);
            }
        }
        
        // Integer settings
        if (isset($settings['site_audit_max_urls'])) {
            $validated['site_audit_max_urls'] = min(5000, max(50, (int) $settings['site_audit_max_urls']));
        }
        
        if (isset($settings['site_audit_max_depth'])) {
            $validated['site_audit_max_depth'] = min(10, max(1, (int) $settings['site_audit_max_depth']));
        }
        
        // Array settings
        if (isset($settings['site_audit_check_types']) && is_array($settings['site_audit_check_types'])) {
            $validated['site_audit_check_types'] = array_map('sanitize_key', $settings['site_audit_check_types']);
        }
        
        if (isset($settings['site_audit_auto_fix_types']) && is_array($settings['site_audit_auto_fix_types'])) {
            $validated['site_audit_auto_fix_types'] = array_map('sanitize_key', $settings['site_audit_auto_fix_types']);
        }
        
        return $validated;
    }

    /**
     * Validate PSI settings
     *
     * @param array $settings Settings to validate
     * @return array Validated settings
     */
    private function validate_psi_settings(array $settings): array {
        $validated = [];
        
        // API key
        if (isset($settings['api_key'])) {
            $api_key = sanitize_text_field($settings['api_key']);
            // Basic validation for Google API key format
            if (empty($api_key) || preg_match('/^AIza[0-9A-Za-z_-]{35}$/', $api_key)) {
                $validated['api_key'] = $api_key;
            }
        }
        
        // Auto-track setting
        if (isset($settings['auto_track'])) {
            $validated['auto_track'] = (bool) $settings['auto_track'];
        }
        
        // Track frequency
        if (isset($settings['track_frequency'])) {
            $validated['track_frequency'] = in_array($settings['track_frequency'], ['daily', 'weekly', 'monthly']) 
                ? $settings['track_frequency'] 
                : 'weekly';
        }
        
        return $validated;
    }
    
    /**
     * Validate Clarity settings
     *
     * @param array $settings Settings to validate
     * @return array Validated settings
     */
    private function validate_clarity_settings(array $settings): array {
        $validated = [];
        
        // Enabled flag
        if (isset($settings['enabled'])) {
            $validated['enabled'] = (bool) $settings['enabled'];
        }
        
        // Project ID
        if (isset($settings['project_id'])) {
            $project_id = sanitize_text_field($settings['project_id']);
            // Basic validation for Clarity project ID format
            if (preg_match('/^[a-z0-9]{8,12}$/', $project_id) || empty($project_id)) {
                $validated['project_id'] = $project_id;
            }
        }
        
        return $validated;
    }
    
    /**
     * Get site basics settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_site_basics(WP_REST_Request $request) {
        $settings = get_option('prorank_seo_site_basics', []);
        
        return rest_ensure_response([
            'success' => true,
            'settings' => $settings,
        ]);
    }
    
    /**
     * Update site basics settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_site_basics(WP_REST_Request $request) {
        $settings = $request->get_param('settings');
        
        if (!is_array($settings)) {
            return $this->get_error_response(
                __('Invalid settings format', 'prorank-seo'),
                400
            );
        }
        
        // Sanitize settings
        $sanitized = [];
        $text_fields = [
            'site_name', 'site_alternate_name', 'site_tagline', 'title_separator',
            'site_logo', 'site_image', 'person_logo', 'google_verification', 'bing_verification',
            'baidu_verification', 'yandex_verification', 'pinterest_verification',
            'norton_verification', 'facebook_page_url', 'facebook_authorship',
            'facebook_admin_id', 'facebook_app_id', 'facebook_secret',
            'twitter_username', 'linkedin_url', 'instagram_url', 'youtube_url',
            'pinterest_url', 'org_or_person', 'org_name', 'org_alternate_name',
            'org_logo', 'org_description', 'org_email', 'org_phone',
            'org_legal_name', 'org_founding_date', 'org_num_employees',
            'org_vat_id', 'org_tax_id', 'org_iso_6523', 'org_duns',
            'org_lei_code', 'org_naics', 'org_street_address', 'org_address_locality',
            'org_address_region', 'org_postal_code', 'org_address_country',
        ];
        
        foreach ($text_fields as $field) {
            if (isset($settings[$field])) {
                $sanitized[$field] = sanitize_text_field($settings[$field]);
            }
        }

        $int_fields = [
            'site_logo_id',
            'site_image_id',
            'org_logo_id',
            'person_logo_id',
            'person_user_id',
            'privacy_policy_page',
            'terms_of_service_page',
            'publishing_principles_page',
            'ownership_funding_page',
            'actionable_feedback_page',
            'corrections_policy_page',
            'ethics_policy_page',
            'diversity_policy_page',
            'diversity_staffing_page',
        ];

        foreach ($int_fields as $field) {
            if (isset($settings[$field])) {
                $sanitized[$field] = absint($settings[$field]);
            }
        }

        if (isset($settings['org_local_business_enabled'])) {
            $sanitized['org_local_business_enabled'] = !empty($settings['org_local_business_enabled']);
        }
        
        // Handle textarea field
        if (isset($settings['custom_meta_tags'])) {
            $sanitized['custom_meta_tags'] = wp_kses($settings['custom_meta_tags'], [
                'meta' => [
                    'name' => [],
                    'content' => [],
                    'property' => [],
                ]
            ]);
        }
        
        // Handle array fields
        if (isset($settings['additional_profiles']) && is_array($settings['additional_profiles'])) {
            $sanitized['additional_profiles'] = array_map(function($profile) {
                return [
                    'platform' => sanitize_text_field($profile['platform'] ?? ''),
                    'url' => esc_url_raw($profile['url'] ?? '')
                ];
            }, $settings['additional_profiles']);
        }
        
        update_option('prorank_seo_site_basics', $sanitized);

        // Keep WordPress core site name/tagline in sync.
        if (!empty($sanitized['site_name'])) {
            update_option('blogname', $sanitized['site_name']);
        }
        if (!empty($sanitized['site_tagline'])) {
            update_option('blogdescription', $sanitized['site_tagline']);
        }

        wp_cache_delete('prorank_site_basics_settings', 'prorank_seo');
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('Settings saved successfully', 'prorank-seo'),
            'settings' => $sanitized,
        ]);
    }
    
    /**
     * Get breadcrumbs settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_breadcrumbs(WP_REST_Request $request) {
        $defaults = [
            'enabled' => true,
            'separator' => '›',
            'home_text' => 'Home',
            'show_current' => true,
            'show_home' => true,
            'auto_insert' => false,
            'position' => 'before',
            'enable_schema' => true,
            'schema_type' => 'both',
            'style' => 'default',
            'display_style' => 'inline',
            'show_on_home' => false,
            'show_on_posts' => true,
            'show_on_pages' => true,
            'show_on_archives' => true,
            'show_on_404' => false,
            'custom_css' => '',
        ];
        
        $settings = get_option('prorank_seo_breadcrumbs', $defaults);
        $settings = wp_parse_args($settings, $defaults);
        
        return rest_ensure_response([
            'success' => true,
            'data' => $settings,
        ]);
    }
    
    /**
     * Update breadcrumbs settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_breadcrumbs(WP_REST_Request $request) {
        $data = $request->get_param('data');
        
        if (!is_array($data)) {
            $data = $request->get_json_params();
        }
        
        if (!is_array($data)) {
            return $this->get_error_response(
                __('Invalid data format', 'prorank-seo'),
                400
            );
        }
        
        // Sanitize settings
        $sanitized = [];
        
        // Boolean fields
        $bool_fields = [
            'enabled', 'show_current', 'show_home', 'auto_insert',
            'enable_schema', 'show_on_home', 'show_on_posts',
            'show_on_pages', 'show_on_archives', 'show_on_404'
        ];
        
        foreach ($bool_fields as $field) {
            if (isset($data[$field])) {
                $sanitized[$field] = (bool) $data[$field];
            }
        }
        
        // Text fields
        $text_fields = [
            'separator', 'home_text', 'position', 'schema_type',
            'style', 'display_style', 'custom_css'
        ];
        
        foreach ($text_fields as $field) {
            if (isset($data[$field])) {
                $sanitized[$field] = sanitize_text_field($data[$field]);
            }
        }
        
        update_option('prorank_seo_breadcrumbs', $sanitized);
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('Breadcrumbs settings saved successfully', 'prorank-seo'),
        ]);
    }
    
    /**
     * Get titles & meta settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_titles_meta(WP_REST_Request $request) {
        $settings = get_option('prorank_seo_titles_meta', []);
        
        return rest_ensure_response($settings);
    }
    
    /**
     * Update titles & meta settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_titles_meta(WP_REST_Request $request) {
        $settings = $request->get_json_params();
        
        if (!is_array($settings)) {
            $settings = [];
        }

        $existing = get_option('prorank_seo_titles_meta', []);
        $old_author_base = is_array($existing) ? ($existing['author_base'] ?? '') : '';
        $new_author_base = $settings['author_base'] ?? '';

        if ($new_author_base !== '') {
            $settings['author_base'] = sanitize_title((string) $new_author_base);
        }

        update_option('prorank_seo_titles_meta', $settings);

        if ($old_author_base !== ($settings['author_base'] ?? '')) {
            flush_rewrite_rules(false);
        }
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('Settings saved successfully', 'prorank-seo'),
        ]);
    }
    
    /**
     * Get content analysis settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_content_analysis(WP_REST_Request $request) {
        $settings = get_option('prorank_seo_content_analysis', [
            'enable_realtime' => true,
            'enable_ai_insights' => true,
            'min_content_length' => 300,
            'target_readability_score' => 60,
            'auto_analyze_delay' => 2000,
            'show_keyword_density' => true,
            'show_entity_detection' => true,
            'show_intent_classification' => true,
        ]);
        
        return rest_ensure_response([
            'success' => true,
            'data' => $settings,
        ]);
    }
    
    /**
     * Update content analysis settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_content_analysis(WP_REST_Request $request) {
        $settings = $request->get_json_params();
        
        if (!is_array($settings)) {
            return $this->get_error_response(
                __('Invalid settings format', 'prorank-seo'),
                400
            );
        }
        
        // Sanitize settings
        $sanitized = [];
        
        // Boolean fields
        $bool_fields = [
            'enable_realtime', 'enable_ai_insights', 'show_keyword_density',
            'show_entity_detection', 'show_intent_classification'
        ];
        
        foreach ($bool_fields as $field) {
            if (isset($settings[$field])) {
                $sanitized[$field] = (bool) $settings[$field];
            }
        }
        
        // Integer fields
        if (isset($settings['min_content_length'])) {
            $sanitized['min_content_length'] = absint($settings['min_content_length']);
        }
        
        if (isset($settings['target_readability_score'])) {
            $sanitized['target_readability_score'] = absint($settings['target_readability_score']);
        }
        
        if (isset($settings['auto_analyze_delay'])) {
            $sanitized['auto_analyze_delay'] = absint($settings['auto_analyze_delay']);
        }
        
        update_option('prorank_seo_content_analysis', $sanitized);
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('Settings saved successfully', 'prorank-seo'),
            'data' => $sanitized,
        ]);
    }
    
    /**
     * Get podcast settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_podcast_settings(WP_REST_Request $request) {
        $defaults = [
            'podcast_title' => '',
            'podcast_subtitle' => '',
            'podcast_author' => '',
            'podcast_email' => '',
            'podcast_category' => '',
            'podcast_subcategory' => '',
            'podcast_explicit' => 'no',
            'podcast_language' => '',
            'podcast_copyright' => '',
            'podcast_image' => '',
            'podcast_owner_name' => '',
            'podcast_owner_email' => '',
            'enable_podcast_feed' => false,
            'podcast_feed_limit' => '',
            'enable_itunes_tags' => true,
            'enable_spotify_tags' => false,
            'enable_youtube_tags' => false,
            'podcast_redirect_feed' => '',
            'enable_transcripts' => false,
            'enable_chapters' => false,
            'enable_value4value' => false,
            'enable_video_podcasts' => false,
            'lightning_address' => '',
            'funding_url' => '',
        ];
        
        $settings = get_option('prorank_seo_podcast', $defaults);
        $settings = wp_parse_args($settings, $defaults);
        
        return rest_ensure_response([
            'success' => true,
            'data' => $settings,
        ]);
    }
    
    /**
     * Update podcast settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_podcast_settings(WP_REST_Request $request) {
        $settings = $request->get_param('settings');
        
        if (!is_array($settings)) {
            return new \WP_Error(
                'invalid_settings',
                __('Invalid settings data', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        $sanitized = [];
        
        // Boolean fields
        $bool_fields = [
            'enable_podcast_feed', 'enable_itunes_tags', 'enable_spotify_tags',
            'enable_youtube_tags', 'enable_transcripts', 'enable_chapters',
            'enable_value4value', 'enable_video_podcasts'
        ];
        
        foreach ($bool_fields as $field) {
            if (isset($settings[$field])) {
                $sanitized[$field] = (bool) $settings[$field];
            }
        }
        
        // Text fields
        $text_fields = [
            'podcast_title', 'podcast_author', 'podcast_category',
            'podcast_subcategory', 'podcast_explicit', 'podcast_language',
            'podcast_copyright', 'podcast_owner_name'
        ];
        
        foreach ($text_fields as $field) {
            if (isset($settings[$field])) {
                $sanitized[$field] = sanitize_text_field($settings[$field]);
            }
        }
        
        // Textarea field
        if (isset($settings['podcast_subtitle'])) {
            $sanitized['podcast_subtitle'] = sanitize_textarea_field($settings['podcast_subtitle']);
        }
        
        // Email fields
        $email_fields = ['podcast_email', 'podcast_owner_email'];
        foreach ($email_fields as $field) {
            if (isset($settings[$field])) {
                $sanitized[$field] = sanitize_email($settings[$field]);
            }
        }
        
        // URL fields
        $url_fields = ['podcast_image', 'podcast_redirect_feed', 'funding_url'];
        foreach ($url_fields as $field) {
            if (isset($settings[$field])) {
                $sanitized[$field] = esc_url_raw($settings[$field]);
            }
        }
        
        // Special fields
        if (isset($settings['podcast_feed_limit'])) {
            $sanitized['podcast_feed_limit'] = absint($settings['podcast_feed_limit']);
        }
        
        if (isset($settings['lightning_address'])) {
            $sanitized['lightning_address'] = sanitize_text_field($settings['lightning_address']);
        }
        
        update_option('prorank_seo_podcast', $sanitized);
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('Podcast settings saved successfully', 'prorank-seo'),
            'data' => $sanitized,
        ]);
    }
    
    /**
     * Get all schemas
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_schemas(WP_REST_Request $request) {
        $schemas = get_option('prorank_seo_schemas', []);
        
        return rest_ensure_response([
            'schemas' => $schemas,
        ]);
    }
    
    /**
     * Get single schema
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_schema(WP_REST_Request $request) {
        $id = $request->get_param('id');
        $schemas = get_option('prorank_seo_schemas', []);
        
        $schema = null;
        foreach ($schemas as $s) {
            if ($s['id'] == $id) {
                $schema = $s;
                break;
            }
        }
        
        if (!$schema) {
            return new \WP_Error(
                'schema_not_found',
                __('Schema not found', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        return rest_ensure_response($schema);
    }
    
    /**
     * Create new schema
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function create_schema(WP_REST_Request $request) {
        $data = $request->get_json_params();
        
        if (empty($data['type']) || empty($data['name'])) {
            return new \WP_Error(
                'invalid_schema',
                __('Schema type and name are required', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        $schemas = get_option('prorank_seo_schemas', []);
        
        // Generate unique ID
        $id = time() . '_' . wp_rand(1000, 9999);
        
        $new_schema = [
            'id' => $id,
            'name' => sanitize_text_field($data['name']),
            'type' => sanitize_text_field($data['type']),
            'targeting' => $data['targeting'] ?? [],
            'mapping' => $data['mapping'] ?? [],
            'status' => $data['status'] ?? 'active',
            'priority' => intval($data['priority'] ?? 10),
            'created' => current_time('mysql'),
            'modified' => current_time('mysql'),
        ];
        
        $schemas[] = $new_schema;
        update_option('prorank_seo_schemas', $schemas);
        
        return rest_ensure_response([
            'success' => true,
            'schema' => $new_schema,
        ]);
    }
    
    /**
     * Update existing schema
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_schema(WP_REST_Request $request) {
        $id = $request->get_param('id');
        $data = $request->get_json_params();
        
        $schemas = get_option('prorank_seo_schemas', []);
        
        $found = false;
        foreach ($schemas as &$schema) {
            if ($schema['id'] == $id) {
                $schema['name'] = sanitize_text_field($data['name'] ?? $schema['name']);
                $schema['type'] = sanitize_text_field($data['type'] ?? $schema['type']);
                $schema['targeting'] = $data['targeting'] ?? $schema['targeting'];
                $schema['mapping'] = $data['mapping'] ?? $schema['mapping'];
                $schema['status'] = $data['status'] ?? $schema['status'];
                $schema['priority'] = intval($data['priority'] ?? $schema['priority']);
                $schema['modified'] = current_time('mysql');
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            return new \WP_Error(
                'schema_not_found',
                __('Schema not found', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        update_option('prorank_seo_schemas', $schemas);
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('Schema updated successfully', 'prorank-seo'),
        ]);
    }
    
    /**
     * Delete schema
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function delete_schema(WP_REST_Request $request) {
        $id = $request->get_param('id');
        
        $schemas = get_option('prorank_seo_schemas', []);
        $filtered = [];
        $found = false;
        
        foreach ($schemas as $schema) {
            if ($schema['id'] != $id) {
                $filtered[] = $schema;
            } else {
                $found = true;
            }
        }
        
        if (!$found) {
            return new \WP_Error(
                'schema_not_found',
                __('Schema not found', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        update_option('prorank_seo_schemas', $filtered);
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('Schema deleted successfully', 'prorank-seo'),
        ]);
    }
    
    /**
     * Get Internal Linking settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_internal_linking(WP_REST_Request $request) {
        $settings = get_option('prorank_seo_internal_linking', [
            'enabled' => true,
            'auto_linking' => false,
            'max_suggestions' => 10,
            'max_links_per_post' => 5,
            'min_word_count' => 100,
            'link_post_types' => ['post', 'page'],
            'exclude_categories' => [],
            'ignore_words' => '',
            'open_in_new_tab' => false,
            'open_internal_new_tab' => false,
            'open_external_new_tab' => true,
            'add_nofollow' => false,
            'add_title_attribute' => true,
            'track_clicks' => false,
            'ai_suggestions' => false,
            'suggestion_algorithm' => 'balanced',
            'anchor_text_length' => 4,
            'skip_sentences_with_links' => true,
            'randomize_suggestions' => false,
        ]);
        
        return rest_ensure_response([
            'success' => true,
            'data' => $settings,
        ]);
    }
    
    /**
     * Update Internal Linking settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_internal_linking(WP_REST_Request $request) {
        $data = $request->get_json_params();
        
        if (empty($data)) {
            $data = $request->get_params();
        }
        
        // Get existing settings
        $settings = get_option('prorank_seo_internal_linking', []);
        
        // Update settings with new data
        $updated_settings = array_merge($settings, [
            'enabled' => isset($data['enabled']) ? (bool) $data['enabled'] : true,
            'auto_linking' => isset($data['auto_linking']) ? (bool) $data['auto_linking'] : false,
            'max_suggestions' => isset($data['max_suggestions']) ? intval($data['max_suggestions']) : 10,
            'max_links_per_post' => isset($data['max_links_per_post']) ? intval($data['max_links_per_post']) : 5,
            'min_word_count' => isset($data['min_word_count']) ? intval($data['min_word_count']) : 100,
            'link_post_types' => isset($data['link_post_types']) ? (array) $data['link_post_types'] : ['post', 'page'],
            'exclude_categories' => isset($data['exclude_categories']) ? (array) $data['exclude_categories'] : [],
            'ignore_words' => isset($data['ignore_words']) ? sanitize_textarea_field($data['ignore_words']) : '',
            'open_in_new_tab' => isset($data['open_in_new_tab']) ? (bool) $data['open_in_new_tab'] : false,
            'open_internal_new_tab' => isset($data['open_internal_new_tab']) ? (bool) $data['open_internal_new_tab'] : false,
            'open_external_new_tab' => isset($data['open_external_new_tab']) ? (bool) $data['open_external_new_tab'] : false,
            'add_nofollow' => isset($data['add_nofollow']) ? (bool) $data['add_nofollow'] : false,
            'add_title_attribute' => isset($data['add_title_attribute']) ? (bool) $data['add_title_attribute'] : false,
            'track_clicks' => isset($data['track_clicks']) ? (bool) $data['track_clicks'] : false,
            'ai_suggestions' => isset($data['ai_suggestions']) ? (bool) $data['ai_suggestions'] : false,
            'suggestion_algorithm' => isset($data['suggestion_algorithm']) ? sanitize_text_field($data['suggestion_algorithm']) : 'balanced',
            'anchor_text_length' => isset($data['anchor_text_length']) ? intval($data['anchor_text_length']) : 4,
            'skip_sentences_with_links' => isset($data['skip_sentences_with_links']) ? (bool) $data['skip_sentences_with_links'] : true,
            'randomize_suggestions' => isset($data['randomize_suggestions']) ? (bool) $data['randomize_suggestions'] : false,
            'post_types' => isset($data['post_types']) ? (array) $data['post_types'] : ['post', 'page'],
            'post_statuses' => isset($data['post_statuses']) ? sanitize_text_field($data['post_statuses']) : 'publish',
            'skip_sentences' => isset($data['skip_sentences']) ? intval($data['skip_sentences']) : 3,
            'max_post_age' => isset($data['max_post_age']) ? sanitize_text_field($data['max_post_age']) : 'no_limit',
            'max_outbound_links' => isset($data['max_outbound_links']) ? intval($data['max_outbound_links']) : 0,
            'max_inbound_links' => isset($data['max_inbound_links']) ? intval($data['max_inbound_links']) : 0,
            'keyword_sources' => isset($data['keyword_sources']) ? (array) $data['keyword_sources'] : ['title', 'slug'],
            'ignore_numbers' => isset($data['ignore_numbers']) ? (bool) $data['ignore_numbers'] : false,
            'language' => isset($data['language']) ? sanitize_text_field($data['language']) : 'en',
            'ignored_words' => isset($data['ignored_words']) ? sanitize_textarea_field($data['ignored_words']) : '',
        ]);
        
        // Save settings
        update_option('prorank_seo_internal_linking', $updated_settings);
        
        // Clear any cached data
        wp_cache_delete('prorank_internal_linking_cache');
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('Internal Linking settings updated successfully', 'prorank-seo'),
            'data' => $updated_settings,
        ]);
    }
    
    /**
     * Toggle schema status
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function toggle_schema_status(WP_REST_Request $request) {
        $id = $request->get_param('id');
        $data = $request->get_json_params();
        $new_status = $data['status'] ?? null;
        
        if (!in_array($new_status, ['active', 'inactive'])) {
            return new \WP_Error(
                'invalid_status',
                __('Invalid status value', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        $schemas = get_option('prorank_seo_schemas', []);
        
        $found = false;
        foreach ($schemas as &$schema) {
            if ($schema['id'] == $id) {
                $schema['status'] = $new_status;
                $schema['modified'] = current_time('mysql');
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            return new \WP_Error(
                'schema_not_found',
                __('Schema not found', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        update_option('prorank_seo_schemas', $schemas);
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('Schema status updated', 'prorank-seo'),
        ]);
    }
    
    /**
     * Get custom fields
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_custom_fields(WP_REST_Request $request) {
        global $wpdb;
        
        // Get ACF fields if available
        $acf_fields = [];
        if (function_exists('acf_get_field_groups')) {
            $field_groups = acf_get_field_groups();
            foreach ($field_groups as $group) {
                $fields = acf_get_fields($group['key']);
                foreach ($fields as $field) {
                    $acf_fields[] = [
                        'label' => $field['label'],
                        'value' => 'acf_' . $field['name'],
                        'type' => $field['type'],
                    ];
                }
            }
        }
        
        // Get post meta keys
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $meta_keys = $wpdb->get_col("
            SELECT DISTINCT meta_key 
            FROM {$wpdb->postmeta} 
            WHERE meta_key NOT LIKE '\_%' 
            LIMIT 100
        ");
        
        $custom_fields = [];
        foreach ($meta_keys as $key) {
            $custom_fields[] = [
                'label' => ucwords(str_replace('_', ' ', $key)),
                'value' => 'meta_' . $key,
                'type' => 'text',
            ];
        }
        
        // Merge ACF and custom fields
        $all_fields = array_merge($acf_fields, $custom_fields);
        
        return rest_ensure_response($all_fields);
    }
}
