<?php
declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;

class SchemaPreviewController extends BaseController {
    protected $rest_base = 'schema/preview';

    public function register_routes(): void {
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<post_id>\\d+)',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [ $this, 'get_schema_preview' ],
                    'permission_callback' => [ $this, 'check_preview_permissions' ],
                    'args'                => [
                        'post_id'     => [
                            'required'          => true,
                            'validate_callback' => static function ( $param ): bool {
                                return is_numeric( $param ) && get_post( (int) $param ) !== null;
                            },
                        ],
                        'schema_data' => [
                            'required' => false,
                            'type'     => 'object',
                        ],
                    ],
                ],
            ]
        );
    }

    public function check_preview_permissions( WP_REST_Request $request ): bool {
        $post_id = (int) $request->get_param( 'post_id' );
        return $post_id > 0 && current_user_can( 'edit_post', $post_id );
    }

    public function get_schema_preview( WP_REST_Request $request ) {
        $post_id = (int) $request->get_param( 'post_id' );
        $post    = get_post( $post_id );

        if ( ! $post ) {
            return $this->error( __( 'Post not found.', 'prorank-seo' ), 'schema_not_found', 404 );
        }

        $schema_data = $request->get_param( 'schema_data' );
        if ( ! is_array( $schema_data ) ) {
            $schema_data = get_post_meta( $post_id, '_prorank_schema_data', true );
        }

        if ( ! is_array( $schema_data ) ) {
            $schema_data = [];
        }

        $custom_json = '';
        if ( ! empty( $schema_data['custom_json'] ) && is_string( $schema_data['custom_json'] ) ) {
            $custom_json = $schema_data['custom_json'];
        }

        if ( $custom_json !== '' ) {
            $decoded = json_decode( $custom_json, true );
            if ( json_last_error() !== JSON_ERROR_NONE || ! is_array( $decoded ) ) {
                return $this->error( __( 'Invalid JSON in custom schema.', 'prorank-seo' ), 'invalid_json', 400 );
            }

            return $this->success( [
                'preview' => wp_json_encode( $decoded ),
            ] );
        }

        $schema_type = $schema_data['type'] ?? 'Article';
        $schema_type = is_string( $schema_type ) ? sanitize_text_field( $schema_type ) : 'Article';
        if ( $schema_type === '' ) {
            $schema_type = 'Article';
        }

        if ( $schema_type === 'Custom' ) {
            return $this->success( [
                'preview' => '// No schema generated for this post',
            ] );
        }

        $properties = [];
        if ( ! empty( $schema_data['properties'] ) && is_array( $schema_data['properties'] ) ) {
            $properties = $schema_data['properties'];
        }

        $schema = $this->build_schema_preview( $post, $schema_type, $properties );
        if ( empty( $schema ) ) {
            return $this->success( [
                'preview' => '// No schema generated for this post',
            ] );
        }

        return $this->success( [
            'preview' => wp_json_encode( $schema ),
        ] );
    }

    private function build_schema_preview( \WP_Post $post, string $schema_type, array $properties ): array {
        $article_types = [ 'Article', 'BlogPosting', 'NewsArticle' ];
        $schema = [];

        if ( in_array( $schema_type, $article_types, true ) ) {
            $schema = [
                '@context'       => 'https://schema.org',
                '@type'          => $schema_type,
                'headline'       => $post->post_title,
                'name'           => $post->post_title,
                'url'            => get_permalink( $post ),
                'datePublished'  => get_the_date( 'c', $post ),
                'dateModified'   => get_the_modified_date( 'c', $post ),
                'mainEntityOfPage' => [
                    '@type' => 'WebPage',
                    '@id'   => get_permalink( $post ),
                ],
            ];

            if ( ! empty( $post->post_excerpt ) ) {
                $schema['description'] = wp_strip_all_tags( $post->post_excerpt );
            }

            if ( has_post_thumbnail( $post ) ) {
                $image_id  = get_post_thumbnail_id( $post );
                $image_url = wp_get_attachment_image_url( $image_id, 'full' );
                if ( $image_url ) {
                    $schema['image'] = [
                        '@type' => 'ImageObject',
                        'url'   => $image_url,
                    ];
                }
            }
        } else {
            $schema = [
                '@context' => 'https://schema.org',
                '@type'    => $schema_type,
                'name'     => $post->post_title,
                'url'      => get_permalink( $post ),
            ];

            if ( ! empty( $post->post_excerpt ) ) {
                $schema['description'] = wp_strip_all_tags( $post->post_excerpt );
            }
        }

        if ( ! empty( $properties ) ) {
            foreach ( $properties as $key => $value ) {
                if ( $value === null || $value === '' ) {
                    continue;
                }
                $schema[ $key ] = $value;
            }
        }

        return $schema;
    }
}
